%__________________________________________________________________________
% ESTIMATE DSC, DC, and SC MODELS 
%__________________________________________________________________________
% Set the list of MODELS (line 17) to replicate the charts from the paper
% Go to line 49 to see the specifications
% Subsequently run Analyse_DSC to see IRFS and statistics
% Subsequently run Plot_Fig_23 and Plot_Fig_Annex_C1_C4 for
%                  the charts as in the paper and the Supplement
%__________________________________________________________________________
  clear
  clc
  cDir  = [pwd '\'];
  addpath(genpath(cDir));

% Estimate for the below list of models   
  MODELS  =  {'DSC','DC','SC','BV','DSC_dr','DSC_p4','DSC_lags','DC_lags'};
  

%__________________________________________________________________________
% SET PRIORS
%__________________________________________________________________________
  ndraw     = 1000;                    %  Nr of draws
  T1        = [1956  3];               %  Sample start
  T2        = [2016  12];              %  Sample end
                                     
% Reduced VAR Minnesota prior  
  P_.type   = 'M';                     %  Minnesota type of prior
  P_.pi1    =  0.2;                    %  Overall tightness
  P_.rho    =  0.5;                    %  Prior mean 1st own lag                    
  P_.pi3    =  0.5;                    %  Lag decay
  P_.pi4    =  1000;                   %  Exogenous variables
  P_.nu     =  8;                      %  df of Wishart
  
% Define SC reliability prior
  P_.L      =  0.9;
  P_.U      =  1;
  P_.a      =  1;
  P_.b      =  1;
  
% Further options
  irf_H    =   80;                     %  Length of IRFs reported
  O_.VRoot =   1;                      %  =1 -> Do VAR stability check
  O_.prt   =   0;                      %  =1 -> Do detailed printout  

  
%__________________________________________________________________________
% CHOICE OF MODEL 
%__________________________________________________________________________
  for mi = 1:length(MODELS)

   model = MODELS{mi};
   switch model
     case{'DSC'}     % Main DSC model
                     P_.DCflag   =    1;       %  Use discriminant regression
                     P_.SCflag   =    1;       %  Use SC Criterion
                     Gibbs_smpl   =   1;
                     iV.priorZ   =   'F';      %  Fixed lags of Z    
                     iV.lagZ     =   [0 0];    %  from lead 0 to lag 0  
                     iV.Mean     =   'EST';
                     p           =    8;       %  Lags of reduced form VAR
                     Dereg_flag  =    0;       %  Do not use deregulation index 
                     fn          =   'R_DSC';  
                    
     case{'DC'}      % Only discriminant analysis                  
                     P_.DCflag   =    1;       
                     P_.SCflag   =    0;       %  Do not use SC criterion
                     Gibbs_smpl  =    1;
                     iV.priorZ   =   'F';     
                     iV.lagZ     =   [0 0];
                     iV.Mean     =   'EST';
                     p           =    8;
                     Dereg_flag  =    0;
                     fn          =   'R_DC';  
                    
     case{'SC'}      % Only sign concordance 
                     P_.DCflag   =    0;       %  Do not use DC regression
                     P_.SCflag   =    1;
                     Gibbs_smpl  =    1;
                     iV.priorZ   =   'F';     
                     iV.lagZ     =   [0 0];    
                     p           =    8;
                     Dereg_flag  =    0;
                     fn          =   'R_SC';
            
     case{'BV'}      % Standard Bayesian proxy VAR
                     P_.DCflag   =    1;       %  Use discriminant regression
                     P_.SCflag   =    0;       
                     Gibbs_smpl  =    0;       %  No Mean adjustment
                     iV.priorZ   =   'F';         
                     iV.lagZ     =   [0 0];      
                     iV.Mean     =   '0';      %  Zero Mean
                     p           =    6;       
                     Dereg_flag  =    0;       
                     fn          =   'R_BV';  

     case{'DSC_dr'}  % Main model with deregulation index
                     P_.DCflag   =    1;       
                     P_.SCflag   =    1;
                     Gibbs_smpl  =    1;
                     iV.priorZ   =   'F';     
                     iV.lagZ     =   [0 0]; 
                     iV.Mean     =   'EST';
                     p           =    8;
                     Dereg_flag  =    1;       %  Use deregulation index 
                     fn          =    'R_DSC_dr';
                    
     case{'DSC_p4'}  % Main model with 4 lags
                     P_.DCflag   =    1;       
                     P_.SCflag   =    1;
                     iV.priorZ   =   'F';     
                     Gibbs_smpl  =    1;
                     iV.lagZ     =   [0 0];    
                     iV.Mean     =   'EST';
                     p           =    4;       %  Lags of reduced form VAR
                     Dereg_flag  =    0;
                     fn          =   'R_DSC_p4';
                    
     case{'DSC_lags'} % Lags of Z drawn from [1 4]
                     P_.DCflag   =    1;       
                     P_.SCflag   =    1;
                     Gibbs_smpl  =    1;
                     iV.priorZ   =   'M';      %  Draw single lag of Z from 
                     iV.lagZ     =   [-1 4];   %  .. range -1 leads to 4 lags   
                     iV.Mean     =   'EST';
                     p           =    8;
                     Dereg_flag  =    0;
                     fn          =   'R_DSC';
                    
     case{'DC_lags'} % Fixed Lags of Z over [1 4]
                     P_.DCflag   =    1;       
                     P_.SCflag   =    0;
                     Gibbs_smpl  =    1;
                     iV.priorZ   =   'F';      %  Use fixed lags of Z from 
                     iV.lagZ     =   [-1 4];   %  .. range -1 leads to 4 lags 
                     iV.Mean     =   'EST';
                     p           =    8;
                     Dereg_flag  =    0;
                     fn          =   'R_DC';
                    
     otherwise,      error('MODEL not defined')
   end
  
 % PREPARE THE DATA
   [A,B]   =  xlsread([cDir 'Data\Data'],'All');
   DateQ   =  A(:,1:2);
   X       =  A(:,3:end);
   Names   =  B(:,3:end);
 
 % Find time range 
   iT      =  DateFind(DateQ,T1):DateFind(DateQ,T2);
   DateQ   =  DateQ(iT,:);
   T       =  size(DateQ,1);
   
 % Generate VAR series
   dLX              =   log(X) - lagmatrix(log(X),1); 
   y                =   nan(T,7); 
   y(:,[3 4])       =   X(iT,[3 4]) / 100;  
   y(:,[1 2 5 6 7]) =   dLX(iT,[1 2 5 6 7]);
   yName            =   Names(1:7);                       

 % Generate exogenous series  
   if  Dereg_flag  == 0, W      =  ones(T,1); 
                         wName  =  {'c'};
   else                  W      = [ones(T,1) X(iT,8)];
                         wName  =  {'c','DregIndex'};
   end
      
 % Generate proxies
   z       =  X(iT,9:10);
   serZ    =  Names(9:10);

 % Add dummy restriction : contribution to FEVD < 100% 
   sName    =  {'MPP'};
   R_       =   Add_R([],yName,sName,'GDP' ,'MPP', 2,-1,100);

 % Run 
   if Gibbs_smpl, fn = [fn '_Gibbs'];  end 
   for i = 1:size(z,2)
       iV.z  =  z(:,i);
    
       rng('default');
       if Gibbs_smpl    
          [Q_,L_,E_,ZZ_,lag_,gam_] = DSC_BVAR_2022_Gibbs(y,W,p,R_,iV,ndraw,irf_H,P_,O_);     
       else
          [Q_,L_,E_,ZZ_,lag_]      = DSC_BVAR_2022(y,W,p,R_,iV,ndraw,irf_H,P_,O_);
          gam_ = [];
       end
   
       f_out = [fn  '_Z' num2str(i) ...
                    '_'  iV.priorZ num2str(iV.lagZ(1)) num2str(iV.lagZ(2))];
     
       save([cDir 'Results_DSC\' f_out], ...
             'DateQ','y','W','z','yName','sName','wName','P_', 'p', 'O_', ...
             'iV','R_','Q_','L_','E_','ZZ_','lag_','gam_')
       disp(' ')
       disp([f_out ' saved'])
       disp(' ')
   end
 end  