% generate_output_simulations: Generate Output for Simulations.  
% Report estimates of breakvector and beta coefficients. 
% Creates and saves histograms and Latex tables

clear 

% General path for matlab files
generalpath = '/Users/ina/Dropbox/Structural_Breaks_in_Panel_Data'; % OSX
cd ([generalpath '/Resubmission_JAE_September_2019/Matlab_files_ready_for_submission'])
% Add folder with Group Lasso estimator files to Matlab path
addpath([generalpath '/Resubmission_JAE_September_2019/Matlab_files_ready_for_submission/group_lasso'])

% Path for saved graphs and latex tables
filepath = [generalpath '/Paper/']; % generalpath generalpath_saveonMac


%%%%% Set Simulation Scenario %%%%%

simScen = 14;

% Set number of simulations
sims = 1000;


%%% Set default parameter values %%%

% DGP
listN          = [50 100 500]; % List over sample sizes (cross-sectional units)
listT          = 20;           % Number of time Periods (or T>20 max. breaks searched over is set to 6 for efficiency)
p              = 1;            % number of regressors (cons,timevar,timevar,timecons,timevar, ...)
timevarReg_ind = 1 ;           % colum vector of regressor positions that are timevarying (one timevar. regressor)
listBsize      = 0.2;          % break size in beta (loop over list of breaksizes in simScen 4 and 14)
listBnumber    = NaN;          % number of breaks in beta; NaN: set in simulation (loop over listBnumber simScen 3 and 11)
breaklocation  = NaN;          % NaN: set in simulation, 1:floor(20/3) = 6, 2:2, 3:floor(2*20/3) = 13
w_tcons_list   = NaN;          % list of different degrees of cross-sectional variation in x

% Estimation
penaltytype = 2;               % 1: BIC-type log(NT)/NT; 2: HQIC log(log(NT))/NT
lasso = 0;                     % 0: Boldea et al. 2016; 
                               % 1: Qian & Su 2015; this option is not 
                               % available in the package uploaded onto JAE
                               % data archive as it depends on the 
                               % AFGL code by Qian & Su 2015


if    simScen == 1
    % DGP with one break at [T/3]=6 or at 2:
    % Histogram of location of estimated breakpoints (using all replications with m_hat = 1)
    % Table of OLS, FE, FFE averages of Bias, SE, MSE (using all replications with m_hat = 1, T_hat = 2,...,19)
    % (generates Figure 1 and Tables 1 and 2 in paper)
    listT         = [20 30 50]; % For T > 20 the number of breaks searched over is set to 6 for efficiency
    breaklocation = 2;          % 1: floor(T/3) = 6, 2: 2
elseif simScen == 2
    % Sets number of breaks in DGP to 2: Estimate location of break points
    % (generates Figure 2 in paper)
elseif simScen == 3
    % Loop over 3 DGPs: No break in beta; one break in beta at T = 6, 
    % two breaks in beta at T = floor(T/3) and T = floor(2T/3); compare IC (BIC vs. Hannan&Quinn)
    % (generates Figure 3 in paper)
    listBnumber = [0, 1, 2];
    penaltytype = 2;            % 1: BIC-type log(NT)/NT; 2: HQIC log(log(NT))/NT
elseif simScen == 5 
    % DGP similar to Application; estimate number of breakpoints (m0 = 1) and location
    p = 15;                     % regressor dimension (cons,timevar,timevar,timecons,timevar, ...)
    timevarReg_ind = [2; 3; 5; 6; 8; 9; 11; 12; 14; 15; 17; 18; 20];
    listN = 200;
    listBsize = [ 0.010, 0.015, 0.020 ];
% elseif simScen == 6
%     % Compare estimators to group-lasso estimator of Qian & Su, generates Figure 4 
%     % For lasso = 1: Only runs on Windows platform (needs to be able to call C function mplsbcd.c)
%     listBnumber = 2;
%     w_tcons_list = [0, 0.10, 0.30]; % loop over w_tcons_list (degree of cross-sectional variation in x)
%     lasso = 0; % 0: Boldea et al. 2019; 1: Qian & Su 2015
elseif simScen == 11 % (Response to referees: Simulation A)
    % beta changes in EVERY (other) period with a trend
    % Hist: Estimated number of change points and for B = 9: Hist: location of change points
    %listN: In loop: For B = 9: [ 500 1000 2000 ] and for B = 19: [ 1000 5000 10000 ]
    listBnumber = [ 9 19 ]; % loop of B over listBnumber [ 9 19 ]
elseif simScen == 12 % (Response to referees: Simulation B)
    % beta is constant, c_it = c_i for t <= 6 andc_it = 1.5*c_i for t > 6
    % Report break number in beta, gamma, location and Chi2 Test for slope estimates (FE) and a_hat 
elseif simScen == 13 % (Response to referees: Simulation C)
    % beta and c_i are constant, report: histogram of number of breaks, frequency table of number of estimated
    % breaks in gamma and rejection frequency of Chi2 Test, frequency table of number of estimated breaks in beta
    listN     = [ 50 500 5000 ]; % [ 50 500 5000 ]  
elseif simScen == 14 % (Response to referees: Simulation D)
    % One break in beta at T = 6, c_i is constant, report: histogram of number of breaks,
    % frequency table of number of estimated breaks in beta
    listN     = [ 50 5000 10000 ]; % [ 50 100 500 ]  [ 1000 3000 5000 ]
    listBsize = [ 0.200 0.100 0.050 0.010 0.005 ]; % breaksize: [ 0.2 0.1 0.05 0.01 0.005 ]
end

% Assumptions for Standard Errors
%se_type = 0.3: Case A (autocorrelation-, conditional heteroskedasticity-robust)
%se_type = 0.2: Case B (conditional heteroskedasticity-robust and no autocorrelation)
%se_type = 0.1: Case C (conditional homoskedasticity and no autocorrelation)
%se_type = 0  :        (no heteroskedasticity and no autocorrleation; 
%                       se from standard OLS on demeaned data)
se_type = 0.3; % (autocorrelation-, conditional heteroskedasticity-robust)



%%%%% Simulate, estimate, save and print results %%%%%
tic

% Set seed for simulations
rng(7777)

for w_tcons = w_tcons_list %%% Loop over w_tcons_list (simscen 6) %%%
disp('w_tcons '); disp(w_tcons);

if     (simScen == 1); outtable_1_2 = []; end

for T = listT              %%% Loop over listT %%%
disp('T'); disp(T)


for B = listBnumber        %%% Loop over number of breaks for simScen 3 (and 11) %%%

    
if     (simScen==11 && B==9);  listN = [ 500 1000 2000 ];   % for #breakpoints = 9
elseif (simScen==11 && B==19); listN = [ 1000 5000 10000 ]; % for #breakpoints = 19
end

% Set breakvector for DGP
if     (simScen == 3 && B==0 || simScen == 12 || simScen == 13)
                                             breakvector = NaN; % no break
elseif (simScen == 3 && B==1 || simScen == 14 || simScen==1 && breaklocation == 1)
                                             breakvector = floor(T/3);
elseif (simScen == 1 && breaklocation == 2); breakvector = 2;
elseif simScen == 5;                         breakvector = 14;
elseif ( (simScen == 3 || simScen == 6) && B == 2 || simScen == 2) 
                                             breakvector = [floor(T/3); floor((2*T)/3)];
elseif (simScen == 11 && B == 9);            breakvector = (2:2:18)';
elseif (simScen == 11 && B == 19);           breakvector = (1:19)';
end

if isnan(breakvector); break_truenumb = 0;
else;                  [break_truenumb,~] = size(breakvector);
end

% Set up matrices to store estimation and test results
[~,number_Nlist] = size(listN);
[~,number_listBsize] = size(listBsize);
if (simScen == 5); number_Nlist = number_listBsize;
end
dim = (break_truenumb+1)*p; 
if (simScen == 12||simScen==13||simScen==14); dim = 2*p; end
break_numbEst_Store_byN      = NaN(sims,1,number_Nlist);
breakvecEst_Store_byN        = NaN(sims,break_truenumb+3,number_Nlist);
betaPlsEstimatorStore_byN    = NaN(sims,dim,number_Nlist);
betaSfeEstimatorStore_byN    = NaN(sims,dim,number_Nlist);
betaFfeEstimatorStore_byN    = NaN(sims,dim,number_Nlist);
varBetaPlsEstimatorStore_byN = NaN(sims,dim,number_Nlist);
varBetaSfeEstimatorStore_byN = NaN(sims,dim,number_Nlist);
varBetaFfeEstimatorStore_byN = NaN(sims,dim,number_Nlist);
WaldTestStoreSfe_byN         = NaN(sims,dim*2,number_Nlist);
WaldTestStoreFfe_byN         = NaN(sims,dim*2,number_Nlist);
m_hat_byN                    = NaN(sims,3,number_Nlist);
m_hat_beta_byN               = NaN(sims,6,number_Nlist);
atleast1chi2_rej_byN         = NaN(sims,4,number_Nlist);
aStore_byN                   = NaN(sims,2,number_Nlist);

for breaksize = listBsize %%% Loop over different sizes of breaks for simScen 5 and 14 %%%
  
% Define coefficient vector beta0 for DGP
if     simScen == 1; beta0 = [-ones(p,1);ones(p,1)].*0.1;
elseif (simScen == 3 || simScen == 2 || simScen == 6)
    beta0 = [-ones(p,1);ones(p,1);-ones(p,1)].*0.1;
    beta0 = beta0(1:p*(break_truenumb+1),:);
elseif simScen == 5
    % Case 1: Break in all coefficients; Every third regressor is time-constant 
    %(e.g.: one break,p=2,breaksize==0.02: beta0 = (0 -0.01 -0.01 0.02 0.01 0.01)' ) 
    beta0 = [0;-ones(p-1,1)*(breaksize/2);breaksize;ones(p-1,1)*(breaksize/2)];
elseif (simScen == 11 && B == 9);      beta0 = (0.1:0.1:1)';
elseif (simScen == 11 && B == 19);     beta0 = (0.05:0.05:1)';
elseif (simScen == 12 || simScen==13); beta0 = (0.1)';
elseif (simScen == 14)
    beta0 = [0.1; NaN];
    beta0(2) = beta0(1) + breaksize; % [0.1;0.3] [0.1;0.2] [0.1;0.15] [0.1;0.11] [0.1;0.105]
end
        

for N = listN
    disp('N'); disp(N)
    %%%%% Perform loop over simulations and estimations %%%%%
    [break_numbEst_Store,breakvecEst_Store,betaPlsEstimatorStore,betaSfeEstimatorStore,betaFfeEstimatorStore,...
        varBetaPlsEstimatorStore,varBetaSfeEstimatorStore,varBetaFfeEstimatorStore,...
        beta0,WaldTestStoreSfe,WaldTestStoreFfe,m_hat,m_hat_beta,atleast1chi2_rej,aStore]...
        = simulate_panelmultibreak(N,T,p,sims,simScen,se_type,breakvector,beta0,timevarReg_ind,penaltytype,w_tcons,lasso);
  
    if (simScen == 5) 
        if     breaksize == listBsize(1,1); N_index = 1;
        elseif breaksize == listBsize(1,2); N_index = 2;
        elseif breaksize == listBsize(1,3); N_index = 3;
        end 
        break_numbEst_Store_byN(:,:,N_index) = break_numbEst_Store;
        breakvecEst_Store_byN(:,:,N_index)   = breakvecEst_Store;
    else
        if     N == listN(1); N_index = 1;
        elseif N == listN(2); N_index = 2;
        elseif N == listN(3); N_index = 3;
        end
        size_est = size(betaPlsEstimatorStore);
        break_numbEst_Store_byN(:,:,N_index)      = break_numbEst_Store;
        breakvecEst_Store_byN(:,:,N_index)        = breakvecEst_Store;
        betaPlsEstimatorStore_byN(:,1:size_est(1,2),N_index)    = betaPlsEstimatorStore;
        betaSfeEstimatorStore_byN(:,1:size_est(1,2),N_index)    = betaSfeEstimatorStore;
        betaFfeEstimatorStore_byN(:,1:size_est(1,2),N_index)    = betaFfeEstimatorStore;
        varBetaPlsEstimatorStore_byN(:,1:size_est(1,2),N_index) = varBetaSfeEstimatorStore;
        varBetaSfeEstimatorStore_byN(:,1:size_est(1,2),N_index) = varBetaSfeEstimatorStore;
        varBetaFfeEstimatorStore_byN(:,1:size_est(1,2),N_index) = varBetaFfeEstimatorStore;  
    end
    if (simScen==12||simScen==13||simScen==14)
        WaldTestStoreSfe_byN(:,:,N_index) = WaldTestStoreSfe; 
        WaldTestStoreFfe_byN(:,:,N_index) = WaldTestStoreFfe; 
        m_hat_byN(:,:,N_index)            = m_hat;
        m_hat_beta_byN(:,:,N_index)       = m_hat_beta;
        atleast1chi2_rej_byN(:,:,N_index) = atleast1chi2_rej;
        aStore_byN(:,:,N_index)           = aStore;
        
    end
    
end % end Loop over listN


if simScen == 14 
        
    % Histogram: Number of Break Points
    histinput = reshape(break_numbEst_Store_byN,[sims,length(listN)]);
    x_range = 0:1:5; x_ticks = 0:1:5; x_tick_labels = 0;
    ind_legend = 1; legend_position = 'northeast'; % Indicator display legend; legend position in figure
    ind_location = 0;                              % Indicator for location vs. number of breaks
    filename_path = [filepath 'EstBreaknumb_sizepercent' num2str(breaksize*1000)  '_N' num2str(max(listN)) '_sims' num2str(sims)];              
    make_histogram(histinput,listN,ind_legend,legend_position,sims,filename_path,x_range,x_ticks,x_tick_labels,ind_location)
    
    % Generate latex table: Estimated number of breaks in beta
    temp = sum(m_hat_beta_byN);
    string_array =  [ reshape(temp(1,1:3,:),[9,1]),reshape(temp(1,4:6,:),[9,1]) ];
    ColLabels = {'Freq. FE','Freq. FFE'};
    RowLabels = {['N = ' num2str(listN(1)) ' $\&$ ' '$\hat{m}_{\beta}=0$'], ...
                 ['N = ' num2str(listN(1)) ' $\&$ ' '$\hat{m}_{\beta}=1$'], ...
                 ['N = ' num2str(listN(1)) ' $\&$ ' '$\hat{m}_{\beta}=2$'], ...
                 ['N = ' num2str(listN(2)) ' $\&$ ' '$\hat{m}_{\beta}=0$'], ...
                 ['N = ' num2str(listN(2)) ' $\&$ ' '$\hat{m}_{\beta}=1$'], ...
                 ['N = ' num2str(listN(2)) ' $\&$ ' '$\hat{m}_{\beta}=2$'], ...
                 ['N = ' num2str(listN(3)) ' $\&$ ' '$\hat{m}_{\beta}=0$'], ...
                 ['N = ' num2str(listN(3)) ' $\&$ ' '$\hat{m}_{\beta}=1$'], ...
                 ['N = ' num2str(listN(3)) ' $\&$ ' '$\hat{m}_{\beta}=2$']     };
    Table_caption = ['Frequencies per estimated number of breaks in $\beta$ for FE (column 1) and FFE (column 2). DGP with breaksize = ' num2str(breaksize*1000) '﻿$\%$, $T=20$ and one change in $c_{it}$ at $[T/3]=6$.'];
    DataFormat = {'%.0f',2}; % {'%.3f',1,'%.1f',2} three digits precision for first column, one digit for the second
    %out = [{'---'},ColLabels;RowLabels.',arrayfun( f3,string_array)]; disp(out); % Display table
    % Generate latex table from string array input.data
    array_to_latex_table(string_array,ColLabels,RowLabels,Table_caption,DataFormat);
    
end

end % end loop over different sizes of Breaks (for simScen 5 and 14)



%%%%% Generate (and save) histograms and latex tables from Monte Carlo results %%%%%
dateRecord = datestr(clock,'yyyy.mm.dd,HH:MM');

% Indicators for estimation results
Ind_1break_gamma = isnan(breakvecEst_Store_byN(:,2,:)); % Indicator estimated 1 break in gamma
Ind_2loc18       = (breakvecEst_Store_byN(:,1,:) > 1 & breakvecEst_Store_byN(:,1,:) < T-1); % (only for m_hat = 1) Indicator locations > 1 and < T-1 (necessary for FE)
Ind_2break_gamma = break_numbEst_Store_byN == 2; % Indicator estimated 2 breaks in gamma
Ind_loc6         = (breakvecEst_Store_byN(:,1,:) == 6); % Indicator for single break at [T/3] = 6    

f3 = @number_to_string_int; % convert integer to string (creating function handles)


if simScen == 1
    
    CorrectEstInd=Ind_1break_gamma.*1; % Indicator correctly estimating one break in gamma
    CorrectEstInd( CorrectEstInd == 0 ) = NaN;     
    histinput = reshape(repmat(CorrectEstInd,1,size(breakvecEst_Store_byN,2),1).* breakvecEst_Store_byN,...
                        [sims*size(breakvecEst_Store_byN,2),length(listN)]                                 );
    ind_legend = 0; legend_position = 'northeast'; % Indicator display legend; legend position in figure
    if     breaklocation==1 
        x_range = 0:1:floor(2*T/3); x_ticks = [0 floor(T/3) floor(2*T/3)]; x_tick_labels = {'0','[T/3]','[2T/3]'};
        filename_path = [filepath 'Break_at_Tover3' '_T' num2str(T) '_N' num2str(max(listN)) '_sims' num2str(sims)]; 
    elseif breaklocation==2
        x_range = 0:1:floor(2*T/3); x_ticks = [0 2 floor(2*T/3)]; x_tick_labels = {'0','2','[2T/3]'};
        filename_path = [filepath 'Break_at2' '_T' num2str(T) '_N' num2str(max(listN)) '_sims' num2str(sims)]; 
        if T==20; ind_legend = 1;
        end
    end
    ind_location = 1; % Indicator for location vs. number of breaks
    make_histogram(histinput,listN,ind_legend,legend_position,sims,filename_path,x_range,x_ticks,x_tick_labels,ind_location)
    
    % Part of Latex tables 1 & 2 in paper (for given T)
    CorrectInd = Ind_1break_gamma.*Ind_2loc18; 
    CorrectInd( CorrectInd == 0 ) = NaN; CorrectInd = repmat(CorrectInd,1,2);  
    beta0 = [-0.1;0.1];
    biasPlsMean = mean( betaPlsEstimatorStore_byN(:,1:2,:) - repmat(beta0',sims,1,number_Nlist).*CorrectInd, 1,'omitnan');
    biasSfeMean = mean( betaSfeEstimatorStore_byN(:,1:2,:) - repmat(beta0',sims,1,number_Nlist).*CorrectInd, 1,'omitnan');
    biasFfeMean = mean( betaFfeEstimatorStore_byN(:,1:2,:) - repmat(beta0',sims,1,number_Nlist).*CorrectInd, 1,'omitnan');
    sdSfeMean   = mean( sqrt(varBetaSfeEstimatorStore_byN(:,1:2,:)).*CorrectInd, 1,'omitnan');
    sdFfeMean   = mean( sqrt(varBetaFfeEstimatorStore_byN(:,1:2,:)).*CorrectInd, 1,'omitnan');
    mseSfe      = mean(varBetaSfeEstimatorStore_byN(:,1:2,:).*CorrectInd ,1,'omitnan') + (mean( betaSfeEstimatorStore_byN(:,1:2,:).*CorrectInd - repmat(beta0',sims,1,number_Nlist), 1,'omitnan')).^2;
    mseFfe      = mean(varBetaFfeEstimatorStore_byN(:,1:2,:).*CorrectInd ,1,'omitnan') + (mean( betaFfeEstimatorStore_byN(:,1:2,:).*CorrectInd - repmat(beta0',sims,1,number_Nlist), 1,'omitnan')).^2;
    input.data  = [biasPlsMean(1,1,:),NaN(1,2,3),biasPlsMean(1,2,:),NaN(1,2,3);...
                   biasSfeMean(1,1,:),sdSfeMean(1,1,:),mseSfe(1,1,:),biasSfeMean(1,2,:),sdSfeMean(1,2,:),mseSfe(1,2,:);...
                   biasFfeMean(1,1,:),sdFfeMean(1,1,:),mseFfe(1,1,:),biasFfeMean(1,2,:),sdFfeMean(1,2,:),mseFfe(1,2,:)    ];
    input.data  = reshape(input.data, 3, 6*number_Nlist);
    outtable_1_2 = [outtable_1_2;input.data];

elseif simScen == 2
    
    CorrectEstInd=Ind_2break_gamma.*1;      % Indicator correctly estimating one break in gamma
    CorrectEstInd( CorrectEstInd == 0 ) = NaN;     
    histinput = reshape(repmat(CorrectEstInd,1,size(breakvecEst_Store_byN,2),1).* breakvecEst_Store_byN,...
                        [sims*size(breakvecEst_Store_byN,2),length(listN)]                                 );
    x_range = 0:1:T; x_ticks = [0 floor(T/3) floor(2*T/3) T]; x_tick_labels = {'0','[T/3]','[2T/3]','T'};
    ind_legend = 1; legend_position = 'northeast'; % Indicator display legend; legend position in figure
    ind_location = 1;                              % Indicator for location vs. number of breaks
    filename_path = [filepath 'EstBreakLocation_2breaks_N' num2str(max(listN)) '_sims' num2str(sims)]; 
    make_histogram(histinput,listN,ind_legend,legend_position,sims,filename_path,x_range,x_ticks,x_tick_labels,ind_location)

elseif simScen == 3
    
    % Histogram: Number of Break Points
    histinput = reshape(break_numbEst_Store_byN,[sims,length(listN)]);
    x_range = 0:1:2; x_ticks = 0:1:2; x_tick_labels = 0; % x-axis
    ind_legend = 0; legend_position = 'northeast'; % Indicator display legend; legend position in figure
    if   (penaltytype==2&&B==0); ind_legend = 1;   % Indicator display legend 
    end
    ind_location = 0;                              % Indicator for location vs. number of breaks
    if     penaltytype == 1; filename_path = [filepath 'Penalty_log(NT)_Breaknumb' num2str(break_truenumb) '_N' num2str(max(listN)) '_sims' num2str(sims)];
    elseif penaltytype == 2; filename_path = [filepath 'Penalty_log(log(NT))_Breaknumb' num2str(break_truenumb) '_N' num2str(max(listN)) '_sims' num2str(sims)];              
    end 
    make_histogram(histinput,listN,ind_legend,legend_position,sims,filename_path,x_range,x_ticks,x_tick_labels,ind_location)

elseif simScen == 11
        
    % Histogram: Number of Break Points
    histinput = reshape(break_numbEst_Store_byN,[sims,length(listN)]);
    if     B == 9;  x_range = 0:1:11; x_ticks = 0:1:11; x_tick_labels = 0; % x-axis
    elseif B == 19; x_range = 0:1:T;  x_ticks = 0:2:T; x_tick_labels = 0;
    end
    ind_legend = 1; legend_position = 'northwest';    % Indicator display legend; legend position in figure
    ind_location = 0;                                 % Indicator for location vs. number of breaks
    filename_path = [filepath 'EstBreaknumb_' num2str(break_truenumb) '_N' num2str(max(listN)) '_sims' num2str(sims)];              
    make_histogram(histinput,listN,ind_legend,legend_position,sims,filename_path,x_range,x_ticks,x_tick_labels,ind_location)
    
    % Histogram: Break point locations
    if B == 9
        CorrectEstInd=(break_numbEst_Store_byN==9).*1; % Indicator correctly estimating 9 breaks in gamma
        CorrectEstInd( CorrectEstInd == 0 ) = NaN;     
        histinput = reshape(repmat(CorrectEstInd,1,size(breakvecEst_Store_byN,2),1).* breakvecEst_Store_byN,...
                            [sims*size(breakvecEst_Store_byN,2),length(listN)]                                 );
        x_range = 0:1:T; x_ticks = 0:2:T; x_tick_labels = 0; % x-axis
        ind_legend = 1; legend_position = 'northeast'; % Indicator display legend; legend position in figure
        ind_location = 1;                              % Indicator for location vs. number of breaks
        filename_path = [filepath 'EstBreaklocation_' num2str(B) 'breaks_N' num2str(max(listN)) '_sims' num2str(sims)];              
        make_histogram(histinput,listN,ind_legend,legend_position,sims,filename_path,x_range,x_ticks,x_tick_labels,ind_location)
    end
    
elseif simScen == 12 
    
    % Histogram: Number of Break Points
    histinput = reshape(break_numbEst_Store_byN,[sims,length(listN)]);
    x_range = 0:1:3; x_ticks = 0:1:3; x_tick_labels = 0; % x-axis
    ind_legend = 1;legend_position = 'northeast';  % Indicator for legend to be displayed
    ind_location = 0;                           % Indicator for location vs. number of breaks
    filename_path = [filepath 'Breakin_c_EstBreaknumb_c_1_5' '_N' num2str(max(listN)) '_sims' num2str(sims)];              
    make_histogram(histinput,listN,ind_legend,legend_position,sims,filename_path,x_range,x_ticks,x_tick_labels,ind_location)

    % Latex table: Estimated number of breaks in gamma
    temp = sum(atleast1chi2_rej_byN);
    array_latex = round([ reshape(sum(m_hat_byN),[9,1]),reshape([NaN(1,1,3),temp(1,1:2,:)],[9,1]) ],0);
    ColLabels = {'Frequency','$\chi2$ rej. FE'};
    RowLabels = {['N = ' num2str(listN(1)) ' $\&$ ' '$\hat{m}=0$'], ...
                 ['N = ' num2str(listN(1)) ' $\&$ ' '$\hat{m}=1$'], ...
                 ['N = ' num2str(listN(1)) ' $\&$ ' '$\hat{m}=2$'], ...
                 ['N = ' num2str(listN(2)) ' $\&$ ' '$\hat{m}=0$'], ...
                 ['N = ' num2str(listN(2)) ' $\&$ ' '$\hat{m}=1$'], ...
                 ['N = ' num2str(listN(2)) ' $\&$ ' '$\hat{m}=2$'], ...
                 ['N = ' num2str(listN(3)) ' $\&$ ' '$\hat{m}=0$'], ...
                 ['N = ' num2str(listN(3)) ' $\&$ ' '$\hat{m}=1$'], ...
                 ['N = ' num2str(listN(3)) ' $\&$ ' '$\hat{m}=2$']     };
    Table_caption = 'Frequencies per estimated number of breaks in $\gamma$ (column 1) and for at least one break in $\beta$ (column2). DGP with $T=20$ and one change in $c_{it}$ at $[T/3]=6$.';
    %out = [{'---'},ColLabels;RowLabels.',arrayfun( f3, string_array)]; disp(out); % Display table
    DataFormat = {'%.0f',2}; % {'%.3f',1,'%.1f',2} three digits precision for first column, one digit for the second
    array_to_latex_table(array_latex,ColLabels,RowLabels,Table_caption,DataFormat);

    % Histogram: Break point locations
    CorrectEstInd = Ind_1break_gamma.*1;
    CorrectEstInd( CorrectEstInd == 0 ) = NaN; 
    CorrectEstInd = repmat(CorrectEstInd,1,3);  
    histinput = reshape(CorrectEstInd .* breakvecEst_Store_byN,[sims*size(breakvecEst_Store_byN,2),3]);
    x_range = 0:1:T; x_ticks = [0:1:10,12:2:T]; x_tick_labels = 0; % x-axis
    ind_legend = 1; legend_position = 'northeast'; % Indicator for legend to be displayed
    ind_location = 1;                             % Indicator for location vs. number of breaks
    filename_path = [filepath 'Breakin_c_EstBreaklocation_c_1_5' '_N' num2str(max(listN)) '_sims' num2str(sims)];              
    make_histogram(histinput,listN,ind_legend,legend_position,sims,filename_path,x_range,x_ticks,x_tick_labels,ind_location)

    % Latex table: Frequencies of correctly estimated break locations (for m_hat = 1)
    CorrectInd = Ind_1break_gamma.*Ind_loc6;
    CorrectInd(CorrectInd == 0) = NaN;
    temp1 = breakvecEst_Store_byN .* repmat(CorrectInd,1,3);
    frequ_table = reshape(temp1(:,1,:),[sims,length(listN)]);
    array_latex = arrayfun( f3, reshape(sum(frequ_table==6),[3,1]) );
    ColLabels = {'Frequ. correct location'};
    RowLabels = {['N = ' num2str(listN(1))],['N = ' num2str(listN(2))],['N = ' num2str(listN(3))]};
    Table_caption = 'Frequencies of simulations with $\hat{m}=1$ and estimated location at $[T/3]=6$. DGP with $T=20$ and one change in $c_{it}$ at $[T/3]=6$.';
    %out = [{'---'},ColLabels;RowLabels.',array_latex]; disp(out); % Display table
    DataFormat = {'%.0f',1}; % {'%.3f',1,'%.1f',2} three digits precision for first column, one digit for the second
    array_to_latex_table(array_latex,ColLabels,RowLabels,Table_caption,DataFormat);    
    
    % Latex table: Estimated number of breaks in beta
    temp = sum(m_hat_beta_byN);
    array_latex = round(reshape(temp(1,1:3,:),[9,1]),0);
    ColLabels = {'Frequency FE'};
    RowLabels = {['N = ' num2str(listN(1)) ' $\&$ ' '$\hat{m}_{\beta}=0$'], ...
                 ['N = ' num2str(listN(1)) ' $\&$ ' '$\hat{m}_{\beta}=1$'], ...
                 ['N = ' num2str(listN(1)) ' $\&$ ' '$\hat{m}_{\beta}=2$'], ...
                 ['N = ' num2str(listN(2)) ' $\&$ ' '$\hat{m}_{\beta}=0$'], ...
                 ['N = ' num2str(listN(2)) ' $\&$ ' '$\hat{m}_{\beta}=1$'], ...
                 ['N = ' num2str(listN(2)) ' $\&$ ' '$\hat{m}_{\beta}=2$'], ...
                 ['N = ' num2str(listN(3)) ' $\&$ ' '$\hat{m}_{\beta}=0$'], ...
                 ['N = ' num2str(listN(3)) ' $\&$ ' '$\hat{m}_{\beta}=1$'], ...
                 ['N = ' num2str(listN(3)) ' $\&$ ' '$\hat{m}_{\beta}=2$']     };
    Table_caption = 'Frequencies per estimated number of breaks $\beta$ for FE. DGP with $T=20$ and one change in $c_{it}$ at $[T/3]=6$.';
    %out = [{'---'},ColLabels;RowLabels.',arrayfun( f3, string_array)]; disp(out); % Display table
    DataFormat = {'%.0f',1}; % {'%.3f',1,'%.1f',2} three digits precision for first column, one digit for the second
    array_to_latex_table(array_latex,ColLabels,RowLabels,Table_caption,DataFormat);
    
    % Latex table: FE estimates and Chi2 Tests for H_0:\beta_1=\beta_2 and H_0:a_1=a_2 (for m_hat = 1)
    CorrectInd = Ind_1break_gamma.*Ind_2loc18; 
    CorrectInd( CorrectInd == 0 ) = NaN; CorrectInd = repmat(CorrectInd,1,2); 
    f  = @putbrackets; f2 = @number_to_string; % Put brackets around se and convert to string (creating function handles)
    sqrt_varBetaSfeEstimatorStore_byN = sqrt(varBetaSfeEstimatorStore_byN);
    EstMean  = arrayfun( f2, mean( betaSfeEstimatorStore_byN(:,1:2,:).*CorrectInd,'omitnan') );
    SeMean   = arrayfun( f,  mean( sqrt_varBetaSfeEstimatorStore_byN(:,1:2,:).*CorrectInd,'omitnan') );
    WaldMean_beta = arrayfun( f2, mean( [WaldTestStoreSfe_byN(:,1,:),WaldTestStoreSfe_byN(:,2,:)].*CorrectInd,'omitnan') );
    WaldMean_a    = arrayfun( f2, mean( [aStore_byN(:,1,:),aStore_byN(:,2,:)].*CorrectInd,'omitnan') );
    array_latex = [ [[EstMean(:,:,1),WaldMean_beta(:,:,1),WaldMean_a(:,:,1)];[SeMean(:,:,1),{' ',' ',' ',' '}]];...
                    [[EstMean(:,:,2),WaldMean_beta(:,:,2),WaldMean_a(:,:,2)];[SeMean(:,:,2),{' ',' ',' ',' '}]];...
                    [[EstMean(:,:,3),WaldMean_beta(:,:,3),WaldMean_a(:,:,3)];[SeMean(:,:,3),{' ',' ',' ',' '}]]      ];
    ColLabels = {'﻿$\hat{\beta_1}$','﻿$\hat{\beta_2}$','\chi2_{\beta}$ $(H_0:\beta_1=\beta_2)$',...
                 '$\chi2_{\beta}$ $\%$ rej.','$\chi2_a$ $(H_0:a_1=a_2)','$\chi2_a$ $\%$ rej.'};
    RowLabels = {['N = ' num2str(listN(1))],'',['N = ' num2str(listN(2))],'',['N = ' num2str(listN(3))],'',};
    Table_caption = 'Average of FE Estimates and Wald statistics over all simulations with $\hat{m}=1$. DGP with $T=20$ and one change in $c_{it}$ at $[T/3]=6$.';
    %out = [{'---'},ColLabels;RowLabels.',input.data]; disp(out); % Display table
    DataFormat = {'%.3f',6}; % {'%.3f',1,'%.1f',2} three digits precision for first column, one digit for the second
    array_to_latex_table(array_latex,ColLabels,RowLabels,Table_caption,DataFormat);
   
elseif simScen == 13 

    % Histogram: Number of Break Points
    histinput = reshape(break_numbEst_Store_byN,[sims,length(listN)]);
    x_range = 0:1:3; x_ticks = 0:1:3; x_tick_labels = 0;
    ind_legend = 1; legend_position = 'northeast'; % Indicator display legend; legend position in figure
    ind_location = 0;                              % Indicator for location vs. number of breaks
    filename_path = [filepath 'NoBreak_EstBreaknumb' '_N' num2str(max(listN)) '_sims' num2str(sims)];              
    make_histogram(histinput,listN,ind_legend,legend_position,sims,filename_path,x_range,x_ticks,x_tick_labels,ind_location)
    
    % Generate latex table: Estimated number of breaks in gamma
    temp = sum(atleast1chi2_rej_byN); temp1 = [NaN(1,1,3),temp(1,1:2,:)]; temp2 = [NaN(1,1,3),temp(1,3:4,:)];
    array_latex = [ reshape(sum(m_hat_byN),[9,1]),reshape(temp1,[9,1]),reshape(temp2,[9,1]) ];
    ColLabels = {'Frequency','$\chi2$ rej. FE','$\chi2$ rej. FFE'};
    RowLabels = {['N = ' num2str(listN(1)) ' $\&$ ' '$\hat{m}=0$'], ...
                 ['N = ' num2str(listN(1)) ' $\&$ ' '$\hat{m}=1$'], ...
                 ['N = ' num2str(listN(1)) ' $\&$ ' '$\hat{m}=2$'], ...
                 ['N = ' num2str(listN(2)) ' $\&$ ' '$\hat{m}=0$'], ...
                 ['N = ' num2str(listN(2)) ' $\&$ ' '$\hat{m}=1$'], ...
                 ['N = ' num2str(listN(2)) ' $\&$ ' '$\hat{m}=2$'], ...
                 ['N = ' num2str(listN(3)) ' $\&$ ' '$\hat{m}=0$'], ...
                 ['N = ' num2str(listN(3)) ' $\&$ ' '$\hat{m}=1$'], ...
                 ['N = ' num2str(listN(3)) ' $\&$ ' '$\hat{m}=2$']     };
    Table_caption = 'Frequencies per estimated number of breaks in $\gamma$ (column 1) and for at least one break in $\beta$ (column2). DGP with $T=20$ and one change in $c_{it}$ at $[T/3]=6$.';
    DataFormat = {'%.0f',3}; % {'%.3f',1,'%.1f',2} three digits precision for first column, one digit for the last two columns
    %out = [{'---'},ColLabels;RowLabels.',arrayfun( f3, string_array)]; disp(out); % Display table
    array_to_latex_table(array_latex,ColLabels,RowLabels,Table_caption,DataFormat); disp(' ');

    % Generate latex table: Estimated number of breaks in beta
    temp = sum(m_hat_beta_byN);
    array_latex =  [ reshape(temp(1,1:3,:),[9,1]),reshape(temp(1,4:6,:),[9,1]) ];
    ColLabels = {'Frequency FE','Frequency FFE'};
    RowLabels = {['N = ' num2str(listN(1)) ' $\&$ ' '$\hat{m}_{\beta}=0$'], ...
                 ['N = ' num2str(listN(1)) ' $\&$ ' '$\hat{m}_{\beta}=1$'], ...
                 ['N = ' num2str(listN(1)) ' $\&$ ' '$\hat{m}_{\beta}=2$'], ...
                 ['N = ' num2str(listN(2)) ' $\&$ ' '$\hat{m}_{\beta}=0$'], ...
                 ['N = ' num2str(listN(2)) ' $\&$ ' '$\hat{m}_{\beta}=1$'], ...
                 ['N = ' num2str(listN(2)) ' $\&$ ' '$\hat{m}_{\beta}=2$'], ...
                 ['N = ' num2str(listN(3)) ' $\&$ ' '$\hat{m}_{\beta}=0$'], ...
                 ['N = ' num2str(listN(3)) ' $\&$ ' '$\hat{m}_{\beta}=1$'], ...
                 ['N = ' num2str(listN(3)) ' $\&$ ' '$\hat{m}_{\beta}=2$']     };
    Table_caption = 'Frequencies per estimated number of breaks $\beta$ for FE (column 1) and FFE (column 2). DGP with $T=20$ and one change in $c_{it}$ at $[T/3]=6$.';
    DataFormat = {'%.0f',2}; % {'%.3f',1,'%.1f',2} three digits precision for first column, one digit for the second
    %out = [{'---'},ColLabels;RowLabels.',arrayfun( f3,string_array)]; disp(out); % Display table
    array_to_latex_table(array_latex,ColLabels,RowLabels,Table_caption,DataFormat);
    
elseif simScen == 5
        
    % Histogram: Number of Break Points
    histinput = reshape(break_numbEst_Store_byN,[sims,length(listBsize)]);
    x_range = 0:1:3; x_ticks = 0:1:3; x_tick_labels = 0;
    ind_legend = 2; legend_position = 'northeast'; % Indicator display legend; legend position in figure
    ind_location = 0;                              % Indicator for location vs. number of breaks
    filename_path = [filepath 'AppDGP_EstBreaknumb_N' num2str(max(listN)) '_sims' num2str(sims)];              
    make_histogram(histinput,listN,ind_legend,legend_position,sims,filename_path,x_range,x_ticks,x_tick_labels,ind_location)
    
    % Histogram: Break point location
    CorrectEstInd = Ind_1break_gamma.*1;
    CorrectEstInd( CorrectEstInd == 0 ) = NaN; 
    histinput = reshape(CorrectEstInd .* breakvecEst_Store_byN(:,1,:),[sims,length(listBsize)]);
    x_range = 0:1:T; x_ticks = [14,T]; x_tick_labels = {'14','20'}; % x-axis
    ind_legend = 0;                             % Indicator for legend to be displayed
    ind_location = 1;                           % Indicator for location vs. number of breaks
    filename_path = [filepath 'AppDGP_EstBreaklocation_N' num2str(max(listN)) '_sims' num2str(sims)];              
    make_histogram(histinput,listN,ind_legend,legend_position,sims,filename_path,x_range,x_ticks,x_tick_labels,ind_location)
    
end

end % end loop over number of Breaks (for simScen 3 and 11)


end % end loop over listT
toc;
timePassedInSeconds = toc;

if simScen == 6
    
    % Histogram: Number of Break Points
    histinput = reshape(break_numbEst_Store_byN,[sims,length(listN)]);
    x_range = 0:1:2; x_ticks = 0:1:2; x_tick_labels = 0; % x-axis
    ind_legend = 0; legend_position = 'northwest'; % Indicator display legend; legend position in figure
    ind_location = 0;                              % Indicator for location vs. number of breaks
    filename_path = [filepath 'BreakNumb_compare_w' num2str(w_tcons) 'Lasso' num2str(lasso) '_N' num2str(max(listN)) '_sims' num2str(sims)];
    make_histogram(histinput,listN,ind_legend,legend_position,sims,filename_path,x_range,x_ticks,x_tick_labels,ind_location)

    % Latex table: FE estimates and Chi2 Tests for H_0:\beta_1=\beta_2 and H_0:a_1=a_2 (for m_hat = 1)
    Ind_2breaks_at6_13 = ( breakvecEst_Store_byN(:,1:2,:) == [6,13] );    
    CorrectInd = Ind_2break_gamma.*Ind_2breaks_at6_13; 
    CorrectInd( CorrectInd == 0 ) = NaN; CorrectInd = repmat(CorrectInd(:,:,3),1,3); 
    f  = @putbrackets; f2 = @number_to_string; % Put brackets around se and convert to string (creating function handles)
    sqrt_varBetaSfeEstimatorStore_byN = sqrt(varBetaSfeEstimatorStore_byN);
    sqrt_varBetaFfeEstimatorStore_byN = sqrt(varBetaFfeEstimatorStore_byN);
    EstMean  = arrayfun( f2, mean( [betaSfeEstimatorStore_byN(:,1:3,3),betaFfeEstimatorStore_byN(:,1:3,3)].*CorrectInd,'omitnan') );
    SeMean   = arrayfun( f,  mean( [sqrt_varBetaSfeEstimatorStore_byN(:,1:3,3),sqrt_varBetaFfeEstimatorStore_byN(:,1:3,3)].*CorrectInd,'omitnan') );
    array_latex = [ EstMean; SeMean ];
    ColLabels = {'﻿$\hat{\beta_{1,FE}}$','﻿$\hat{\beta_{2,FE}}$','﻿$\hat{\beta_{3,FE}}$','﻿$\hat{\beta_{1,FFE}}$','﻿$\hat{\beta_{2,FFE}}$','﻿$\hat{\beta_{3,FFE}}$'};
    RowLabels = {['w = ' num2str(w_tcons)],' '};
    Table_caption = 'Average of FE Estimates over all simulations with $\hat{m}=1$. DGP with $T=20$ and two change points in $beta$ at $[T/3]=6$ and $[2T/3]=13$.';
    %out = [{'---'},ColLabels;RowLabels.',input.data]; disp(out); % Display table
    DataFormat = {'%.3f',6}; % {'%.3f',1,'%.1f',2} three digits precision for first column, one digit for the second
    array_to_latex_table(array_latex,ColLabels,RowLabels,Table_caption,DataFormat);
    
end

end % end loop over w_tcons_list (for simScen 6)


if simScen == 1
    % Generate latex tables 1 & 2 in paper %
    string_array = [outtable_1_2(1:9,1:6);outtable_1_2(1:9,7:12);outtable_1_2(1:9,13:18)];
    ColLabels = {'Bias','Sd','MSE','Bias','Sd','MSE'};  
    RowLabels = {'T20 OLS','T20 FE','T20 FFE', 'T30 OLS','T30 FE','T30 FFE', 'T50 OLS','T50 FE','T50 FFE',...
                 'T20 OLS','T20 FE','T20 FFE', 'T30 OLS','T30 FE','T30 FFE', 'T50 OLS','T50 FE','T50 FFE',...
                 'T20 OLS','T20 FE','T20 FFE', 'T30 OLS','T30 FE','T30 FFE', 'T50 OLS','T50 FE','T50 FFE'    };
    Table_caption = 'Properties of slope estimators for a single change point at [T/3]. Average of Bias, SE and MSE of slope estimates for DGP with one change in $\beta$ at $[T/3]=6$. For $T>20$ the maximum number of breaks searched over is set to 6.';
    DataFormat = {'%.3f',6}; % {'%.3f',1,'%.1f',2} three digits precision for first column, one digit for the second
    %out = [{'---'},ColLabels;RowLabels.',arrayfun( f2,string_array)]; disp(out); % Display table
    array_to_latex_table(string_array,ColLabels,RowLabels,Table_caption,DataFormat);
end






