%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% "Tax Shocks with High and Low Uncertainty" 
% Bertolotti F. and Marcellino M.
% Journal of Applied Econometrics
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [irp, irn, irfpup, irfpdown, irfnup, irfndown, irpneu, irnneu, ...
    irfpneuup, irfpneudown, irfnneuup, irfnneudown, estdata, mult]=varsn_2s(ucl,lcl,colm,colsv,shock)

%% TVAR
rng('default')
rng(1)

%% set preliminary parameters

p=2;        % number of lags in the VAR
irh=10;     % time horizon for the impulse responses
reps=10000;  % number of repetitions of the bootstrap
ucl=ucl;     % upper confidence level for IRFs
lcl=lcl;     % lower confidence level for IRFs
f=100;      % number of intervals for gird search
pl=30;      % lower percentile for grid search
pu=70;      % upper percentile for grid search
colm=colm;    % sets the macro variable of interest (pos 2 in the VAR)
colsv=colsv;   % sets the column of the switching variable

%% Import Data

data_macro_series % first set of macro variables (see m.file for the content)
data_proxy_series  % narrative shock variables (see m.file for the content)
data_switching_series  % uncertainty variables (see m.file for the content)

%% Generate the vector of Y's

dates0 = data_macro(:,1);

% Select or apply proper tranformation to the switching variable
if colsv==13 % residuals of jln's macro unc. on QoQ GDP growth
    % [~, ~, Z0]=regress(data_SV(:,5),[ones(size(data_SV(:,5),1),1) data_SV(:,12)]);
    [~, ~, Z0]=regress(data_SV(:,5),[ones(size(data_SV(:,5),1),1),(data_macro(:,5)-lagmatrix(data_macro(:,5),1))]);
elseif colsv==11   % detrend epu
    [~, ~, Z0]=regress(data_SV(:,11),[ones(size(data_SV(:,11),1),1) (1:size(data_SV(:,11),1))']);
else
    Z0=data_SV(:,colsv);
end

% Build the matrix of endogenous variables: Cost of debt, GDP (or other macro variable), Inflation,
% Tax revenues, Government spending, Uncertainty variable, FFR
Y0=[data_macro(:,8) data_macro(:,colm)  data_macro(:,9) data_macro(:,7) data_macro(:,6) Z0 data_macro(:,13)];

% Build the vector of Romer and Romer narrative tax shocks
if shock==1 
    V0=data_proxy(:,6);
elseif shock==2
    [~, ~, V0]=regress(data_proxy(:,6),[data_proxy(:,8) lagmatrix(data_proxy(:,8),[1 2 3 4])]);
end
V0m=data_proxy(:,9); % RR MP Shocks

ngdp=data_macro(:,14);
rrabs=data_proxy(:,5);

% Find extremes of the estimation sample
extr=get_extremes([Y0 V0 V0m]);
s=max(extr(1,:));
e=min(extr(2,:));

Y0=Y0(s:e,:);
V0=V0(s:e,:);
V0m=V0m(s:e,:);
Z0=Z0(s:e,:);
dates0=dates0(s:e,:);
ngdp=ngdp(s:e,:);
rrabs=rrabs(s:e,:);

% Narrative shocks of different sign
Vs0=zeros(size(V0,1),2);
for j=1:size(V0,1)
    if V0(j)>0
        Vs0(j,1)=V0(j);
    elseif V0(j)<0
        Vs0(j,2)=V0(j);
    end
end


%% generate X's 

X0=[];
for j=1:p
    X0=[X0 lagmatrix(Y0,j)];
end

Y=Y0(p+1:end,:);
Vs=Vs0(p+1:end,:);
Vm=V0m(p+1:end,:);
X=X0(p+1:end,:);
Z=Z0(p+1:end,:);
dates=dates0(p+1:end,:);
ngdp=ngdp(p+1:end,:);
rrabs=rrabs(p+1:end,:);

T=size(Y,1);
N=size(Y,2);
X=[ones(T,1) X Vm Vs];

%% linear model - point estimate and IRFs

% compute the coefficients of the VAR and create the CF matrix
beta=(X'*X)\(X'*Y);
res=Y-X*beta;
estdata=[dates Y Vm Vs];
Cf=[beta(2:end-3,:)'; eye(N*(p-1)) zeros(N*(p-1),N)];

% Compute the impulse responses
% Positive shock
irp(:,1)=[beta(end-1,:)'; zeros(N*(p-1),1)];
for j=2:irh
    irp(:,j)=Cf*irp(:,j-1);
end

% Negative shock
irn(:,1)=[beta(end,:)'; zeros(N*(p-1),1)];
for j=2:irh
    irn(:,j)=Cf*irn(:,j-1);
end

%% linear model - bootstrap
betamat=zeros(N*p+4,N,reps+1); betamat(:,:,reps+1)=beta;
irfpmat=zeros(N*p,irh,reps); 
irfnmat=zeros(N*p,irh,reps);
irf1mat=zeros(N*p,irh,reps);
irpneumat=zeros(N*p,irh,reps);
irnneumat=zeros(N*p,irh,reps);

Ymat=zeros(T,N,reps);
Xmat=zeros(T,N*p+4,reps);

for j=1:reps
    
    % generate new Y
    vv=randi([1,T],T,1);
    Yb=Y+res(vv,:);
    for jj=p+1:T
        Yb(jj,:)=beta(1,:)+beta(end-1,:).*Vs(jj,1)+beta(end,:).*Vs(jj,2)+beta(end-2,:).*Vm(jj)+res(vv(jj),:);
        for hh=1:p
            Yb(jj,:)=Yb(jj,:)+Yb(jj-hh,:)*beta(N*(hh-1)+2:N*hh+1,:);
        end
    end
    Ymat(:,:,j)=Yb;
    
    % generate new X
    Xb=[];
    for jj=1:p
        Xb=[Xb lagmatrix(Yb,jj)];
    end
    for jj=1:p
        Xb(1:jj,N*(jj-1)+1:N*(jj-1)+N)=Y0(1:jj,:);
    end
    Xb=[ones(T,1) Xb Vm Vs];
    Xmat(:,:,j)=Xb;
    
    % compute estimated coefficients and CF matrix
    betamat(:,:,j)=(Xb'*Xb)\(Xb'*Yb);
    Cf=[betamat(2:end-3,:,j)'; eye(N*(p-1)) zeros(N*(p-1),N)];

    % Compute the impulse responses
    % Positive Tax shock
    irfpmat(:,1,j)=[betamat(end-1,:,j)'; zeros(N*(p-1),1)];
    for jj=2:irh
        irfpmat(:,jj,j)=Cf*irfpmat(:,jj-1,j);
    end
    
    % Negative Tax shock
    irfnmat(:,1,j)=[betamat(end,:,j)'; zeros(N*(p-1),1)];
    for jj=2:irh
        irfnmat(:,jj,j)=Cf*irfnmat(:,jj-1,j);
    end

    % MP shock
    irf1mat(:,1,j)=[betamat(end-2,:,j)'; zeros(N*(p-1),1)];
    for jj=2:irh
        irf1mat(:,jj,j)=Cf*irf1mat(:,jj-1,j);
    end

   % Compute IRFs to a tax shock removing the effect of monetary policy
    irpneumat(:,:,j)=irfpmat(:,:,j); irnneumat(:,:,j)=irfnmat(:,:,j); 

    for jj=1:irh
    reacp=irpneumat(N,jj,j); % j-th period effect of a 1% positive tax shock on FFR 
    reacn=irnneumat(N,jj,j); % j-th period effect of a 1% negative tax shock on FFR

    mpshockp=reacp/irf1mat(N,1,j);
    mpshockn=reacn/irf1mat(N,1,j);

    irf_reac_p=irf1mat(:,1:end-jj+1,j).*mpshockp; % IRFs of all aggregates to a MP shock of size mpshockp
    irf_reac_n=irf1mat(:,1:end-jj+1,j).*mpshockn; % IRFs of all aggregates to a MP shock of size mpshockn

    irpneumat(:,jj:end,j)=irpneumat(:,jj:end,j)-irf_reac_p;
    irnneumat(:,jj:end,j)=irnneumat(:,jj:end,j)-irf_reac_n;

    end
    
end

irfpup=prctile(irfpmat,ucl,3);
irfpdown=prctile(irfpmat,lcl,3);
irfnup=prctile(irfnmat,ucl,3);
irfndown=prctile(irfnmat,lcl,3);
irfpneuup=prctile(irpneumat,ucl,3);
irfpneudown=prctile(irpneumat,lcl,3);
irfnneuup=prctile(irnneumat,ucl,3);
irfnneudown=prctile(irnneumat,lcl,3);
irpneu=prctile(irpneumat,50,3);
irnneu=prctile(irnneumat,50,3);

%% Multipliers

% Define useful objects
dp=(Vs(:,1)>0); % Identifier for observations in low-regime
dn=(Vs(:,2)<0); % Identifier for observations in high-regime
hor=1;

clear mult2 bands2
% Second method+
[mult2(1), bands2(1,:)]=multiplier_m2(hor,irp(2,:),squeeze(irfpmat(2,:,:))',ngdp(dp==1),rrabs(dp==1));
[mult2(2), bands2(2,:)]=multiplier_m2(hor,-irn(2,:),-squeeze(irfnmat(2,:,:))',ngdp(dn==1),rrabs(dn==1));
mult=[mult2' bands2];





