%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% "Tax Shocks with High and Low Uncertainty" 
% Bertolotti F. and Marcellino M.
% Journal of Applied Econometrics
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [irp0, irp1, irn0, irn1, irfp0up, irfp0down, irfp1up, irfp1down, ...
    irfn0up, irfn0down, irfn1up, irfn1down, irp0neu, irp1neu, irn0neu, irn1neu, irfp0upneu, irfp0downneu, irfp1upneu, irfp1downneu, ...
    irfn0upneu, irfn0downneu, irfn1upneu, irfn1downneu, estdata, mult, thresh]=tvarsn_2shocks(ucl,lcl,colm,colsv,shock)

%% TVAR with Sign Nonlinearity
% Set random seed for replicability
rng('default')
rng(1)

%% set preliminary parameters

p=2;        % number of lags in the VAR
irh=10;     % time horizon for the impulse responses
reps=10000;  % number of repetitions of the bootstrap
ucl=ucl;     % upper confidence level for IRFs
lcl=lcl;     % lower confidence level for IRFs
f=100;      % number of intervals for gird search
pl=30;      % lower percentile for grid search
pu=70;      % upper percentile for grid search
colm=colm;    % sets the macro variable of interest (pos 2 in the VAR) (position of the variable in data_macro_series.m file)
colsv=colsv;   % sets the column of the switching variable (position of the variable in data_switching_series.m file)

%% Import Data

data_macro_series % first set of macro variables (see m.file for the content)
data_proxy_series  % narrative shock variables (see m.file for the content)
data_switching_series  % uncertainty variables (see m.file for the content)

%% Generate the vector of Y's

dates0 = data_macro(:,3);

% Select or apply proper tranformation to the switching variable
if colsv==13 % residuals of jln's macro unc.  on QoQ GDP growth
    [~, ~, Z0]=regress(data_SV(:,5),[ones(size(data_SV(:,5),1),1),(data_macro(:,5)-lagmatrix(data_macro(:,5),1))]);
elseif colsv==11   % detrend epu if epu is chosen
    [~, ~, Z0]=regress(data_SV(:,11),[ones(size(data_SV(:,11),1),1) (1:size(data_SV(:,11),1))']);
else
    Z0=data_SV(:,colsv); % no transformation is a standard switching variable is chosen
end

% Build the matrix of endogenous variables: Cost of debt, GDP (or other macro variable), Inflation,
% Tax revenues, Government spending, Uncertainty variable, FFR
Y0=[data_macro(:,8) data_macro(:,colm)  data_macro(:,9) data_macro(:,7) data_macro(:,6) Z0 data_macro(:,13)];

% Smoothing of the threshold variable for the identification of regimes
if colsv~=12
    Z0=tsmovavg(Z0,'s',6,1);
end

% Build the vector of Romer and Romer narrative tax shocks
if shock==1 
    V0=data_proxy(:,6);
elseif shock==2 % Use residuals of shocks if the use of RR tax shocks orthogonal to spending shocks is chosen
    [~, ~, V0]=regress(data_proxy(:,6),[data_proxy(:,8) lagmatrix(data_proxy(:,8),[1 2 3 4])]);
end
V0m=data_proxy(:,9); % RR monetary policy shocks

% Store objects used to compute fiscal multipliers
ngdp=data_macro(:,14);
rrabs=data_proxy(:,5);

% Find extremes of the estimation sample
extr=get_extremes([Y0 Z0 V0 V0m]);
s=max(extr(1,:));
e=min(extr(2,:));

Y0=Y0(s:e,:);
Z0=Z0(s:e,:);
V0=V0(s:e,:);
V0m=V0m(s:e,:);
dates0=dates0(s:e,:);
ngdp=ngdp(s:e,:);
rrabs=rrabs(s:e,:);

% Narrative shocks of different sign
Vs0=zeros(size(V0,1),2);
for j=1:size(V0,1)
    if V0(j)>0
        Vs0(j,1)=V0(j);
    elseif V0(j)<0
        Vs0(j,2)=V0(j);
    end
end


%% Generate X's 

X0=[];
for j=1:p
    X0=[X0 lagmatrix(Y0,j)];
end

Y=Y0(p+1:end,:);
Z=Z0(p+1:end,:);
Vs=Vs0(p+1:end,:);
Vm=V0m(p+1:end,:);
V=V0(p+1:end,:);
X=X0(p+1:end,:);
dates=dates0(p+1:end,:);
ngdp=ngdp(p+1:end,:);
rrabs=rrabs(p+1:end,:);

T=size(Y,1); % Sample size
N=size(Y,2); % Nummber of endogenous variables in the VAR
X=[ones(T,1) X Vm Vs]; % Matrix of regressors in the VAR

%% Nonlinear model - Estimation

betanlmat=zeros(N*p*2+8,N,f);
RSS=zeros(f,N);
grid=linspace(prctile(Z,pl),prctile(Z,pu),f);

% Loop over possible values for the threshold of the switching variable
% which determines regimes, store RSS for each threshold
for j=1:f
    
    % generate X's
    d=zeros(T,1);
    for jj=2:T
        d(jj)=Z(jj-1)>=grid(j);
    end
    Xnl=X;
    Xnl=[Xnl Xnl.*repmat(d,1,N*p+4)];
    
    % estimate parameters of the VAR
    betanlmat(:,:,j)=(Xnl'*Xnl)\(Xnl'*Y);
    resnl=Y-Xnl*betanlmat(:,:,j);
    RSS(j,:)=sum(resnl.^2);
    
end

% Determine the threshold with the minimum RSS
posmin=find(sum(RSS,2)==min(sum(RSS,2)));
posmin=posmin(1);
thresh=grid(posmin);

% Determine a vector d which stores the definition of regimes
% based on the optimal threshold (thresh) determined above.

d=zeros(T,1);
for jj=2:T
    d(jj)=Z(jj-1)>=thresh;
end
Xnl=X;
Xnl=[Xnl Xnl.*repmat(d,1,N*p+4)];
betanl=(Xnl'*Xnl)\(Xnl'*Y);
resnl=Y-Xnl*betanl;
estdata=[dates Y Z d Vs Vm];

% Compute Impulse Responses (Point estimates)
betanl0=betanl(1:N*p+4,:);
betanl1=betanl(1:N*p+4,:)+betanl(N*p+5:end,:);
Cf0=[betanl(2:N*p+1,:)'; eye(N*(p-1)) zeros(N*(p-1),N)];
Cf1=[betanl(2:N*p+1,:)'+betanl(N*p+6:end-3,:)'; eye(N*(p-1)) zeros(N*(p-1),N)];

% Monetary Shock - low uncertainty
ir10(:,1)=[betanl(N*p+2,:)'; zeros(N*(p-1),1)];
for j=2:irh
    ir10(:,j)=Cf0*ir10(:,j-1);
end

% Monetary Shock - High uncertainty
ir11(:,1)=[betanl(N*p+2,:)'+betanl(end-2,:)'; zeros(N*(p-1),1)];
for j=2:irh
    ir11(:,j)=Cf0*ir11(:,j-1);
end

% Tax shock - Positive Shock - Low uncertainty
irp20(:,1)=[betanl(N*p+3,:)'; zeros(N*(p-1),1)];
for j=2:irh
    irp20(:,j)=Cf0*irp20(:,j-1);
end

% Tax shock - Negative Shock - Low uncertainty
irn20(:,1)=[betanl(N*p+4,:)'; zeros(N*(p-1),1)];
for j=2:irh
    irn20(:,j)=Cf0*irn20(:,j-1);
end

% Tax shock - Positive Shock - High uncertainty
irp21(:,1)=[betanl(N*p+3,:)'+betanl(end-1,:)'; zeros(N*(p-1),1)];
for j=2:irh
    irp21(:,j)=Cf1*irp21(:,j-1);
end

% Tax shock - Negative Shock - High uncertainty
irn21(:,1)=[betanl(N*p+4,:)'+betanl(end,:)'; zeros(N*(p-1),1)];
for j=2:irh
    irn21(:,j)=Cf1*irn21(:,j-1);
end

irp0=irp20;
irp1=irp21;
irn0=irn20;
irn1=irn21;


%% Bootstrap for the nonlinear model

% Pre-determine matrices for objects defined in the bootstrap
betanlmat=zeros((N*p+4)*2,N,reps);
ir10mat=zeros(N*p,irh,reps);
ir11mat=zeros(N*p,irh,reps);
irp20mat=zeros(N*p,irh,reps);
irn20mat=zeros(N*p,irh,reps);
irp21mat=zeros(N*p,irh,reps);
irn21mat=zeros(N*p,irh,reps);
irp0mat=zeros(N*p,irh,reps);
irn0mat=zeros(N*p,irh,reps);
irp1mat=zeros(N*p,irh,reps);
irn1mat=zeros(N*p,irh,reps);
irp0neumat=zeros(N*p,irh,reps);
irn0neumat=zeros(N*p,irh,reps);
irp1neumat=zeros(N*p,irh,reps);
irn1neumat=zeros(N*p,irh,reps);


% Start the main bootstrap loop
for j=1:reps
    
    vv=randi([1,T],T,1);
    
    % Generate new "bootstrapped" Y
    
    Yb=Y+resnl(vv,:);
    for jj=p+1:T
        if d(jj)==0 % generate Y if the observation falls in the low-uncertainty regime based on low-uncertainty regime estimated parameters
            Yb(jj,:)=betanl0(1,:)+betanl0(end-1,:).*Vs(jj,1)+betanl0(end,:).*Vs(jj,2)+betanl0(end-2,:).*Vm(jj)+resnl(vv(jj),:);
            for hh=1:p
                Yb(jj,:)=Yb(jj,:)+Yb(jj-hh,:)*betanl0(N*(hh-1)+2:N*hh+1,:);
            end
        else % generate Y if the observation falls in the high-uncertainty regime based on high-uncertainty regime estimated parameters
            Yb(jj,:)=betanl1(1,:)+betanl1(end-1,:).*Vs(jj,1)+betanl1(end,:).*Vs(jj,2)+betanl1(end-2,:).*Vm(jj)+resnl(vv(jj),:);
            for hh=1:p
                Yb(jj,:)=Yb(jj,:)+Yb(jj-hh,:)*betanl1(N*(hh-1)+2:N*hh+1,:);
            end
        end
    end
    
    % Generate new X based on bootstrapped Y
    Xb=[];
    for jj=1:p
        Xb=[Xb lagmatrix(Yb,jj)];
    end
    for jj=1:p
        Xb(1:jj,N*(jj-1)+1:N*(jj-1)+N)=Y0(1:jj,:);
    end
    Xb=[ones(T,1) Xb Vm Vs];
    Xb=[Xb Xb.*repmat(d,1,N*p+4)];
    
    % Compute estimated coefficients with the new bootstrapped data and Companion form matrix
    betanlmat(:,:,j)=(Xb'*Xb)\(Xb'*Yb);
    
    Cf0=[betanlmat(2:N*p+1,:,j)'; eye(N*(p-1)) zeros(N*(p-1),N)];
    Cf1=[betanlmat(2:N*p+1,:,j)'+betanlmat(N*p+6:end-3,:,j)'; eye(N*(p-1)) zeros(N*(p-1),N)];

    % Compute IRFs and store them
    
    % Monetary shock - Low uncertainty
    ir10mat(:,1,j)=[betanlmat(N*p+2,:,j)'; zeros(N*(p-1),1)];
    for jj=2:irh
        ir10mat(:,jj,j)=Cf0*ir10mat(:,jj-1,j);
    end
    
    % Monetary shock  - High uncertainty
    ir11mat(:,1,j)=[betanlmat(N*p+2,:,j)'+betanlmat(end-2,:,j)'; zeros(N*(p-1),1)];
    for jj=2:irh
        ir11mat(:,jj,j)=Cf1*ir11mat(:,jj-1,j);
    end
        
    % Tax shock - Positive  - Low uncertainty
    irp20mat(:,1,j)=[betanlmat(N*p+3,:,j)'; zeros(N*(p-1),1)];
    for jj=2:irh
        irp20mat(:,jj,j)=Cf0*irp20mat(:,jj-1,j);
    end
    
    % Tax shock - Negative - Low uncertainty
    irn20mat(:,1,j)=[betanlmat(N*p+4,:,j)'; zeros(N*(p-1),1)];
    for jj=2:irh
        irn20mat(:,jj,j)=Cf0*irn20mat(:,jj-1,j);
    end
    
    % Tax shock - Positive - High uncertainty
    irp21mat(:,1,j)=[betanlmat(N*p+3,:,j)'+betanlmat(end-1,:,j)'; zeros(N*(p-1),1)];
    for jj=2:irh
        irp21mat(:,jj,j)=Cf1*irp21mat(:,jj-1,j);
    end
    
    % Tax shock - Negative - High uncertainty
    irn21mat(:,1,j)=[betanlmat(N*p+4,:,j)'+betanlmat(end,:,j)'; zeros(N*(p-1),1)];
    for jj=2:irh
        irn21mat(:,jj,j)=Cf1*irn21mat(:,jj-1,j);
    end
    
    irp0mat(:,:,j)=irp20mat(:,:,j);
    irp1mat(:,:,j)=irp21mat(:,:,j);
    irn0mat(:,:,j)=irn20mat(:,:,j);
    irn1mat(:,:,j)=irn21mat(:,:,j);
    
    % Compute IRFs to a tax shock removing the effect of monetary policy
    
    irp0neumat(:,:,j)=irp0mat(:,:,j); irp1neumat(:,:,j)=irp1mat(:,:,j); 
    irn0neumat(:,:,j)=irn0mat(:,:,j); irn1neumat(:,:,j)=irn1mat(:,:,j);

    for jj=1:irh
    reacp0=irp0neumat(N,jj,j); % j-th period effect of a 1% positive tax shock on FFR in the low regime
    reacp1=irp1neumat(N,jj,j); % j-th period effect of a 1% positive tax shock on FFR in the high regime
    reacn0=irn0neumat(N,jj,j); % j-th period effect of a 1% negative tax shock on FFR in the low regime
    reacn1=irn1neumat(N,jj,j); % j-th period effect of a 1% negative tax shock on FFR in the high regime

    % Calibrate the size of the initial monetary policy shock to
    % neutralize reaction of FFR
    mpshockp0=reacp0/ir10mat(N,1,j); 
    mpshockn0=reacn0/ir10mat(N,1,j);
    mpshockp1=reacp1/ir11mat(N,1,j);
    mpshockn1=reacn1/ir11mat(N,1,j);

    irf_reac_p0=ir10mat(:,1:end-jj+1,j).*mpshockp0; % IRFs of all aggregates to a MP shock of size mpshockp0
    irf_reac_p1=ir11mat(:,1:end-jj+1,j).*mpshockp1; % IRFs of all aggregates to a MP shock of size mpshockp0
    irf_reac_n0=ir10mat(:,1:end-jj+1,j).*mpshockn0; % IRFs of all aggregates to a MP shock of size mpshockn0
    irf_reac_n1=ir11mat(:,1:end-jj+1,j).*mpshockn1; % IRFs of all aggregates to a MP shock of size mpshockn1

    % Compute the neutralized IRFs 
    irp0neumat(:,jj:end,j)=irp0neumat(:,jj:end,j)-irf_reac_p0; 
    irp1neumat(:,jj:end,j)=irp1neumat(:,jj:end,j)-irf_reac_p1;
    irn0neumat(:,jj:end,j)=irn0neumat(:,jj:end,j)-irf_reac_n0;
    irn1neumat(:,jj:end,j)=irn1neumat(:,jj:end,j)-irf_reac_n1;

    end
    
end

% Determine the confidence bands for IRFs by computing selected percentiles
% from the matrices storing bootstrapped IRFs

irfp0up=prctile(irp0mat,ucl,3);
irfp0down=prctile(irp0mat,lcl,3);
irfn0up=prctile(irn0mat,ucl,3);
irfn0down=prctile(irn0mat,lcl,3);
irfp1up=prctile(irp1mat,ucl,3);
irfp1down=prctile(irp1mat,lcl,3);
irfn1up=prctile(irn1mat,ucl,3);
irfn1down=prctile(irn1mat,lcl,3);
irfp0upneu=prctile(irp0neumat,ucl,3);
irfp0downneu=prctile(irp0neumat,lcl,3);
irfn0upneu=prctile(irn0neumat,ucl,3);
irfn0downneu=prctile(irn0neumat,lcl,3);
irfp1upneu=prctile(irp1neumat,ucl,3);
irfp1downneu=prctile(irp1neumat,lcl,3);
irfn1upneu=prctile(irn1neumat,ucl,3);
irfn1downneu=prctile(irn1neumat,lcl,3);
irp0neu=prctile(irp0neumat,50,3);
irn0neu=prctile(irn0neumat,50,3);
irp1neu=prctile(irp1neumat,50,3);
irn1neu=prctile(irn1neumat,50,3);

%% Multipliers

% Define useful objects
dp0=(V>0 & d==0); % Identifier for observations with a positive shock and low-regime
dp1=(V>0 & d==1); % Identifier for observations with a positive shock and high-regime
dn0=(V>0 & d==0); % Identifier for observations with a negative shock and low-regime
dn1=(V>0 & d==1); % Identifier for observations with a negative shock and high-regime
hor=1;

clear mult2 bands2
% Compute tax multipliers as described in the paper
[mult2(1), bands2(1,:)]=multiplier_m2(hor,irp0(2,:),squeeze(irp0mat(2,:,:))',ngdp(dp0==1),rrabs(dp0==1));
[mult2(2), bands2(2,:)]=multiplier_m2(hor,irp1(2,:),squeeze(irp1mat(2,:,:))',ngdp(dp1==1),rrabs(dp1==1));
[mult2(3), bands2(3,:)]=multiplier_m2(hor,-irn0(2,:),-squeeze(irn0mat(2,:,:))',ngdp(dn0==1),rrabs(dn0==1));
[mult2(4), bands2(4,:)]=multiplier_m2(hor,-irn1(2,:),-squeeze(irn1mat(2,:,:))',ngdp(dn1==1),rrabs(dn1==1));
mult=[mult2' bands2];






    




