%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% "Tax Shocks with High and Low Uncertainty" 
% Bertolotti F. and Marcellino M.
% Journal of Applied Econometrics
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [pv_abs, pv_cum]=tvarsn_2s_tests(colm,colsv,shock)

%% TVARSN
rng('default')
rng(1)

%% set preliminary parameters

p=2;        % number of lags in the VAR
irh=10;     % time horizon for the impulse responses
reps=1000;  % number of repetitions of the bootstrap
f=100;      % number of intervals for gird search
pl=30;      % lower percentile for grid search
pu=70;      % upper percentile for grid search
colm=colm;    % sets the macro variable of interest (pos 2 in the VAR)
colsv=colsv;   % sets the column of the switching variable

%% Import Data

data_macro_series % first set of macro variables (see m.file for the content)
data_proxy_series  % narrative shock variables (see m.file for the content)
data_switching_series  % uncertainty variables (see m.file for the content)

%% Generate the vector of Y's

dates0 = data_macro(:,3);

% Select or apply proper tranformation to the switching variable
if colsv==13 % residuals of jln's macro unc. on QoQ GDP growth
    % [~, ~, Z0]=regress(data_SV(:,5),[ones(size(data_SV(:,5),1),1) data_SV(:,12)]);
    [~, ~, Z0]=regress(data_SV(:,5),[ones(size(data_SV(:,5),1),1),(data_macro(:,5)-lagmatrix(data_macro(:,5),1))]);
elseif colsv==11   % detrend epu
    [~, ~, Z0]=regress(data_SV(:,11),[ones(size(data_SV(:,11),1),1) (1:size(data_SV(:,11),1))']);
else
    Z0=data_SV(:,colsv);
end

% Build the matrix of endogenous variables: Cost of debt, GDP (or other macro variable), Inflation,
% Tax revenues, Government spending, Uncertainty variable, FFR
Y0=[data_macro(:,8) data_macro(:,colm)  data_macro(:,9) data_macro(:,7) data_macro(:,6) Z0 data_macro(:,13)];

% Smoothing of the threshold variable for the identification of regimes
if colsv~=12
    Z0=tsmovavg(Z0,'s',6,1);
end

% Build the vector of Romer and Romer narrative tax shocks
if shock==1 
    V0=data_proxy(:,6);
elseif shock==2
    [~, ~, V0]=regress(data_proxy(:,6),[data_proxy(:,8) lagmatrix(data_proxy(:,8),[1 2 3 4])]);
end
V0m=data_proxy(:,9);

ngdp=data_macro(:,14);
rrabs=data_proxy(:,5);

% Find extremes of the estimation sample
extr=get_extremes([Y0 Z0 V0m V0]);
s=max(extr(1,:));
e=min(extr(2,:));

Y0=Y0(s:e,:);
Z0=Z0(s:e,:);
V0=V0(s:e,:);
V0m=V0m(s:e,:);
dates0=dates0(s:e,:);
ngdp=ngdp(s:e,:);
rrabs=rrabs(s:e,:);

% Narrative shocks of different sign
Vs0=zeros(size(V0,1),2);
for j=1:size(V0,1)
    if V0(j)>0
        Vs0(j,1)=V0(j);
    elseif V0(j)<0
        Vs0(j,2)=V0(j);
    end
end


%% generate X's 

X0=[];
for j=1:p
    X0=[X0 lagmatrix(Y0,j)];
end

Y=Y0(p+1:end,:);
Z=Z0(p+1:end,:);
Vs=Vs0(p+1:end,:);
V=V0(p+1:end,:);
Vm=V0m(p+1:end,:);
X=X0(p+1:end,:);
dates=dates0(p+1:end,:);
ngdp=ngdp(p+1:end,:);
rrabs=rrabs(p+1:end,:);

T=size(Y,1);
N=size(Y,2);
X=[ones(T,1) X Vm Vs];

%% Linear model - point estimate and IRFs

% compute the coefficients of the VAR and create the CF matrix
beta=(X'*X)\(X'*Y);
res=Y-X*beta;
Cf=[beta(2:end-3,:)'; eye(N*(p-1)) zeros(N*(p-1),N)];

% compute the impulse responses
% Positive shock
irp(:,1)=[beta(end-1,:)'; zeros(N*(p-1),1)];
for j=2:irh
    irp(:,j)=Cf*irp(:,j-1);
end

% Negative shock
irn(:,1)=[beta(end,:)'; zeros(N*(p-1),1)];
for j=2:irh
    irn(:,j)=Cf*irn(:,j-1);
end

%% Linear model - Bootstrap
betamat=zeros(N*p+4,N,reps+1); 
irfpmat=zeros(N*p,irh,reps+1); 
irfnmat=zeros(N*p,irh,reps+1); 
Ymat=zeros(T,N,reps);
Xmat=zeros(T,N*p+4,reps);

for j=1:reps
    
    % generate new Y
    vv=randi([1,T],T,1);
    Yb=Y+res(vv,:);
    for jj=p+1:T
        Yb(jj,:)=beta(1,:)+beta(end-1,:).*Vs(jj,1)+beta(end,:).*Vs(jj,2)+beta(end-2,:).*Vm(jj)+res(vv(jj),:);
        for hh=1:p
            Yb(jj,:)=Yb(jj,:)+Yb(jj-hh,:)*beta(N*(hh-1)+2:N*hh+1,:);
        end
    end
    Ymat(:,:,j)=Yb;
    
    % generate new X
    Xb=[];
    for jj=1:p
        Xb=[Xb lagmatrix(Yb,jj)];
    end
    for jj=1:p
        Xb(1:jj,N*(jj-1)+1:N*(jj-1)+N)=Y0(1:jj,:);
    end
    Xb=[ones(T,1) Xb Vm Vs];
    Xmat(:,:,j)=Xb;
    
    % compute estimated coefficients and CF matrix
    betamat(:,:,j)=(Xb'*Xb)\(Xb'*Yb);
    Cf=[betamat(2:end-3,:,j)'; eye(N*(p-1)) zeros(N*(p-1),N)];

    % compute the impulse responses
    % positive shock
    irfpmat(:,1,j)=[betamat(end-1,:,j)'; zeros(N*(p-1),1)];
    for jj=2:irh
        irfpmat(:,jj,j)=Cf*irfpmat(:,jj-1,j);
    end
    
    % negative shock
    irfnmat(:,1,j)=[betamat(end,:,j)'; zeros(N*(p-1),1)];
    for jj=2:irh
        irfnmat(:,jj,j)=Cf*irfnmat(:,jj-1,j);
    end
    
end



%% Nonlinear model - estimation

betanlmat=zeros(N*p*2+8,N,f);
RSS=zeros(f,N);
grid=linspace(prctile(Z,pl),prctile(Z,pu),f);

for j=1:f
    
    % generate X's
    d=zeros(T,1);
    for jj=2:T
        d(jj)=Z(jj-1)>=grid(j);
    end
    Xnl=X;
    Xnl=[Xnl Xnl.*repmat(d,1,N*p+4)];
    
    % estimate parameters of the VAR
    betanlmat(:,:,j)=(Xnl'*Xnl)\(Xnl'*Y);
    resnl=Y-Xnl*betanlmat(:,:,j);
    RSS(j,:)=sum(resnl.^2);
    
end

% determine the threshold with the minimum RSS

posmin=find(sum(RSS,2)==min(sum(RSS,2)));
% posmin=find(sum(RSS(:,yp),2)==min(sum(RSS(:,yp),2)));
posmin=posmin(1);
thresh=grid(posmin);

d=zeros(T,1);
for jj=2:T
    d(jj)=Z(jj-1)>=thresh;
end
Xnl=X;
Xnl=[Xnl Xnl.*repmat(d,1,N*p+4)];
betanl=(Xnl'*Xnl)\(Xnl'*Y);

% impulse responses
Cf0=[betanl(2:N*p+1,:)'; eye(N*(p-1)) zeros(N*(p-1),N)];
Cf1=[betanl(2:N*p+1,:)'+betanl(N*p+6:end-3,:)'; eye(N*(p-1)) zeros(N*(p-1),N)];

% tax increase - low uncertainty
irp0(:,1)=[betanl(N*p+3,:)'; zeros(N*(p-1),1)];
for j=2:irh
    irp0(:,j)=Cf0*irp0(:,j-1);
end

% tax cut - low uncertainty
irn0(:,1)=[betanl(N*p+4,:)'; zeros(N*(p-1),1)];
for j=2:irh
    irn0(:,j)=Cf0*irn0(:,j-1);
end

% tax increase - high uncertainty
irp1(:,1)=[betanl(N*p+3,:)'+betanl(end-1,:)'; zeros(N*(p-1),1)];
for j=2:irh
    irp1(:,j)=Cf1*irp1(:,j-1);
end

% tax cut - high uncertainty
irn1(:,1)=[betanl(N*p+4,:)'+betanl(end,:)'; zeros(N*(p-1),1)];
for j=2:irh
    irn1(:,j)=Cf1*irn1(:,j-1);
end

%% Nonlinearity tests

irp0mNLT=zeros(N*p,irh,reps);
irp1mNLT=zeros(N*p,irh,reps);
irn0mNLT=zeros(N*p,irh,reps);
irn1mNLT=zeros(N*p,irh,reps);

for j=1:reps
    
    % take data from the bootstrap of the linear model
    Yb=Ymat(:,:,j);
    Xb=Xmat(:,:,j);
    Xb=[Xb Xb.*repmat(d,1,N*p+4)];
    
    % compute estimated coefficients and CF matrix
    betanlb=(Xb'*Xb)\(Xb'*Yb);
    
    Cf0=[betanlb(2:N*p+1,:)'; eye(N*(p-1)) zeros(N*(p-1),N)];
    Cf1=[betanlb(2:N*p+1,:)'+betanlb(N*p+6:end-3,:)'; eye(N*(p-1)) zeros(N*(p-1),N)];
    
    % positive shock, low uncertainty regime
    irp0mNLT(:,1,j)=[betanlb(N*p+3,:)'; zeros(N*(p-1),1)];
    for jj=2:irh
        irp0mNLT(:,jj,j)=Cf0*irp0mNLT(:,jj-1,j);
    end
    
    % negative shock, low uncertainty regime
    irn0mNLT(:,1,j)=[betanlb(N*p+4,:)'; zeros(N*(p-1),1)];
    for jj=2:irh
        irn0mNLT(:,jj,j)=Cf0*irn0mNLT(:,jj-1,j);
    end
    
    % positive shock, high uncertainty regime
    irp1mNLT(:,1,j)=[betanlb(N*p+3,:)'+betanlb(end-1,:)'; zeros(N*(p-1),1)];
    for jj=2:irh
        irp1mNLT(:,jj,j)=Cf1*irp1mNLT(:,jj-1,j);
    end
    
    % negative shock, low uncertainty regime
    irn1mNLT(:,1,j)=[betanlb(N*p+4,:)'+betanlb(end,:)'; zeros(N*(p-1),1)];
    for jj=2:irh
        irn1mNLT(:,jj,j)=Cf1*irn1mNLT(:,jj-1,j);
    end
end

% First Test: Absolute Difference

% Build the empirical distribution from the boostrapped IRFs
absdist=zeros(reps,9);

for j=1:reps
    absdist(j,1)=max(abs(irfpmat(2,:,j)-irp0mNLT(2,:,j))); % Positive Shock lin. model vs. Positive Shock in Reg. 0
    absdist(j,2)=max(abs(irfpmat(2,:,j)-irp1mNLT(2,:,j))); % Positive Shock lin. model vs. Positive Shock in Reg. 1
    absdist(j,3)=max(abs(irfnmat(2,:,j)-irn0mNLT(2,:,j))); % Negative Shock lin. model vs. Negative Shock in Reg. 0
    absdist(j,4)=max(abs(irfnmat(2,:,j)-irn1mNLT(2,:,j))); % Negative Shock lin. model vs. Negative Shock in Reg. 1
    absdist(j,5)=max(abs(irp0mNLT(2,:,j)-irp1mNLT(2,:,j))); % Positive Shock in Reg. 0 vs. Positive Shock in Reg. 1
    absdist(j,6)=max(abs(irn0mNLT(2,:,j)-irn1mNLT(2,:,j))); % Negative Shock in Reg. 0 vs. Negative Shock in Reg. 1
    absdist(j,7)=max(abs(irfnmat(2,:,j)-irfpmat(2,:,j))); % Negative Shock lin. model vs. Positive Shock lin. model
    absdist(j,8)=max(abs(irp0mNLT(2,:,j)-irn0mNLT(2,:,j))); % Positive Shock in Reg. 0 vs. Negative Shock in Reg. 0
    absdist(j,9)=max(abs(irp1mNLT(2,:,j)-irn1mNLT(2,:,j))); % Positive Shock in Reg. 1 vs. Negative Shock in Reg. 1
end

abst(1)=max(abs(irp(2,:)-irp0(2,:)));
abst(2)=max(abs(irp(2,:)-irp1(2,:)));
abst(3)=max(abs(irn(2,:)-irn0(2,:)));
abst(4)=max(abs(irn(2,:)-irn1(2,:)));
abst(5)=max(abs(irp0(2,:)-irp1(2,:)));
abst(6)=max(abs(irn0(2,:)-irn1(2,:)));
abst(7)=max(abs(irn(2,:)-irp(2,:)));
abst(8)=max(abs(irp0(2,:)-irn0(2,:)));
abst(9)=max(abs(irp1(2,:)-irn1(2,:)));

pv_abs=nan(9,1);
for j=1:9
    pv_abs(j)=sum(absdist(:,j)>abst(j))/reps;
end

% Second test: cumulative difference
% Build the empirical distribution from the boostrapped IRFs
cumdist=zeros(reps,9);

for j=1:reps
    cumdist(j,1)=abs(sum(irfpmat(2,:,j)-irp0mNLT(2,:,j))); % Positive Shock lin. model vs. Positive Shock in Reg. 0
    cumdist(j,2)=abs(sum(irfpmat(2,:,j)-irp1mNLT(2,:,j))); % Positive Shock lin. model vs. Positive Shock in Reg. 1
    cumdist(j,3)=abs(sum(irfnmat(2,:,j)-irn0mNLT(2,:,j))); % Negative Shock lin. model vs. Negative Shock in Reg. 0
    cumdist(j,4)=abs(sum(irfnmat(2,:,j)-irn1mNLT(2,:,j))); % Negative Shock lin. model vs. Negative Shock in Reg. 1
    cumdist(j,5)=abs(sum(irp0mNLT(2,:,j)-irp1mNLT(2,:,j))); % Positive Shock in Reg. 0 vs. Positive Shock in Reg. 1
    cumdist(j,6)=abs(sum(irn0mNLT(2,:,j)-irn1mNLT(2,:,j))); % Negative Shock in Reg. 0 vs. Negative Shock in Reg. 1
    cumdist(j,7)=abs(sum(irfnmat(2,:,j)-irfpmat(2,:,j))); % Negative Shock lin. model vs. Positive Shock lin. model
    cumdist(j,8)=abs(sum(irp0mNLT(2,:,j)-irn0mNLT(2,:,j))); % Positive Shock in Reg. 0 vs. Negative Shock in Reg. 0
    cumdist(j,9)=abs(sum(irp1mNLT(2,:,j)-irn1mNLT(2,:,j))); % Positive Shock in Reg. 1 vs. Negative Shock in Reg. 1
end

cumst(1)=abs(sum(irp(2,:)-irp0(2,:)));
cumst(2)=abs(sum(irp(2,:)-irp1(2,:)));
cumst(3)=abs(sum(irn(2,:)-irn0(2,:)));
cumst(4)=abs(sum(irn(2,:)-irn1(2,:)));
cumst(5)=abs(sum(irp0(2,:)-irp1(2,:)));
cumst(6)=abs(sum(irn0(2,:)-irn1(2,:)));
cumst(7)=abs(sum(irn(2,:)-irp(2,:)));
cumst(8)=abs(sum(irp0(2,:)-irn0(2,:)));
cumst(9)=abs(sum(irp1(2,:)-irn1(2,:)));

pv_cum=nan(9,1);
for j=1:9
    pv_cum(j)=sum(cumdist(:,j)>cumst(j))/reps;
end



