%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% "Tax Shocks with High and Low Uncertainty" 
% Bertolotti F. and Marcellino M.
% Journal of Applied Econometrics
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function fit=tvar_2s_fit(ucl,lcl,colm,colsv,shock)

%% TVAR
rng('default')
rng(1)

%% set preliminary parameters

p=2;        % number of lags in the VAR
irh=10;     % time horizon for the impulse responses
reps=1000;  % number of repetitions of the bootstrap
ucl=ucl;     % upper confidence level for IRFs
lcl=lcl;     % lower confidence level for IRFs
f=100;      % number of intervals for gird search
pl=30;      % lower percentile for grid search
pu=70;      % upper percentile for grid search
colm=colm;    % sets the macro variable of interest (pos 2 in the VAR)
colsv=colsv;   % sets the column of the switching variable

%% Import Data

data_macro_series % first set of macro variables (see m.file for the content)
data_proxy_series  % narrative shock variables (see m.file for the content)
data_switching_series  % uncertainty variables (see m.file for the content)

%% Generate the vector of Y's

dates0 = data_macro(:,1);

% Select or apply proper tranformation to the switching variable
if colsv==13 % residuals of jln's macro unc.  on QoQ GDP growth
    [~, ~, Z0]=regress(data_SV(:,5),[ones(size(data_SV(:,5),1),1),(data_macro(:,5)-lagmatrix(data_macro(:,5),1))]);
elseif colsv==11   % detrend epu
    [~, ~, Z0]=regress(data_SV(:,11),[ones(size(data_SV(:,11),1),1) (1:size(data_SV(:,11),1))']);
else
    Z0=data_SV(:,colsv);
end

% Build the matrix of endogenous variables: Cost of debt, GDP (or other macro variable), Inflation,
% Tax revenues, Government spending, FFR
Y0=[data_macro(:,8) data_macro(:,colm)  data_macro(:,9) data_macro(:,7) data_macro(:,6) data_macro(:,13)];

% Smoothing of the threshold variable for the identification of regimes
if colsv~=12
    Z0=tsmovavg(Z0,'s',6,1);
end

% Build the vector of Romer and Romer narrative tax shocks
if shock==1 
    V0=data_proxy(:,6);
elseif shock==2
    [~, ~, V0]=regress(data_proxy(:,6),[data_proxy(:,8) lagmatrix(data_proxy(:,8),[1 2 3 4])]);
end
V0m=data_proxy(:,9); % RR monetary policy shocks

ngdp=data_macro(:,14);
rrabs=data_proxy(:,5);

% Find extremes of the estimation sample
extr=get_extremes([Y0 Z0 V0 V0m]);
s=max(extr(1,:));
e=min(extr(2,:));

Y0=Y0(s:e,:);
Z0=Z0(s:e,:);
V0=V0(s:e,:);
V0m=V0m(s:e,:);
dates0=dates0(s:e,:);
ngdp=ngdp(s:e,:);
rrabs=rrabs(s:e,:);

%% generate X's 

X0=[];
for j=1:p
    X0=[X0 lagmatrix(Y0,j)];
end

Y=Y0(p+1:end,:);
Z=Z0(p+1:end,:);
V=V0(p+1:end,:);
Vm=V0m(p+1:end,:);
X=X0(p+1:end,:);

T=size(Y,1);
N=size(Y,2);
X=[ones(T,1) X Vm V];

%% nonlinear model - estimation

betanlmat=zeros(N*p*2+6,N,f);
RSS=zeros(f,N);
grid=linspace(prctile(Z,pl),prctile(Z,pu),f);

for j=1:f
    
    % generate X's
    d=zeros(T,1);
    for jj=2:T
        d(jj)=Z(jj-1)>=grid(j);
    end
    Xnl=X;
    Xnl=[Xnl Xnl.*repmat(d,1,N*p+3)];
    
    % estimate parameters of the VAR
    betanlmat(:,:,j)=(Xnl'*Xnl)\(Xnl'*Y);
    resnl=Y-Xnl*betanlmat(:,:,j);
    RSS(j,:)=sum(resnl.^2);
    
end

% determine the threshold with the minimum RSS

posmin=find(sum(RSS,2)==min(sum(RSS,2)));
posmin=posmin(1);
thresh=grid(posmin);

d=zeros(T,1);
for jj=2:T
    d(jj)=Z(jj-1)>=thresh;
end
Xnl=X;
Xnl=[Xnl Xnl.*repmat(d,1,N*p+3)];
betanl=(Xnl'*Xnl)\(Xnl'*Y);
resnl=Y-Xnl*betanl;


%% Compute Model Fit Measures
cov=resnl'*resnl; 
secondterm=0;
for j=1:T 
    secondterm=secondterm+resnl(j,:)*inv(cov)*resnl(j,:)';
end
LL=(-T/2)*(p*log(2*3.14)-log(det(cov)))-(1/2)*secondterm;

AIC=2*((p+2)*6)-2*LL;
BIC=-2*LL+((p+2)*6)*T;
fit=[AIC, BIC]';





