%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% "Tax Shocks with High and Low Uncertainty" 
% Bertolotti F. and Marcellino M.
% Journal of Applied Econometrics
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [ir0, ir1, irf0up, irf0down, irf1up, irf1down, ir0neu, ir1neu, irf0neuup,...
    irf0neudown, irf1neuup, irf1neudown, estdata, mult, thresh]=tvar_2s(ucl,lcl,colm,colsv,shock)

%% TVAR
rng('default')
rng(1)

%% set preliminary parameters

p=2;        % number of lags in the VAR
irh=10;     % time horizon for the impulse responses
reps=10000;  % number of repetitions of the bootstrap
ucl=ucl;     % upper confidence level for IRFs
lcl=lcl;     % lower confidence level for IRFs
f=100;      % number of intervals for gird search
pl=30;      % lower percentile for grid search
pu=70;      % upper percentile for grid search
colm=colm;    % sets the macro variable of interest (pos 2 in the VAR)
colsv=colsv;   % sets the column of the switching variable

%% Import Data

data_macro_series % first set of macro variables (see m.file for the content)
data_proxy_series  % narrative shock variables (see m.file for the content)
data_switching_series  % uncertainty variables (see m.file for the content)

%% Generate the vector of Y's

dates0 = data_macro(:,3);

% Select or apply proper tranformation to the switching variable
if colsv==13 % residuals of jln's macro unc.  on QoQ GDP growth
    [~, ~, Z0]=regress(data_SV(:,5),[ones(size(data_SV(:,5),1),1),(data_macro(:,5)-lagmatrix(data_macro(:,5),1))]);
elseif colsv==11   % detrend epu
    [~, ~, Z0]=regress(data_SV(:,11),[ones(size(data_SV(:,11),1),1) (1:size(data_SV(:,11),1))']);
else
    Z0=data_SV(:,colsv);
end

% Build the matrix of endogenous variables: Cost of debt, GDP (or other macro variable), Inflation,
% Tax revenues, Government spending, Uncertainty variable, FFR
Y0=[data_macro(:,8) data_macro(:,colm)  data_macro(:,9) data_macro(:,7) data_macro(:,6) Z0 data_macro(:,13)];

% Smoothing of the threshold variable for the identification of regimes
if colsv~=12
    Z0=tsmovavg(Z0,'s',6,1);
end

% Build the vector of Romer and Romer narrative tax shocks
if shock==1 
    V0=data_proxy(:,6);
elseif shock==2
    [~, ~, V0]=regress(data_proxy(:,6),[data_proxy(:,8) lagmatrix(data_proxy(:,8),[1 2 3 4])]);
end
V0m=data_proxy(:,9); % RR monetary policy shocks

ngdp=data_macro(:,14);
rrabs=data_proxy(:,5);

% Find extremes of the estimation sample
extr=get_extremes([Y0 Z0 V0 V0m]);
s=max(extr(1,:));
e=min(extr(2,:));

Y0=Y0(s:e,:);
Z0=Z0(s:e,:);
V0=V0(s:e,:);
V0m=V0m(s:e,:);
dates0=dates0(s:e,:);
ngdp=ngdp(s:e,:);
rrabs=rrabs(s:e,:);

%% generate X's 

X0=[];
for j=1:p
    X0=[X0 lagmatrix(Y0,j)];
end

Y=Y0(p+1:end,:);
Z=Z0(p+1:end,:);
V=V0(p+1:end,:);
Vm=V0m(p+1:end,:);
X=X0(p+1:end,:);
dates=dates0(p+1:end,:);
ngdp=ngdp(p+1:end,:);
rrabs=rrabs(p+1:end,:);

T=size(Y,1);
N=size(Y,2);
X=[ones(T,1) X Vm V];

%% nonlinear model - estimation

betanlmat=zeros(N*p*2+6,N,f);
RSS=zeros(f,N);
grid=linspace(prctile(Z,pl),prctile(Z,pu),f);

for j=1:f
    
    % generate X's
    d=zeros(T,1);
    for jj=2:T
        d(jj)=Z(jj-1)>=grid(j);
    end
    Xnl=X;
    Xnl=[Xnl Xnl.*repmat(d,1,N*p+3)];
    
    % estimate parameters of the VAR
    betanlmat(:,:,j)=(Xnl'*Xnl)\(Xnl'*Y);
    resnl=Y-Xnl*betanlmat(:,:,j);
    RSS(j,:)=sum(resnl.^2);
    
end

% determine the threshold with the minimum RSS

posmin=find(sum(RSS,2)==min(sum(RSS,2)));
posmin=posmin(1);
thresh=grid(posmin);

d=zeros(T,1);
for jj=2:T
    d(jj)=Z(jj-1)>=thresh;
end
Xnl=X;
Xnl=[Xnl Xnl.*repmat(d,1,N*p+3)];
betanl=(Xnl'*Xnl)\(Xnl'*Y);
resnl=Y-Xnl*betanl;
estdata=[dates Y Z d Vm V];

% impulse responses
betanl0=betanl(1:N*p+3,:);
betanl1=betanl(1:N*p+3,:)+betanl(N*p+4:end,:);
Cf0=[betanl(2:N*p+1,:)'; eye(N*(p-1)) zeros(N*(p-1),N)];
Cf1=[betanl(2:N*p+1,:)'+betanl(N*p+5:end-2,:)'; eye(N*(p-1)) zeros(N*(p-1),N)];

% Tax Shock - Low uncertainty regime
ir0(:,1)=[betanl(N*p+3,:)'; zeros(N*(p-1),1)];
for j=2:irh
    ir0(:,j)=Cf0*ir0(:,j-1);
end

% Tax Shock - High uncertainty regime
ir1(:,1)=[betanl(N*p+3,:)'+betanl(end,:)'; zeros(N*(p-1),1)];
for j=2:irh
    ir1(:,j)=Cf1*ir1(:,j-1);
end

% MP Shock - Low uncertainty regime
ir10(:,1)=[betanl(N*p+2,:)'; zeros(N*(p-1),1)];
for j=2:irh
    ir10(:,j)=Cf0*ir10(:,j-1);
end

% MP Shock - High uncertainty regime
ir11(:,1)=[betanl(N*p+2,:)'+betanl(end-1,:)'; zeros(N*(p-1),1)];
for j=2:irh
    ir11(:,j)=Cf1*ir11(:,j-1);
end

%% bootstrap for the nonlinear model

betanlmat=zeros((N*p+3)*2,N,reps);
ir0mat=zeros(N*p,irh,reps);
ir1mat=zeros(N*p,irh,reps);
ir10mat=zeros(N*p,irh,reps);
ir11mat=zeros(N*p,irh,reps);
ir0neumat=zeros(N*p,irh,reps);
ir1neumat=zeros(N*p,irh,reps);

for j=1:reps
    
    % generate new Y
    vv=randi([1,T],T,1);
    
    % first method
    Yb=Y+resnl(vv,:);
    for jj=p+1:T
        if d(jj)==0
            Yb(jj,:)=betanl0(1,:)+betanl0(end,:).*V(jj)+betanl0(end-1,:).*Vm(jj)+resnl(vv(jj),:);
            for hh=1:p
                Yb(jj,:)=Yb(jj,:)+Yb(jj-hh,:)*betanl0(N*(hh-1)+2:N*hh+1,:);
            end
        else
            Yb(jj,:)=betanl1(1,:)+betanl1(end,:).*V(jj)+betanl0(end-1,:).*Vm(jj)+resnl(vv(jj),:);
            for hh=1:p
                Yb(jj,:)=Yb(jj,:)+Yb(jj-hh,:)*betanl1(N*(hh-1)+2:N*hh+1,:);
            end
        end
    end
    
    % generate new X
    Xb=[];
    for jj=1:p
        Xb=[Xb lagmatrix(Yb,jj)];
    end
    for jj=1:p
        Xb(1:jj,N*(jj-1)+1:N*(jj-1)+N)=Y0(1:jj,:);
    end
    Xb=[ones(T,1) Xb Vm V];
    Xb=[Xb Xb.*repmat(d,1,N*p+3)];
    
    % Compute estimated coefficients and CF matrix
    betanlmat(:,:,j)=(Xb'*Xb)\(Xb'*Yb);
    
    Cf0=[betanlmat(2:N*p+1,:,j)'; eye(N*(p-1)) zeros(N*(p-1),N)];
    Cf1=[betanlmat(2:N*p+1,:,j)'+betanlmat(N*p+5:end-2,:,j)'; eye(N*(p-1)) zeros(N*(p-1),N)];

    % Tax Shock - Low-uncertainty regime
    ir0mat(:,1,j)=[betanlmat(N*p+3,:,j)'; zeros(N*(p-1),1)];
    for jj=2:irh
        ir0mat(:,jj,j)=Cf0*ir0mat(:,jj-1,j);
    end
    
    % Tax Shock - High-uncertainty regime
    ir1mat(:,1,j)=[betanlmat(N*p+3,:,j)'+betanlmat(end,:,j)'; zeros(N*(p-1),1)];
    for jj=2:irh
        ir1mat(:,jj,j)=Cf1*ir1mat(:,jj-1,j);
    end
    
    % MP Shock - Low-uncertainty regime
    ir10mat(:,1,j)=[betanlmat(N*p+2,:,j)'; zeros(N*(p-1),1)];
    for jj=2:irh
        ir10mat(:,jj,j)=Cf0*ir10mat(:,jj-1,j);
    end
    
    % Tax Shock - High-uncertainty regime
    ir11mat(:,1,j)=[betanlmat(N*p+2,:,j)'+betanlmat(end-1,:,j)'; zeros(N*(p-1),1)];
    for jj=2:irh
        ir11mat(:,jj,j)=Cf1*ir11mat(:,jj-1,j);
    end
    
    % Compute IRFs to a tax shock removing the effect of monetary policy
    ir0neumat(:,:,j)=ir0mat(:,:,j); ir1neumat(:,:,j)=ir1mat(:,:,j); 

    for jj=1:irh
    reac0=ir0neumat(N,jj,j); % j-th period effect of a 1% tax shock on FFR in the low regime
    reac1=ir1neumat(N,jj,j); % j-th period effect of a 1% tax shock on FFR in the high regime

    mpshock0=reac0/ir10mat(N,1,j);
    mpshock1=reac1/ir11mat(N,1,j);

    irf_reac_0=ir10mat(:,1:end-jj+1,j).*mpshock0; % IRFs of all aggregates to a MP shock of size mpshock0
    irf_reac_1=ir11mat(:,1:end-jj+1,j).*mpshock1; % IRFs of all aggregates to a MP shock of size mpshock1

    ir0neumat(:,jj:end,j)=ir0neumat(:,jj:end,j)-irf_reac_0;
    ir1neumat(:,jj:end,j)=ir1neumat(:,jj:end,j)-irf_reac_1;

    end
    
end

irf0up=prctile(ir0mat,ucl,3);
irf0down=prctile(ir0mat,lcl,3);
irf1up=prctile(ir1mat,ucl,3);
irf1down=prctile(ir1mat,lcl,3);
irf0neuup=prctile(ir0neumat,ucl,3);
irf0neudown=prctile(ir0neumat,lcl,3);
irf1neuup=prctile(ir1neumat,ucl,3);
irf1neudown=prctile(ir1neumat,lcl,3);
ir0neu=prctile(ir0neumat,50,3);
ir1neu=prctile(ir1neumat,50,3);

%% Multipliers

% Define useful objects
d0=(d==0 & V~=0); % Identifier for observations in low-regime
d1=(d==1 & V~=0); % Identifier for observations in high-regime
hor=1;

clear mult2 bands2
% Second method+
[mult2(1), bands2(1,:)]=multiplier_m2(hor,ir0(2,:),squeeze(ir0mat(2,:,:))',ngdp(d0==1),abs(rrabs(d0==1)));
[mult2(2), bands2(2,:)]=multiplier_m2(hor,ir1(2,:),squeeze(ir1mat(2,:,:))',ngdp(d1==1),abs(rrabs(d1==1)));
mult=[mult2' bands2];






