function S = BRRTtest(Serie1,Serie2,d1,d2,rob1,rob2,nlags)
% Balboa-Rodrigues-Rubia-Taylor test for bivariate fractional
% integration of order d=(d1,d2)' based on FGLS estimation of a SUR system
%Inputs:
%
% Serie1, Serie2, Tx1 vectors of observations
% d1, d2: long-memory coefficients to test for
% rob1, rob2: demean_Robinson_option Prefiltering options: 0: no filter, 
%                                              1: constant,
%                                              2: linear trend 
%                                              3: constant and trend
%                                              4: constant and quadratic trend
%
% nlags : Number of lags in Augmentation       0: no lag
%                                              p: any integer number p>0 
%                                              1000: Schwert rule with 4*(T/100)^0.25
%
% Output
%
% S: Structure with fields:
% S.d: Values of d=d(d1,d2)' used in the analysis
% S.LM: LM test statistic robust to unknown heteroskedasticity. Distributed
% asymptotically as a Chi-square with two degrees of freedom
%S.LM_pv: p-value 
%S.Sigmae: Covariance matrix of innovations e(t)=e(e1(t),e2(t))'



[~, ~, sal1]=DKH_FItest(Serie1,d1,rob1,nlags);
[~, ~, sal2]=DKH_FItest(Serie2,d2,rob2,nlags);


%Equation 1
Y1=sal1.Y;
X1=sal1.X;
Aux=sal2.X;
Aux(:,end)=[];
X1=[X1(:,end) X1(:,1:end-1) Aux ];
k1=size(X1,2);

%Equation 2
Y2=sal2.Y;
X2=sal2.X;
Aux=sal1.X;
Aux(:,end)=[];
X2=[X2(:,end) Aux  X2(:,1:end-1)];

T=length(Y1);

% Estimating Sigma
Beta1LS=X1\Y1;
u1=Y1-X1*Beta1LS;
Beta2LS=X2\Y2;
u2=Y2-X2*Beta2LS;

sigma11=u1'*u1/T;
sigma22=u2'*u2/T;
sigma12=u1'*u2/T;
sigma21=u2'*u1/T;
Sigmae=[sigma11 sigma12; sigma21 sigma22];

Sigmainve=inv(Sigmae);
pigma11=Sigmainve(1,1);
pigma12=Sigmainve(1,2);
pigma21=Sigmainve(2,1);
pigma22=Sigmainve(2,2);

% Estimating A matrix
A11=pigma11*(X1'*X1);
A12=pigma12*(X1'*X2);
A21=pigma21*(X2'*X1);
A22=pigma22*(X2'*X2);

A=[A11 A12; A21 A22];

% Estimating Second Term

W1=pigma11*(X1'*Y1) + pigma12*(X1'*Y2);
W2=pigma21*(X2'*Y1) + pigma22*(X2'*Y2);

W=[W1; W2];

% FGLS Estimator of Beta
Ainv=A\eye(size(A,2));
BetaFGLS=Ainv*W;

%Computing FGLS Residuals;
e1=Y1-X1*BetaFGLS(1:k1);
e2=Y2-X2*BetaFGLS(k1+1:end);

%Estimating HC Matrix

Pigma=[pigma11 pigma12; pigma21 pigma22];
B=partesandwich(X1,X2,e1,e2,Pigma);

Sigma=Ainv*B*Ainv;

R=zeros(2,k1*2);
R(1,1)=1;
R(2,k1+1)=1;

Aux=R*BetaFGLS;
M=R*Sigma*R';
LM=Aux'*inv(M)*Aux;

% Output
S.d=[d1 d2];
S.LM=LM;
S.LM_pv=1-chi2cdf(LM,2);
S.Sigmae=Sigmae;



function [TestLSHC, TestLS, statistics]=DKH_FItest(Y,d,demean_Robinson_option,aug_option)
%Function to compute the DKH augmented regression .
%
%Inputs
%
% Y: Tx1 Vector of observations
% d: value to test H0:y is FI(d)
% demean_Robinson_option Prefiltering options: 0: no filter, 
%                                              1: constant,
%                                              2: linear trend 
%                                              3: constant and trend
%                                              4: constant and quadratic trend
%
% aug_option: Number of lags in Augmentation   0: no lag
%                                              p: any number 
%                                              1000 Schwert rule with 4*(T/100)^0.25
%

%Filtering the process according to the posited value of d under the null

T=length(Y);

if demean_Robinson_option==1             % Demeaning with constant
    optionuwe='constant';
    [e,betauwe,tuwe]=Uwedemeaning_generalized(Y,d,optionuwe);
elseif demean_Robinson_option==2         % Demeaning with trend  
    optionuwe='trend';
     [e,betauwe,tuwe]=Uwedemeaning_generalized(Y,d,optionuwe);  
elseif demean_Robinson_option==3         % Demeaning with constant and trend
    optionuwe='constant&trend';
     [e,betauwe,tuwe]=Uwedemeaning_generalized(Y,d,optionuwe); 
     
elseif demean_Robinson_option==4         % Demeaning with constant and trend and quadratic trend
    optionuwe='constant&trend&quadratictrend';
     [e,betauwe,tuwe]=Uwedemeaning_generalized(Y,d,optionuwe);  
else
    e=ARFIMA_Filtering2(Y,d);
    betauwe=[];
    tuwe=[];     
end


% Constructing the BH regressor
 x=zeros(T-1,1);
 
 for j=1:T-1
     Aux=1:j;
     Aux=1./Aux;
     Aux2=flipud ( e(1:j));
     x(j)=Aux*Aux2;      
 end
 
if aug_option==0
     X=x;
     Y=e(2:end);
   
else
    if aug_option==1000
        nlags = floor(4*(T/100)^0.25);  %Schwert Rule 4*(T/100)^0.25 o 12*(T/100)^0.25 (more conservative)
    else
        nlags=aug_option;
    end
        
   
     if nlags==1
         Y=e(2:end);
         X=[e(1:end-1) x];         
     else
         xlag=lagmatrix(e,1:nlags);
         Y=e(nlags+1:end);
         X=[xlag(nlags+1:end,:) x(nlags:end)];
         
     end
end


% OLS Tests
 n=length(Y);
 betaLS=X\Y;
 u=Y-X*betaLS;
 k=length(betaLS);
 RSS=u'*u;
 sigma2=RSS/(T-k);
 se=sqrt(sigma2);
 TSS=(Y-mean(Y))'*(Y-mean(Y));
 R2=1-RSS/TSS;
 AIC=2*k+n*log(sigma2);
 
%iid covariance matrix
XX=(X'*X);
V1=XX\eye(size(XX,2));
%V1=inv(X'*X);
Viid=sigma2*V1;
 
%Sandwich covariance matrix
zup=size(X,2);
autri = X.*repmat(u,1,zup);
V2=autri'*autri;

Vhac=V1*V2*V1;

TestLS=betaLS(end)/sqrt(Viid(end,end));  
TestLSHC=betaLS(end)/sqrt(Vhac(end,end));  

t1LS=betaLS./sqrt(diag(Viid)); 
t1LSHC=betaLS./sqrt(diag(Vhac)); 
if k>2
    t1LS=[t1LS(end); t1LS(2:end)];
    t1LSHC=[t1LSHC(end) ; t1LSHC(2:end)];
elseif k==2
    t1LS=[t1LS(2); t1LS(1)];
    t1LSHC=[t1LSHC(2) ; t1LSHC(1)];
 end



% Additional statistics 
statistics.Beta=betaLS;
statistics.tstatsLS=t1LS;
statistics.tstatsHC=t1LSHC;
statistics.numberofparameters=k;
statistics.observations=n;
statistics.AIC=AIC;
statistics.se=se;
statistics.R2=R2;
statistics.R2bar=1- ((n-1)/(n-k))*(1-R2);
statistics.residuals=u;
statistics.Y=Y;
statistics.X=X;

statistics.Betadeteterministics=betauwe;
statistics.Tstatsdeteterministics=tuwe;

   
function e=ARFIMA_Filtering2(Y,d)

T=length(Y);
A=zeros(T,1);
A(1)=1;
e=zeros(T,1);
e(1)=Y(1);

if d==0
    e=Y;
elseif d==1
    e=diff(Y);
    e=[0; e];
else  
    for k=2:T
        j=k-1;
        A(k)=((j-d-1)/j)*A(k-1);
        Aux1=A(1:k);
        Aux2=flipud(Y(1:k));
        e(k)=Aux1'*Aux2;
    end
end
    
    
function [e_demean,beta,tstat]=Uwedemeaning_generalized(Y,d,optionuwe)

eta=ARFIMA_Filtering2(Y,d);
T=length(Y);
switch optionuwe
    case 'constant'
        x=ARFIMA_Filtering2(ones(size(Y)),d);
    case 'trend'
        tt=1:T; 
        x=ARFIMA_Filtering2(tt',d);
        
    case 'constant&trend'
         x1=ARFIMA_Filtering2(ones(size(Y)),d);
         tt=1:T; 
         x2=ARFIMA_Filtering2(tt',d);
         x=[x1 x2];
         
    case 'constant&trend&quadratictrend'

         x1=ARFIMA_Filtering2(ones(size(Y)),d);
         tt=1:T; 
         x2=ARFIMA_Filtering2(tt'./T,d);
         pis=tt.^2;
         x3=ARFIMA_Filtering2(pis'./T^2,d);
         
         x=[x1 x2 x3];
         
        
end
       
if d==1
    switch optionuwe
        case 'constant'
            e_demean=eta;
            beta=[];
            tstat=[];
            
        case {'constant&trend','constant&trend&quadratictrend'}
            x(:,1)=[];
            Xico=x'*x;
            inversa=Xico\eye(size(Xico,2));
            beta=inversa*(x'*eta);
            e_demean=eta-x*beta;
            T2=length(eta);
            nlagsNW=floor(4*(T2/100)^0.25);
            results=nwest(Y,x,nlagsNW);
            tstat=results.tstat;
            
        case 'trend'
            Xico=x'*x;
            inversa=Xico\eye(size(Xico,2));
            beta=inversa*(x'*eta);
            e_demean=eta-x*beta;
            T2=length(eta);
            nlagsNW=floor(4*(T2/100)^0.25);
            results=nwest(Y,x,nlagsNW);
            tstat=results.tstat;             
    end
    
else             
    Xico=x'*x;
    inversa=Xico\eye(size(Xico,2));
    beta=inversa*(x'*eta);
    e_demean=eta-x*beta;
    T2=length(eta);
    nlagsNW=floor(4*(T2/100)^0.25);
    results=nwest(Y,x,nlagsNW);
    tstat=results.tstat;       
            
            
end


function B=partesandwich(X1,X2,e1,e2,W)

B11=partesandwich_aux(X1,X2,e1,e2,W,1,1);
B12=partesandwich_aux(X1,X2,e1,e2,W,1,2);
B21=partesandwich_aux(X1,X2,e1,e2,W,2,1);
B22=partesandwich_aux(X1,X2,e1,e2,W,2,2);

B=[B11 B12; B21 B22];





function Total=partesandwich_aux(X1,X2,e1,e2,W,i,j)

%Sandwich covariance matrix

K=size(X1,2);
Total=0;
E=[e1 e2];

if i==1
    X_i=X1;    
else
    X_i=X2;    
end

if j==1
    X_j=X1;
else
    X_j=X2;
end


for r=1:2
    for s=1:2
        A1= X_i.*repmat(E(:,r),1,K);
        A2= X_j.*repmat(E(:,s),1,K);
        V=A1'*A2;
        V2=W(i,r)*W(j,s)*V;
        Total=Total+V2;
    end
    
end
        

function results=nwest(y,x,nlag)
% PURPOSE: computes Newey-West adjusted heteroscedastic-serial
%          consistent Least-squares Regression
%---------------------------------------------------
% USAGE: results = nwest(y,x,nlag)
% where: y = dependent variable vector (nobs x 1)
%        x = independent variables matrix (nobs x nvar)
%     nlag = lag length to use
%---------------------------------------------------
% RETURNS: a structure
%        results.meth  = 'newlyw'
%        results.beta  = bhat
%        results.tstat = t-stats
%        results.yhat  = yhat
%        results.resid = residuals
%        results.sige  = e'*e/(n-k)
%        results.rsqr  = rsquared
%        results.rbar  = rbar-squared
%        results.dw    = Durbin-Watson Statistic
%        results.nobs  = nobs
%        results.nvar  = nvars
%        results.y     = y data vector
% --------------------------------------------------
% SEE ALSO: nwest_d, prt(results), plt(results)
%---------------------------------------------------
% References:  Gallant, R. (1987),
%  "Nonlinear Statistical Models," pp.137-139.
%---------------------------------------------------

% written by:
% James P. LeSage, Dept of Economics
% University of Toledo
% 2801 W. Bancroft St,
% Toledo, OH 43606
% % jlesage@spatial-econometrics.com



if (nargin ~= 3); error('Wrong # of arguments to nwest'); end

[nobs nvar] = size(x);

results.meth    = 'nwest';
results.y       = y;
results.nobs    = nobs;
results.nvar    = nvar;

xpxi = inv(x'*x);
results.beta    = xpxi*(x'*y);
results.yhat    = x*results.beta;
results.resid   = y - results.yhat;
sigu = results.resid'*results.resid;
results.sige    = sigu/(nobs-nvar);

% perform Newey-West correction
emat = [];
for i=1:nvar
emat = [emat
        results.resid'];
end
       
    hhat=emat.*x';
    G=zeros(nvar,nvar); w=zeros(2*nlag+1,1);
    a=0;

    while a~=nlag+1
        ga=zeros(nvar,nvar);
        w(nlag+1+a,1)=(nlag+1-a)/(nlag+1);
        za=hhat(:,(a+1):nobs)*hhat(:,1:nobs-a)';
          if a==0
           ga=ga+za;
          else
           ga=ga+za+za';
          end
        G=G+w(nlag+1+a,1)*ga;
        a=a+1;
    end % end of while
    
        V=xpxi*G*xpxi;
        nwerr= sqrt(diag(V));

results.tstat = results.beta./nwerr; % Newey-West t-statistics
ym = y - ones(nobs,1)*mean(y);
rsqr1 = sigu;
rsqr2 = ym'*ym;
results.rsqr = 1.0 - rsqr1/rsqr2; % r-squared
rsqr1 = rsqr1/(nobs-nvar);
rsqr2 = rsqr2/(nobs-1.0);
results.rbar = 1 - (rsqr1/rsqr2); % rbar-squared
ediff = results.resid(2:nobs) - results.resid(1:nobs-1);
results.dw = diag((ediff'*ediff)./(sigu))'; % durbin-watson



    
   



