%% *************************************************************************************************************
% This code replicates Table 1 and Figure 1 of Angelini G. and Fanelli L. (2019), 
% "Exogenous uncertainty and the identification of Structural Vector Autoregressions with external instruments"
% Journal of Applied Econometrics.
% For questions write an email to giovanni.angelini3@unibo.it
%% *************************************************************************************************************

clc
clear
close all

addpath functions

% Global variables
global NLags
global TAll
global Sigma_Eta
global sig
global mS

%% Load data
DataSet= load('DataSet.txt');
AllDataSet = DataSet;
M=size(DataSet,2); % Number of variables
NLags = 4; % Number of lags in the VAR
TAll = size(DataSet,1)-NLags; % Sample size
BootstrapIterations = 1000; % Number of bootstrap replications
options = optimset('MaxFunEvals',200000,'TolFun',1e-500,'MaxIter',200000,'TolX',1e-50);  % Options for the fmiunc function

% Variables in the DataSet
% 1 Dprod	
% 2 UF1	
% 3 UM1	
% 4 Dhouse star
% 5 Oil supply shock

%% Usefull Matrices
r = 2;
n= M-r;
g_shock = 1;

DuplicationMatrix = DuplicationMatrixFunction(M);

mDD=(DuplicationMatrix'*DuplicationMatrix)^(-1)*DuplicationMatrix';
mNN=DuplicationMatrix*mDD;

KommutationMatrix = CommutationMatrixFunction(M);
NMatrix = 0.5*(eye(M^2)+KommutationMatrix);

DuplicationMatrixr = DuplicationMatrixFunction(r);
mDDg=(DuplicationMatrixr'*DuplicationMatrixr)^(-1)*DuplicationMatrixr';

DuplicationMatrixN = DuplicationMatrixFunction(n);
mDDn=(DuplicationMatrixN'*DuplicationMatrixN)^(-1)*DuplicationMatrixN';

KommutationMatrixNg = zeros(n*g_shock,n*g_shock);
KommutationMatrixNg(1,1)=1;
KommutationMatrixNg(2,3)=1;
KommutationMatrixNg(3,2)=1;

%% VAR estimation
VAR_Const = [NaN; NaN; NaN; NaN ; NaN];

% eq.s 29-30 of the paper
VAR_A = {[NaN NaN NaN 0 0; NaN NaN NaN 0 0; NaN NaN NaN 0 0; 0 NaN 0 NaN 0; 0 0 0 0 0],...
         [NaN NaN NaN 0 0; NaN NaN NaN 0 0; NaN NaN NaN 0 0; 0 NaN 0 NaN 0; 0 0 0 0 0],....
         [NaN NaN NaN 0 0; NaN NaN NaN 0 0; NaN NaN NaN 0 0; 0 0   0 0   0; 0 0 0 0 0],....
         [NaN NaN NaN 0 0; NaN NaN NaN 0 0; NaN NaN NaN 0 0; 0 0   0 0   0; 0 0 0 0 0]} ; 

Mdl = varm('Constant',VAR_Const,'AR',VAR_A);
[EstMdl,EstSE,logL_Rest,Errors] = estimate(Mdl,AllDataSet);    

A_Const = EstMdl.Constant;
A_1 = EstMdl.AR{1,1};
A_2 = EstMdl.AR{1,2};
A_3 = EstMdl.AR{1,3};
A_4 = EstMdl.AR{1,4};
Sigma_Eta = EstMdl.Covariance;

% Misspecification tests
%
% [h,Norm1_P,Norm1_Test] = jbtest(Errors(:,1))
% [h,Norm2_P,Norm2_Test] = jbtest(Errors(:,2))
% [h,Norm3_P,Norm3_Test] = jbtest(Errors(:,3))
% [h,Norm4_P,Norm4_Test] = jbtest(Errors(:,4))
% [h,Norm5_P,Norm5_Test] = jbtest(Errors(:,5))
% 
% [h,AR1_P,AR1_Test] = lbqtest(Errors(:,1),'lags',4)
% [h,AR2_P,AR2_Test] = lbqtest(Errors(:,2),'lags',4)
% [h,AR3_P,AR3_Test] = lbqtest(Errors(:,3),'lags',4)
% [h,AR4_P,AR4_Test] = lbqtest(Errors(:,4),'lags',4)
% [h,AR5_P,AR5_Test] = lbqtest(Errors(:,5),'lags',4)
% 
% [h,ARCH1_P,ARCH1_Test] = archtest(Errors(:,1),'lags',4)
% [h,ARCH2_P,ARCH2_Test] = archtest(Errors(:,2),'lags',4)
% [h,ARCH3_P,ARCH3_Test] = archtest(Errors(:,3),'lags',4)
% [h,ARCH4_P,ARCH4_Test] = archtest(Errors(:,4),'lags',4)
% [h,ARCH5_P,ARCH5_Test] = archtest(Errors(:,5),'lags',4)

Sigma_Eta_Sample = Sigma_Eta;

Omega_eta = 2*mDD*kron(Sigma_Eta_Sample,Sigma_Eta_Sample)*(mDD)';
StandardErrors_Omega=sqrt(diag(1/TAll*((mDD*kron(Sigma_Eta_Sample,Sigma_Eta_Sample)*(mDD)'))));
               
%% Partial Shocks Identification Strategy
Sigma_u = Sigma_Eta_Sample(1:3,1:3);
Sigma_uv = Sigma_Eta_Sample(1:3,end-1:end);
Sigma_vu = Sigma_Eta_Sample(end-1:end,1:3);
Sigma_v = Sigma_Eta_Sample(end-1:end,end-1:end);
                 
DiffZero = tril(Sigma_Eta_Sample);
VechSigmaU = DiffZero(DiffZero~=0);

P_matrix = zeros(M*(M+1)/2,M*(M+1)/2);
P_matrix(1,1) = 1;
P_matrix(2,2) = 1;
P_matrix(3,3) = 1;
P_matrix(4,6) = 1;
P_matrix(5,7) = 1;
P_matrix(6,10) = 1;
P_matrix(7,4) = 1;
P_matrix(8,5) = 1;
P_matrix(9,8) = 1;
P_matrix(10,9) = 1;
P_matrix(11,11) = 1;
P_matrix(12,12) = 1;
P_matrix(13,13) = 1;
P_matrix(14,14) = 1;
P_matrix(15,15) = 1;

lambda = P_matrix*VechSigmaU;
Omega_lambda = P_matrix*Omega_eta*P_matrix'; % eq. A.16 of the Supplementary Material

F_lambda = [-mDDg*kron(Sigma_vu*Sigma_u^(-1),Sigma_vu*Sigma_u^(-1))*(mDDn)' 2*mDDg*(kron(Sigma_vu*Sigma_u^(-1),eye(r))) zeros(0.5*r*(r+1),0.5*r*(r+1));
             zeros(n*r,0.5*n*(n+1))                                         eye(n*r)                                    zeros(n*r,0.5*r*(r+1))]; % pag. 18 of the Supplementary Material
         
Omega_zeta = F_lambda*Omega_lambda*F_lambda';

% Minimum distance
sig = [Sigma_vu*inv(Sigma_u)*Sigma_uv;
       Sigma_vu']; 
   
mS = Omega_zeta;

vg_initial = [.7 .0 .0 .0 .0]; % Starting point

[Param,Distance,exitflag,output,grad,Hessian_Matrix] = fminunc('MinimumDistance',vg_initial,options); % eq. 23 of the Paper

phi_est = Param(end-1:end)';
B_est = [Param(1:3)];

F_theta_est = [zeros(0.5*r*(r+1),n*g_shock)              2*mDDg*kron(phi_est,eye(r));
               kron(eye(n),phi_est)*KommutationMatrixNg  kron(B_est',eye(r))]; % eq. 24 of the Paper
         
VAR_est = inv(F_theta_est'*inv(mS)*F_theta_est)/TAll;
SE_est = diag(VAR_est).^0.5;

% Estimation under the null of Exogenous Financial and Marco Uncertainty

vg_initial_restricted = [Param(1) Param(4) Param(5)];

[Restricted_Param,Restricted_Distance,exitflag,output,grad,Restricted_Hessian_Matrix] = fminunc('MinimumDistance_Restricted',vg_initial_restricted,options);

Partial_ExogTest = 1-chi2cdf(Restricted_Distance*TAll-Distance*TAll,2);

%% Full Shocks Identification strategy

Sigma_Eta_Sample = Sigma_Eta;
StructuralParam = 12;

InitialValue_SVAR_MD_Initial=abs(randn(StructuralParam,1)/10);

[StructuralParam_Estiamtion_MATRIX,Likelihood_MATRIX_Sample,exitflag,output,grad,Hessian_MATRIX] = fminunc('Likelihood_SVAR',InitialValue_SVAR_MD_Initial',options); % eq. A.17 of the Supplementary Material

InitialValue_SVAR_MD_Initial_Sample = InitialValue_SVAR_MD_Initial;    
    
SE_Hessian_MATRIX = diag(inv(Hessian_MATRIX)).^0.5;

Estimation = [StructuralParam_Estiamtion_MATRIX' SE_Hessian_MATRIX];

% Control positiveness    
G=[StructuralParam_Estiamtion_MATRIX(1)  StructuralParam_Estiamtion_MATRIX(2) StructuralParam_Estiamtion_MATRIX(3) 0                                     0;
   StructuralParam_Estiamtion_MATRIX(4)  StructuralParam_Estiamtion_MATRIX(5) 0                                    0                                     0;
   StructuralParam_Estiamtion_MATRIX(6)  StructuralParam_Estiamtion_MATRIX(7) StructuralParam_Estiamtion_MATRIX(8) 0                                     0;
   StructuralParam_Estiamtion_MATRIX(9)  0                                    0                                    StructuralParam_Estiamtion_MATRIX(10) 0;
   StructuralParam_Estiamtion_MATRIX(11) 0                                    0                                    0                                     StructuralParam_Estiamtion_MATRIX(12)]; 

SE_G=[SE_Hessian_MATRIX(1)  SE_Hessian_MATRIX(2)  SE_Hessian_MATRIX(3) 0                     0;
      SE_Hessian_MATRIX(4)  SE_Hessian_MATRIX(5)  0                    0                     0;
      SE_Hessian_MATRIX(6)  SE_Hessian_MATRIX(7)  SE_Hessian_MATRIX(8) 0                     0;
      SE_Hessian_MATRIX(9)  0                     0                    SE_Hessian_MATRIX(10) 0;
      SE_Hessian_MATRIX(11) 0                     0                    0                     SE_Hessian_MATRIX(12)];
    
    if G(1,1)<0
    G(:,1)=-G(:,1);
    end
    if G(2,2)<0
    G(:,2)=-G(:,2); 
    end
    if G(3,3)<0
    G(:,3)=-G(:,3);
    end
    if G(4,4)<0
    G(:,4)=-G(:,4); 
    end
    if G(5,5)<0
    G(:,5)=-G(:,5);
    end 
    
% Estimation under the null of Exogenous Financial and Marco Uncertainty
StructuralParam = 10;

InitialValue_SVAR_MD_Initial=abs(randn(StructuralParam,1)/10);
[StructuralParam_Estiamtion_MATRIX_REST,Likelihood_MATRIX_REST_Sample,exitflag,output,grad,Hessian_MATRIX_REST] = fminunc('Likelihood_SVAR_Restricted',InitialValue_SVAR_MD_Initial',options);

SE_Hessian_MATRIX_REST = diag(inv(Hessian_MATRIX_REST)).^0.5;

Estimation_REST = [StructuralParam_Estiamtion_MATRIX_REST' SE_Hessian_MATRIX_REST];

% Control positiveness    
G_REST=[StructuralParam_Estiamtion_MATRIX_REST(1) StructuralParam_Estiamtion_MATRIX_REST(2) StructuralParam_Estiamtion_MATRIX_REST(3) 0                                         0;
        0                                         StructuralParam_Estiamtion_MATRIX_REST(4) 0                                         0                                         0;
        0                                         StructuralParam_Estiamtion_MATRIX_REST(5) StructuralParam_Estiamtion_MATRIX_REST(6) 0                                         0;
        StructuralParam_Estiamtion_MATRIX_REST(7) 0                                         0                                         StructuralParam_Estiamtion_MATRIX_REST(8) 0;
        StructuralParam_Estiamtion_MATRIX_REST(9) 0                                         0                                         0                                         StructuralParam_Estiamtion_MATRIX_REST(10)];
  
SE_G_REST=[SE_Hessian_MATRIX_REST(1) SE_Hessian_MATRIX_REST(2) SE_Hessian_MATRIX_REST(3) 0                          0;
           0                         SE_Hessian_MATRIX_REST(4) 0                         0                          0;
           0                         SE_Hessian_MATRIX_REST(5) SE_Hessian_MATRIX_REST(6) 0                          0;
           SE_Hessian_MATRIX_REST(7) 0                         0                         SE_Hessian_MATRIX_REST(8)  0;
           SE_Hessian_MATRIX_REST(9) 0                         0                         0                          SE_Hessian_MATRIX_REST(10)];
    
    if G_REST(1,1)<0
    G_REST(:,1)=-G_REST(:,1);
    end
    if G_REST(2,2)<0
    G_REST(:,2)=-G_REST(:,2); 
    end
    if G_REST(3,3)<0
    G_REST(:,3)=-G_REST(:,3);
    end
    if G_REST(4,4)<0
    G_REST(:,4)=-G_REST(:,4); 
    end
    if G_REST(5,5)<0
    G_REST(:,5)=-G_REST(:,5);
    end  
    
%% Bootstrap
DataSet_Sample = DataSet;
wbGS = waitbar(0,'Running the bootstrap');
    
    for boot = 1 : BootstrapIterations
        
    waitbar(boot/BootstrapIterations)  
%     boot
    DB_Boot=[];

    TBoot=datasample(1:TAll,TAll);
    Errors_Boot=Errors(TBoot,:);

    StoreErrors_Boot(:,:,boot)=Errors_Boot;

    % Dataset
    T=TAll;
%     DataSet=DataSet_1Regime;
    DataSet_Bootstrap=zeros(T+NLags,M);
    DataSet_Bootstrap(1:NLags,:)=DataSet_Sample(1:NLags,:);

        for t = 1+NLags : T+NLags
        DataSet_Bootstrap(t,:)=A_Const + A_1 * DataSet_Bootstrap(t-1,:)' +  A_2 * DataSet_Bootstrap(t-2,:)' +  A_3 * DataSet_Bootstrap(t-3,:)' +  A_4 * DataSet_Bootstrap(t-4,:)' +  Errors_Boot(t-NLags,:)';
        end

    DataSet=DataSet_Bootstrap;

    EstMdl_Boot = estimate(Mdl,DataSet);    
    A_Const_Boot(:,:,boot) = EstMdl_Boot.Constant;
    A_1_Boot(:,:,boot) = EstMdl_Boot.AR{1,1};
    A_2_Boot(:,:,boot) = EstMdl_Boot.AR{1,2};
    A_3_Boot(:,:,boot) = EstMdl_Boot.AR{1,3};
    A_4_Boot(:,:,boot) = EstMdl_Boot.AR{1,4};
    Sigma_Eta = EstMdl_Boot.Covariance;
    Sigma_Eta_Boot(:,:,boot)=Sigma_Eta;

    %% Full Shocks Identification Strategy   

    InitialValue_SVAR_MD_Initial=StructuralParam_Estiamtion_MATRIX;
    [StructuralParam_Estiamtion_MATRIX,Likelihood_MATRIX,exitflag,output,grad,Hessian_MATRIX] = fminunc('Likelihood_SVAR',InitialValue_SVAR_MD_Initial',options);


    G_Boot(:,:,boot)=[StructuralParam_Estiamtion_MATRIX(1)  StructuralParam_Estiamtion_MATRIX(2) StructuralParam_Estiamtion_MATRIX(3) 0                                     0;
                      StructuralParam_Estiamtion_MATRIX(4)  StructuralParam_Estiamtion_MATRIX(5) 0                                    0                                     0;
                      StructuralParam_Estiamtion_MATRIX(6)  StructuralParam_Estiamtion_MATRIX(7) StructuralParam_Estiamtion_MATRIX(8) 0                                     0;
                      StructuralParam_Estiamtion_MATRIX(9)  0                                    0                                    StructuralParam_Estiamtion_MATRIX(10) 0;
                      StructuralParam_Estiamtion_MATRIX(11) 0                                    0                                    0                                     StructuralParam_Estiamtion_MATRIX(12)];
   
        if G_Boot(1,1,boot)<0
        G_Boot(:,1,boot)=-G_Boot(:,1,boot);
        end
        if G_Boot(2,2,boot)<0
        G_Boot(:,2,boot)=-G_Boot(:,2,boot); 
        end
        if G_Boot(3,3,boot)<0
        G_Boot(:,3,boot)=-G_Boot(:,3,boot);
        end
        if G_Boot(4,4,boot)<0
        G_Boot(:,4,boot)=-G_Boot(:,4,boot); 
        end
        if G_Boot(5,5,boot)<0
        G_Boot(:,5,boot)=-G_Boot(:,5,boot);
        end 
    
    % Restricted
    StructuralParam = 10;
    InitialValue_SVAR_MD_Initial=StructuralParam_Estiamtion_MATRIX_REST;
    [StructuralParam_Estiamtion_MATRIX_REST,Likelihood_MATRIX_REST,exitflag,output,grad,Hessian_MATRIX_REST] = fminunc('Likelihood_SVAR_Restricted',InitialValue_SVAR_MD_Initial',options);


    % Control positiveness    
    G_REST_Boot(:,:,boot)=[StructuralParam_Estiamtion_MATRIX_REST(1) StructuralParam_Estiamtion_MATRIX_REST(2) StructuralParam_Estiamtion_MATRIX_REST(3) 0                                         0;
                           0                                         StructuralParam_Estiamtion_MATRIX_REST(4) 0                                         0                                         0;
                           0                                         StructuralParam_Estiamtion_MATRIX_REST(5) StructuralParam_Estiamtion_MATRIX_REST(6) 0                                         0;
                           StructuralParam_Estiamtion_MATRIX_REST(7) 0                                         0                                         StructuralParam_Estiamtion_MATRIX_REST(8) 0;
                           StructuralParam_Estiamtion_MATRIX_REST(9) 0                                         0                                         0                                         StructuralParam_Estiamtion_MATRIX_REST(10)];


        if G_REST_Boot(1,1,boot)<0
        G_REST_Boot(:,1,boot)=-G_REST_Boot(:,1,boot);
        end
        if G_REST_Boot(2,2,boot)<0
        G_REST_Boot(:,2,boot)=-G_REST_Boot(:,2,boot); 
        end
        if G_REST_Boot(3,3,boot)<0
        G_REST_Boot(:,3,boot)=-G_REST_Boot(:,3,boot);
        end
        if G_REST_Boot(4,4,boot)<0
        G_REST_Boot(:,4,boot)=-G_REST_Boot(:,4,boot); 
        end
        if G_REST_Boot(5,5,boot)<0
        G_REST_Boot(:,5,boot)=-G_REST_Boot(:,5,boot);
        end     
    
    
    end    
    
%% IRF (under the null of exogeneity)    
HorizonIRF = 60;
J=[eye(M) zeros(M,M*(NLags-1))]; 

CompanionMatrix = [A_1 A_2 A_3 A_4;
                   eye(M*(NLags-1)) zeros(M*(NLags-1),M)];                      
    for h = 0 : HorizonIRF
    IRF_Sample(:,:,h+1)=J*CompanionMatrix^h*J'*G_REST;
    end
 
IRF_Boot=[];
    for i = 1 : BootstrapIterations 
    CompMatrix_Boot=   [A_1_Boot(:,:,i)...
                        A_2_Boot(:,:,i)...
                        A_3_Boot(:,:,i)...
                        A_4_Boot(:,:,i)];
                               
    CompanionMatrix_Boot= [CompMatrix_Boot;
                           eye(M*(NLags-1)) zeros(M*(NLags-1),M)];        
        for h = 0 : HorizonIRF
        IRF_Boot(:,:,h+1,i)=J*CompanionMatrix_Boot^h*J'*G_REST_Boot(:,:,i);
        end
    end
  
% Graphs settings
LineWidth_IRF=2;
FontSizeIRFGraph=12;
alpha_coverage = 10;

%% Figure 1  
Titles=cell(1,3);
Titles{1,1}='$$\varepsilon_a$$ $$Shock$$';
Titles{1,2}='$$\varepsilon_F$$ $$Shock$$';
Titles{1,3}='$$\varepsilon_M$$ $$Shock$$';

YLabel=cell(3,1);
YLabel{1,1}='$$a$$';
YLabel{2,1}='$$U_F$$';
YLabel{3,1}='$$U_M$$';

index =1;
figure(1)
Shock_1=[1 100 100];

for jr = 1 : 3
    for jc = 1 : 3

    TETA_Iter_Sample = squeeze(IRF_Sample(jr,jc,:));
    TETA_Iter_Boot = squeeze(IRF_Boot(jr,jc,:,:));
    SE_TetaIter = (mean(TETA_Iter_Boot'.^2)' - mean(TETA_Iter_Boot')'.^2).^0.5;

    for j = 1 : BootstrapIterations % Algorithm 3 Montiel Olea - Plagborg Moller    
    abs(TETA_Iter_Boot(:,j)-TETA_Iter_Sample)./SE_TetaIter;
    m_max(j,:) = max(abs(TETA_Iter_Boot(:,j)-TETA_Iter_Sample)./SE_TetaIter);
    end

    IRF_Iter_LB = TETA_Iter_Sample - SE_TetaIter.*prctile(m_max,100-alpha_coverage);
    IRF_Iter_UB = TETA_Iter_Sample + SE_TetaIter.*prctile(m_max,100-alpha_coverage);
    
    subplot(3,3,index)  
    x = 1:1:HorizonIRF+1;
    y= TETA_Iter_Sample'*Shock_1(jr);
    yu = IRF_Iter_LB'*Shock_1(jr);
    yl = IRF_Iter_UB'*Shock_1(jr);
    trasparent=fill([x fliplr(x)], [yu fliplr(yl)],[0 0.4470 0.7410], 'linestyle', 'none');
    set(trasparent,'facealpha',0.3);
    hold all
    p1= plot(y,'Color',[0 0.4470 0.7410], 'LineWidth',LineWidth_IRF);
    p8=plot(zeros(HorizonIRF+1,1),'k','LineWidth',1);
    ylabel(YLabel{jr,1},'interpreter','latex');
    title(Titles{1,jc},'interpreter','latex');
    set(gca,'FontSize',FontSizeIRFGraph);
    axis tight
    index=index+1;
    
    end
end

%% Correlation Structural Shocks
StructuralShocks = (G_REST^(-1)*Errors')';
CorrMatrix =corr([StructuralShocks(:,1:3) Errors(:,4:5)]);
PValCorr = 2*(1-normcdf(abs(CorrMatrix),0,1/TAll^0.5));

%% Print Results
clc

disp('************************************************************')
disp('************************************************************')
disp('Table 1')
disp('************************************************************')
disp('Partial Shocks Identification')
disp('G:')
disp(Param')
disp('se(G):')
disp(SE_est)
disp('TQ exog:')
disp(Restricted_Distance*TAll-Distance*TAll)
disp('p-val exog:')
disp(1-chi2cdf(Restricted_Distance*TAll-Distance*TAll,2))
disp('Correlations:')
disp([
Param(4)/Sigma_Eta_Sample(4,4)^0.5;
Param(5)/Sigma_Eta_Sample(5,5)^0.5
])
disp('p-val correlations:')
disp(2*(1-normcdf(abs([
Param(4)/Sigma_Eta_Sample(4,4)^0.5;
Param(5)/Sigma_Eta_Sample(5,5)^0.5
]),0,1/TAll^0.5)))
disp('************************************************************')
disp('Full Shocks Identification')
disp('G:')
disp(G)
disp('se(G):')
disp(SE_G)
disp('LR overall:')
disp(2*(logL_Rest +Likelihood_MATRIX_Sample))
disp('p-val overall:')
disp(1-chi2cdf(2*(logL_Rest +Likelihood_MATRIX_Sample),3))
disp('LR exog:')
disp(2*(-Likelihood_MATRIX_Sample + Likelihood_MATRIX_REST_Sample))
disp('p-val exog:')
disp(1-chi2cdf(2*(-Likelihood_MATRIX_Sample + Likelihood_MATRIX_REST_Sample),2))
disp('Correlations:')
disp(CorrMatrix([4 5],[1 2 3]))
disp('p-val correlations:')
disp(PValCorr([4 5],[1 2 3]))
disp('************************************************************')
disp('************************************************************')

%% Close waitbar
close(wbGS)

