library(ggplot2)
library(magrittr)
library(tidyr)
library(dplyr)
library(ggfan)
library(matrixStats)
library(latex2exp)
library(here)

fpath = "results_theory/20210529"
figpath = "paper/figs"

choice_network = 2
N = 10000
intervals = seq(0.1, 0.8, 0.05)
quants <- c(0.1, 0.25, 0.5, 0.75, 0.9)
qnames <- c("10th","25th","50th","75th","90th")
ymax = 3.5
lowcolor = "orangered"
highcolor = "yellow"
mediancolor = "red3"

if (choice_network==1) {
  network_string = 'ER'
} else if (choice_network==2){
  network_string = 'PL'
}
fname = paste0(network_string,"_N",format(N,scientific=1),"_dcT.csv")

dcT_data <- read.csv(here(fpath, fname), header=FALSE)

dcT_sim <- as.matrix(dcT_data)
T <- dim(dcT_sim)[2]
days <- 2:(T+1)
x_axis_breaks = c(seq(0,T,20), T+1)

dcT_sim_long <- data.frame(x=days, t(dcT_sim)) %>% gather(key=sim, value=y, -x)
dcT_sim_q <- colQuantiles(dcT_sim, probs=quants)
dcT_sim_q_df <- as.data.frame(dcT_sim_q)
names(dcT_sim_q_df) <- qnames
dcT_sim_q_df <- cbind(dcT_sim_q_df, days)
dcT_sim_q_df_long <- gather(dcT_sim_q_df, key=percentile, value=value, -days)

print(ggplot(dcT_sim_long, aes(x=x,y=y)) +
  geom_fan(stat = "interval", intervals=intervals) +
  geom_line(data=dcT_sim_q_df_long, mapping=aes(x=days,y=value, group=percentile, linetype=percentile,
                                               color=percentile, size=percentile)) +
  scale_linetype_manual(values=c(3,2,1,4,5)) +
  scale_color_manual(values=c("grey30","grey30",mediancolor,"grey30","grey30")) +
  scale_size_manual(values = c(0.5,0.5,1,0.5,0.5)) +

  scale_fill_gradient(low=lowcolor, high=highcolor) +
  guides(fill="none") +

  xlab("Days since the outbreak") +
  ylab(expression(paste("Proportion new cases ( ", Delta, c[t], ", per cent)"))) +
  scale_x_continuous(expand = c(0, 0), breaks=x_axis_breaks, limits=c(0,200)) +
  scale_y_continuous(expand = c(0, 0), breaks =seq(0, ymax, 0.5),
                     labels=format(seq(0, ymax, 0.5), drop0trailing = TRUE), limits=c(0,ymax)) +
  theme_bw() +
  theme(panel.border = element_blank(), panel.grid.major = element_blank(),
        panel.grid.minor = element_blank(), axis.line = element_line(),
        legend.position="bottom",
        aspect.ratio=0.8,
        axis.text=element_text(colour="black"),
        axis.title.y=element_text(vjust=2.5), axis.title.x=element_text(vjust=-1.5),
        axis.title=element_text(size=12),
        legend.title=element_text(size=12),
        legend.text=element_text(size=12)))

figname = paste0("theory_",network_string,"_N",format(N,scientific=1),"_dcT.png")
ggsave(here(figpath, figname), dpi=300, width = 5, height = 4, unit="in")



