%--------------------------------------------------------------------------
%  ------------------------ Replication code for ------------------------  
%      --- "Trade Openness and Growth: A Network-Based Approach" --- 
%  -------- Georg Duernecker, Moritz Meyer, Fernando Vega-Redondo --------
%  ------------------ Journal of Applied Econometrics ------------------ 
%--------------------------------------------------------------------------
%                       This version: May 2022
%                      Written for: Matlab R2020a
%--------------------------------------------------------------------------
% Part 1: Load and prepare data for estimation
% Part 2: Bayesian model averaging and MC^3 
% Part 3: Diagnostic statistics
% Part 4: Report output
% Part 5: Functions used in estimation
%--------------------------------------------------------------------------
% Parts 1 and 2 are based on the Gauss code used in Enrique Moral-Benito,
% "Likelihood-based Estimation of Dynamic Panels with Predetermined 
% Regressors", Journal of Business and Economic Statistics, 31, 451-472.
%--------------------------------------------------------------------------
clear
%--------------------------------------------------------------------------

%--------------------------------------------------------------------------
% Part 1: Load and rearrange the data
%--------------------------------------------------------------------------
variables   = char([{'  '},...                              % Variables in the data set
    {'Lagged GDP per capita'},...                           % Lagged log of real GDP per capita
    {'Total population'},...                                % x-variables ....
    {'Population growth'},...
    {'Labor force participation rate'},...
    {'Investment price'},...
    {'Consumption share'},...
    {'Investment share'},...
    {'Government share'},...
    {'Globalization Index'},...
    {'Trade share'},...
    {'Sachs-Warner Index'},...
    {'Population density'},...
    {'Urban population'},...
    {'Life expectancy'},...
    {'Population share aged 0-14 years'},...
    {'Population share aged +65 years'},...
    {'Armed conflict'},...
    {'Democracy index'},...
    {'Average years of primary schooling'},...
    {'Average years of secondary schooling'},...            % ------------------
    {'Land area in squ km'},...                             % z - variables ....
    {'Land share in the geographic tropics'},...
    {'Population share in the geographic tropics    '},...
    {'Land area within 100km of navigable water'},...
    {'Landlocked country'},...
    {'Land share in Koeppen-Geiger tropics'},...
    {'Air distance to NYC, Rotterdam, Tokyo'},...
    {'Timing of national independence'},...
    {'Socialist rule in 1950-95'},...
    {'Former Spanish colony'},...
    {'Western European country'},...
    {'Sub-Saharan country'},...
    {'Latin-American country'},...
    {'East Asian country'}]);
%--------------------------------------------------------------------------

%--------------------------------------------------------------------------
y_var =     1:2;    % Indices of (lagged) dependent variable in dataset
x_var =     3:21;   % Indices of time-varying covariates
z_var =     22:35;  % Indices of constant covariates
%--------------------------------------------------------------------------

%--------------------------------------------------------------------------
rawdata = csvread('Data_BMA.csv',1,2);      % Load the data set
%--------------------------------------------------------------------------
t       = 4;        % Number of observations per country
n       = 82;       % Number of countries
%--------------------------------------------------------------------------
nobs    = n*t;                  % Total number of observations
ktotx   = numel(x_var);         % Number of x-variables
ktotz   = numel(z_var);         % Number of z-variables
ktoty   = ktotx + ktotz + 1;    % Number of covariates
%--------------------------------------------------------------------------
R       = transfbench(rawdata,n,t,ktoty);   % De-mean, rescale, rearrange data
%--------------------------------------------------------------------------


%--------------------------------------------------------------------------
Y1 = R(:,2+ktoty*(0:t-1));      % dependent variable
%--------------------------------------------------------------------------
X0 = R(:,3:2+ktotx);
Z0 = R(:,2+ktotx+1:2+ktotx+ktotz);
%--------------------------------------------------------------------------


%--------------------------------------------------------------------------
% Part 2: Bayesian model averaging and MC^3 
%--------------------------------------------------------------------------
pmsize  = 5;                                    % Prior expected model size
b       = ( (ktotx+ktotz) - pmsize)/pmsize;
%--------------------------------------------------------------------------
% Initialize arrays
%--------------------------------------------------------------------------
mod     = zeros(ktoty,1);
bet     = zeros(ktoty,1);
stdbet  = zeros(ktoty,1);
sig     = zeros(ktoty,1);
ppmsize = 0;
%--------------------------------------------------------------------------

%--------------------------------------------------------------------------
% Initial model 
%--------------------------------------------------------------------------
mt = [0; 0; 0; 0; 0; 0; 0; 1; 0; 0; 1; 0; 0; 0; 0; 0; 0; 1; 0; 0; 0; 0; 0; 0; 0; 0; 0; 0; 0; 0; 1; 0; 0];
%--------------------------------------------------------------------------

%--------------------------------------------------------------------------
mtx = mt(1:ktotx);
mtz = mt(ktotx+1:end);
%--------------------------------------------------------------------------
outx = mtx==0;
outz = mtz==0;
%--------------------------------------------------------------------------

%--------------------------------------------------------------------------
kx = sum(mtx);
kz = sum(mtz);
ky = kx + kz + 1;
%--------------------------------------------------------------------------

%--------------------------------------------------------------------------
% Z, Y2 and Q variables are model specific variables
%--------------------------------------------------------------------------
aux = [];
%--------------------------------------------------------------------------
for tt = 1:t-1
    aux = [ aux 1+ktoty*tt+1+find(mtx==1)'];
end
%--------------------------------------------------------------------------
Y2              = R(:,aux);
%--------------------------------------------------------------------------
X0j             = X0';
X0j(outx==1,:)  = [];
%--------------------------------------------------------------------------
Z0j             = Z0';
Z0j(outz==1,:)  = [];
%--------------------------------------------------------------------------

%--------------------------------------------------------------------------
Z               = [R(:,1) X0j' Z0j'];
Q               = eye(n)-Z*inv(Z'*Z)*Z';
%--------------------------------------------------------------------------

%--------------------------------------------------------------------------
% Starting values
t0ii    = ones(3+2*kx+kz+t+(t-1)*kx+t*(t-1)*kx/2,1);
%--------------------------------------------------------------------------
warning off all
options = optimset('Display','off', 'MaxFunEvals', 5000000, 'MaxIter', 5000000, 'TolX', 1e-16, 'TolFun', 1e-16,'LargeScale', 'off', 'HessUpdate', 'bfgs');
%--------------------------------------------------------------------------
% Maximize likelihood function (minimize the negative)
[theta,fout,exitflag,output,gradout,hessianout]   = fminunc(@(x) likbench(x,t,kx,kz,n,Y1,Y2,Z,Q), t0ii, options);
%--------------------------------------------------------------------------

%--------------------------------------------------------------------------
% Collect estimation output
%--------------------------------------------------------------------------
bt          = theta(1:1+kx+kz);
stdh        = sqrt(diag((inv(hessianout))));
bstdt       = stdh(1:1+kx+kz);
%--------------------------------------------------------------------------

%--------------------------------------------------------------------------
marg_lik    = convertbench(theta,t,kx,kz,n,Y1,Y2,Z,Q); % Marginal likelihood
log_post_m  = marg_lik - ((1+kx+kz)/2)*log(nobs);
log_prior_m = log( ( gamma( 1 + ky ) )*( gamma(b+ktoty-ky) ) );
%--------------------------------------------------------------------------

%--------------------------------------------------------------------------
% Store the output
%--------------------------------------------------------------------------
mty     = [1; mt];
bt0     = zeros(ktoty,1);   bt0(mty==1)     = bt;
bstdt0  = zeros(ktoty,1);   bstdt0(mty==1)  = real(bstdt);
%--------------------------------------------------------------------------

%--------------------------------------------------------------------------
% "pool_..." is the pool of unique models visited by the Markov chain
%--------------------------------------------------------------------------
pool_prior = log_prior_m;    % Model prior
pool_post  = log_post_m;     % Model posterior
pool_par   = bt0;            % Coefficient estimates of current model
pool_std   = bstdt0;         % Standard errors of coefficient estimates
pool_m     = mty;            % Current model
%--------------------------------------------------------------------------

%--------------------------------------------------------------------------
% "m_model_..." is the sequence of models in the Markov chain
%--------------------------------------------------------------------------
m_model_prior = log_prior_m;
m_model_post  = log_post_m;
m_model_m     = mty;
%--------------------------------------------------------------------------

%--------------------------------------------------------------------------
tic             % Start clock
%--------------------------------------------------------------------------
Smax  = 1e+6;   % Maximum number of Markov steps 
steps = 0;      % Number of steps of the Markov chain
loop  = 0;      % Counts how many time intermediate output was saved
%--------------------------------------------------------------------------
while  steps < Smax
%--------------------------------------------------------------------------
    steps   = steps + 1;
    m_repl  = 0;            % 1: Current model has been visited before; it 
                            % is stored in "pool_..."
%--------------------------------------------------------------------------
% Characteristics of the current model
%--------------------------------------------------------------------------
    mcu             = mt;
    betacu          = bt;
    bscu            = bstdt;
    log_post_cu     = log_post_m;
    log_prior_cu    = log_prior_m;
%--------------------------------------------------------------------------
    
%--------------------------------------------------------------------------
    found_model = 0;
%--------------------------------------------------------------------------
    while found_model == 0
%--------------------------------------------------------------------------
        % Select the candidate model
%--------------------------------------------------------------------------
        s1  = rand*(ktotx+ktotz);
        s   = floor(s1) + 1;
%--------------------------------------------------------------------------
        mt(s)   = abs( mt(s) - 1 );
        m2      = mt;
%--------------------------------------------------------------------------
        mtx2 = m2(1:ktotx);
        mtz2 = m2(ktotx+1:end);
%--------------------------------------------------------------------------
        kx = sum(mtx2);
        kz = sum(mtz2);
        ky = kx + kz + 1;
%--------------------------------------------------------------------------
        outx = mtx2 == 0;
        outz = mtz2 == 0;
%--------------------------------------------------------------------------

%--------------------------------------------------------------------------
        % Z, Y2 and Q variables are model specific variables
%--------------------------------------------------------------------------
        aux = [];
        for tt = 1:t-1
            aux = [ aux 1+ktoty*tt+1+find(mtx2==1)'];
        end
        Y2              = R(:,aux);
%--------------------------------------------------------------------------
        X0j             = X0';
        X0j(outx==1,:)  = [];
%--------------------------------------------------------------------------
        Z0j             = Z0';
        Z0j(outz==1,:)  = [];
%--------------------------------------------------------------------------

%--------------------------------------------------------------------------
        Z       = [R(:,1) X0j' Z0j'];
        Q       = eye(n)-Z*inv(Z'*Z)*Z';
%--------------------------------------------------------------------------
        
%--------------------------------------------------------------------------
        t0ii    = ones(3+2*kx+kz+t+(t-1)*kx+t*(t-1)*kx/2,1);
%--------------------------------------------------------------------------

%--------------------------------------------------------------------------
        aux0 = size(pool_m,2);
        aux1 = sum( abs(pool_m(2:end,:)-m2*ones(1,aux0)) );
%--------------------------------------------------------------------------
        
%--------------------------------------------------------------------------
        if min(aux1)==0     % Check whether model has been visited before
%--------------------------------------------------------------------------
            ind_min = find(aux1==0, 1, 'first');
%--------------------------------------------------------------------------
            beta2       = pool_par(pool_m(:,ind_min)==1,ind_min);
            bs2         = pool_std(pool_m(:,ind_min)==1,ind_min);
            log_post_2  = pool_post(ind_min);
            log_prior_2 = pool_prior(ind_min);
            found_model = 1;
            m_repl      = 1;
%--------------------------------------------------------------------------
        else       
%--------------------------------------------------------------------------
% Estimation of the candidate model
%--------------------------------------------------------------------------
            aux_mty     = [1; m2]';
            aux_mcu     = [1; mcu]';
%--------------------------------------------------------------------------
% Case 1: both x and z's in the model
%--------------------------------------------------------------------------
            if ( kx>0 )&&( kz>0 )
%--------------------------------------------------------------------------
                found_model = 1;
%--------------------------------------------------------------------------
                [theta,fout,exitflag,output,gradout,hessianout]   = fminunc(@(x) likbench(x,t,kx,kz,n,Y1,Y2,Z,Q), t0ii, options);
%--------------------------------------------------------------------------
                beta2       = theta(1:1+kx+kz);
                stdh        = sqrt(diag((inv(hessianout))));
                bs2         = stdh(1:1+kx+kz);
                marg_lik    = convertbench(theta,t,kx,kz,n,Y1,Y2,Z,Q);
                log_post_2  = marg_lik - ((1+kx+kz)/2)*log(nobs);
                log_prior_2 = log( ( gamma( 1 + ky ) )*( gamma(b+ktoty-ky) ) );
%--------------------------------------------------------------------------
            
%--------------------------------------------------------------------------
% Case 2: only x's in the model
%--------------------------------------------------------------------------
            elseif ( kx>0 )&&( kz==0 ) 
%--------------------------------------------------------------------------
                found_model = 1;
%--------------------------------------------------------------------------
                [theta,fout,exitflag,output,gradout,hessianout]   = fminunc(@(x) likxbench(x,t,kx,n,Y1,Y2,Z,Q), t0ii, options);
%--------------------------------------------------------------------------
                beta2       = theta(1:1+kx+kz);
                stdh        = sqrt(diag((inv(hessianout))));
                bs2         = stdh(1:1+kx+kz);
                marg_lik    = convertxbench(theta,t,kx,n,Y1,Y2,Z,Q);
                log_post_2  = marg_lik - ((1+kx+kz)/2)*log(nobs);
                log_prior_2 = log( ( gamma( 1 + ky ) )*( gamma(b+ktoty-ky) ) );
%--------------------------------------------------------------------------
            
%--------------------------------------------------------------------------
% Case 3: only z's in the model    
%--------------------------------------------------------------------------
            elseif ( kx==0 )&&( kz>0 )
%--------------------------------------------------------------------------
                found_model = 1;
%--------------------------------------------------------------------------
                [theta,fout,exitflag,output,gradout,hessianout]   = fminunc(@(x) likzbench(x,t,kx,kz,n,Y1,Y2,Z,Q), t0ii, options);
%--------------------------------------------------------------------------
                beta2       = theta(1:1+kx+kz);
                stdh        = sqrt(diag((inv(hessianout))));
                bs2         = stdh(1:1+kx+kz);
                marg_lik    = convertzbench(theta,t,kx,kz,n,Y1,Y2,Z,Q);
                log_post_2  = marg_lik - ((1+kx+kz)/2)*log(nobs);
                log_prior_2 = log( ( gamma( 1 + ky ) )*( gamma(b+ktoty-ky) ) );
%--------------------------------------------------------------------------
            
%--------------------------------------------------------------------------
% Case 4: no x' and z's in the model => make new draw
%--------------------------------------------------------------------------
            elseif ( kx==0 )&&( kz==0 )
%--------------------------------------------------------------------------
                found_model = 0;
%--------------------------------------------------------------------------
            end
%--------------------------------------------------------------------------
        end
%--------------------------------------------------------------------------
    end
%--------------------------------------------------------------------------

%--------------------------------------------------------------------------
    mty     = [1; m2];
    bt0     = zeros(ktoty,1);   bt0(mty==1)     = beta2;
    bstdt0  = zeros(ktoty,1);   bstdt0(mty==1)  = real(bs2);
%--------------------------------------------------------------------------

%--------------------------------------------------------------------------
    if m_repl == 0  % Add model to pool if it is not there already
%--------------------------------------------------------------------------
        pool_m     = [ pool_m     mty ];
        pool_par   = [ pool_par   bt0 ];
        pool_std   = [ pool_std   bstdt0 ];
        pool_prior = [ pool_prior log_prior_2 ];
        pool_post  = [ pool_post  log_post_2 ];
%--------------------------------------------------------------------------
    end
%--------------------------------------------------------------------------
    

%--------------------------------------------------------------------------
% Select the best model (in terms of BIC)
%--------------------------------------------------------------------------
    logbayesf   = log_prior_2 + log_post_2 - (log_prior_cu + log_post_cu);
    s2          = rand;
%--------------------------------------------------------------------------
    if logbayesf  > log(s2)     % Candidate model is accepted
%--------------------------------------------------------------------------
        mt          = m2;
        mtx         = mtx2;
        bt          = beta2;
        bstdt       = bs2;
        log_post_m  = log_post_2;
        log_prior_m = log_prior_2;
%--------------------------------------------------------------------------
    else
%--------------------------------------------------------------------------
        mt          = mcu;
        bt          = betacu;
        bstdt       = bscu;
        log_post_m  = log_post_cu;
        log_prior_m = log_prior_cu;
%--------------------------------------------------------------------------
    end
%--------------------------------------------------------------------------


%--------------------------------------------------------------------------
% Store the results
%--------------------------------------------------------------------------
    mty     = [1; mt];
    bt1     = zeros(ktoty,1);   bt1(mty==1)     = bt;
    bstdt1  = zeros(ktoty,1);   bstdt1(mty==1)  = real(bstdt);
    sig1    = zeros(ktoty,1);   sig1((mty==1)&(abs( bt1./bstdt1 )>1.96))  = 1;
%--------------------------------------------------------------------------
    mod     = mod + mty;
    bet     = bet + bt1;
    stdbet  = stdbet + (bstdt1 + (bt1.*bt1) );
    sig     = sig + sig1;
    pmsizet = sum(mty);
    ppmsize = ppmsize + pmsizet;
%--------------------------------------------------------------------------

%--------------------------------------------------------------------------
    if numel(m_model_prior) == 1e+4
%--------------------------------------------------------------------------
        loop = loop + 1;
%--------------------------------------------------------------------------
        save(['MarkovChain_',num2str(loop),'.mat'],'m_model_prior','m_model_post','m_model_m')
%--------------------------------------------------------------------------
        m_model_prior = [ ];
        m_model_post  = [ ];
        m_model_m     = [ ];
%--------------------------------------------------------------------------
        disp(['Number of Markov steps: ',   num2str(steps), ...
              ', % of total: ',             num2str(100*steps/Smax,'%10.2f'), ...
              ', Total time (hrs): ',       num2str(toc/3600,'%10.2f')]);
%--------------------------------------------------------------------------
    end
%--------------------------------------------------------------------------

%--------------------------------------------------------------------------
    m_model_prior = [ m_model_prior log_prior_m ];
    m_model_post  = [ m_model_post  log_post_m ];
    m_model_m     = [ m_model_m     mty ];
%--------------------------------------------------------------------------
end     % End MC^3 loop
%--------------------------------------------------------------------------


%--------------------------------------------------------------------------
% Part 3: Diagnostic statistics
%--------------------------------------------------------------------------
chain_model_prior = [];
chain_model_post  = [];
chain_model_m     = [];
%--------------------------------------------------------------------------
for i = 1:loop
%--------------------------------------------------------------------------
    file = ['MarkovChain_',num2str(i),'.mat'];
%--------------------------------------------------------------------------
    load(file)
%--------------------------------------------------------------------------
    chain_model_prior = [ chain_model_prior m_model_prior];
    chain_model_post  = [ chain_model_post  m_model_post];
    chain_model_m     = [ chain_model_m     m_model_m];
%--------------------------------------------------------------------------
    delete(file)
%--------------------------------------------------------------------------
end
%--------------------------------------------------------------------------
RunDiagnostics = input('Run diagnostics? Yes = 1 ');
%--------------------------------------------------------------------------
if RunDiagnostics
%--------------------------------------------------------------------------
    [CorrPiFreq,CorrBayesFreq,RafteryLewis] = diagnostics(steps,chain_model_m,chain_model_post,chain_model_prior);
%--------------------------------------------------------------------------
else
%--------------------------------------------------------------------------
    CorrPiFreq      = [];
    CorrBayesFreq   = [];
    RafteryLewis    = [];
%--------------------------------------------------------------------------
end
%--------------------------------------------------------------------------



%--------------------------------------------------------------------------
% Part 4: Report output
%--------------------------------------------------------------------------
postprob    = mod/steps;                         % Posterior inclusion probability
%--------------------------------------------------------------------------
postmeanc   = real(bet./mod);                   % Posterior mean of coefficient estimate
poststdc    = real(stdbet./mod - postmeanc.^2); % Standard deviation
tstatc      = real(postmeanc./poststdc);        % Pseudo t-statistics
sigc        = sig./mod;                         % Fraction of significant estimates
%--------------------------------------------------------------------------
postpmsize  = ppmsize/steps;                     % Posterior model size
Pbest       = exp(max(pool_post + pool_prior))/sum(exp(pool_post + pool_prior));
% ----------------------------------------------------------------------


%--------------------------------------------------------------------------
mean_ = char([{'Mean     '}, num2str(postmeanc,'%6.3f')]);
std_  = char([{'Stddev   '}, num2str(poststdc, '%6.2f')]);
prob_ = char([{'PIP      '}, num2str(postprob, '%6.2f')]);
sig_  = char([{'%sig     '}, num2str(sigc,     '%6.2f')]);
%--------------------------------------------------------------------------

%--------------------------------------------------------------------------
disp([variables mean_ std_ prob_ sig_])       % Display results
%--------------------------------------------------------------------------
if RunDiagnostics
%--------------------------------------------------------------------------
    disp('-----------')
    disp(['Markov steps (x 1000) | ', num2str(steps)])
    disp(['Posterior model size  | ', num2str(postpmsize,'%10.2f')])
    disp(['Pr(best model)        | ', num2str(Pbest,'%10.2f')])
    disp(['Corr(Pi,Freq)         | ', num2str(CorrPiFreq,'%10.2f')])
    disp(['Corr(Bayes,Freq)      | ', num2str(CorrBayesFreq,'%10.2f')])
    disp(['Raftery-Lewis factor  | ', num2str(RafteryLewis)])
%--------------------------------------------------------------------------
end
%--------------------------------------------------------------------------


%--------------------------------------------------------------------------
% Part 5: Functions used in estimation
%--------------------------------------------------------------------------

%--------------------------------------------------------------------------
function R = transfbench(rawdata,n,t,ktoty)
%--------------------------------------------------------------------------
dmtx        = eye(n*t) - kron( ones(n,n), (1/n)*eye(t) );
csddata     = dmtx*rawdata;
%--------------------------------------------------------------------------
csddata(:,3) = csddata(:,3)/1000000;        % Total population
csddata(:,6) = csddata(:,6)/100;            % Investment price
csddata(:,13) = csddata(:,13)/1000;         % Population density
csddata(:,15) = csddata(:,15)/100;          % Life expectancy
csddata(:,20) = csddata(:,20)/10;           % Average years of primary schooling
csddata(:,21) = csddata(:,21)/10;           % Average years of secondary schooling
csddata(:,22) = csddata(:,22)/10000000;     % Land area in squ km
csddata(:,25) = csddata(:,25)/100;          % Land area within 100km of navigable water
csddata(:,28) = csddata(:,28)/10000;        % Air distance to NYC, Rotterdam, Tokyo
%--------------------------------------------------------------------------

%--------------------------------------------------------------------------
R = zeros(n,t*ktoty + 1);
%--------------------------------------------------------------------------
for j = 1:n
%--------------------------------------------------------------------------
    aux = csddata((j-1)*t+1,2);
%--------------------------------------------------------------------------
    for tt = 1:t
        aux = [aux csddata((j-1)*t+tt,[1 3:size(csddata,2)]) ];
    end
%--------------------------------------------------------------------------
    R(j,:) = aux;
%--------------------------------------------------------------------------
end
%--------------------------------------------------------------------------
end
%--------------------------------------------------------------------------




% ----------------------------------------------------------------------
function M = convertbench(theta,t,kx,kz,n,Y1,Y2,Z,Q)
% ----------------------------------------------------------------------
alpha_  = theta(1);
beta_   = theta(1+1:1+kx)';
delta_  = theta(1+kx+1:1+kx+kz)';
gamma0_ = theta(1+kx+kz+1:1+kx+kz+1);
gamma1_ = theta(1+kx+kz+1+1:1+kx+kz+1+kx)';
% ----------------------------------------------------------------------
t0 = theta;
% ----------------------------------------------------------------------
B11 = diag(-ones(t-1,1)*t0(1),-1) + eye(t);
B12 = zeros(t,(t-1)*kx);
% ----------------------------------------------------------------------
for tt = 2:t
    B12(tt,(tt-2)*kx+1:(tt-2)*kx+kx) =  -t0(1+1:1+kx)';
end
% ----------------------------------------------------------------------
aux             = [t0(1+kx+kz+1:1+kx+kz+1+kx)' t0(1+kx+1:1+kx+kz)'];
P1              = ones(t,1)*aux;
P1(1,:)         = P1(1,:) + [t0(1) t0(1+1:1+kx)' zeros(1,kz)];
% ----------------------------------------------------------------------
o11     = diag(t0(1+kx+kz+1+kx+1+1:1+kx+kz+1+kx+1+t).^2);
o11     = o11 + (t0(1+kx+kz+1+kx+1)^2)*ones(t);
% ----------------------------------------------------------------------
aux     = t0(1+kx+kz+1+kx+1+t+1:1+kx+kz+1+kx+1+t+kx*(t-1))';
o120    = ones(t,1)*aux;
ind     = 1+kx+kz+1+kx+1+t+kx*(t-1)+1;
% ----------------------------------------------------------------------
for tt = 1:t-1
    o120(tt,kx*(tt-1)+1:end) = o120(tt,kx*(tt-1)+1:end) + t0(ind:ind+kx*(t-tt)-1)';
    ind = ind+kx*(t-tt);
end
% ----------------------------------------------------------------------
U1     = (B11*Y1'+B12*Y2'-P1*Z')';
H       = (Y2-U1*inv(o11)*o120)'*Q*(Y2-U1*inv(o11)*o120);
% ----------------------------------------------------------------------
G22_inv     = H/n;
o11_        = o11;
o12_        = o120;
o21_        = o12_';
o22_        = G22_inv + o21_*inv(o11_)*o12_;
o_          = [o11_ o12_; o21_ o22_];
% ----------------------------------------------------------------------
F12         = -inv(o11_)*o12_;
F21         = F12';
% ----------------------------------------------------------------------
P2_         = Y2'*Z*inv(Z'*Z) + F21*U1'*Z*inv(Z'*Z);
% ----------------------------------------------------------------------
A_      = B11;
% ----------------------------------------------------------------------
B_ = [];
aux = [ gamma0_ gamma1_ delta_ ];
for tt = 1:t
% ----------------------------------------------------------------------
    if tt == 1
        B_ = [B_; aux + [ alpha_ beta_ zeros(size(delta_)) ] ];
    else
        B_ = [B_; aux + beta_*P2_( (tt-2)*kx+1:(tt-2)*kx+kx, :) ];
    end
% ----------------------------------------------------------------------
end
% ----------------------------------------------------------------------
C_ = [zeros(1,(t-1)*kx); kron( eye(t-1), beta_ )];
C_ = [eye(t) C_];
% ----------------------------------------------------------------------
var_ = inv(A_)*C_*o_*C_'*inv(A_');
res_ = ( Y1' - inv(A_)*B_*Z' );
% ----------------------------------------------------------------------
M = -(n/2)*log( det(var_) ) - (1/2)*trace( inv(var_)*(res_*res_'));
% ----------------------------------------------------------------------
end
% ----------------------------------------------------------------------



% ----------------------------------------------------------------------
function M = convertxbench(theta,t,kx,n,Y1,Y2,Z,Q)
% ----------------------------------------------------------------------
alpha_  = theta(1);
beta_   = theta(1+1:1+kx)';
gamma0_ = theta(1+kx+1:1+kx+1);
gamma1_ = theta(1+kx+1+1:1+kx+1+kx)';
% ----------------------------------------------------------------------
t0 = theta;
% ----------------------------------------------------------------------
B11 = diag(-ones(t-1,1)*t0(1),-1) + eye(t);
B12 = zeros(t,(t-1)*kx);
% ----------------------------------------------------------------------
for tt = 2:t
    B12(tt,(tt-2)*kx+1:(tt-2)*kx+kx) =  -t0(1+1:1+kx)';
end
% ----------------------------------------------------------------------
aux             = t0(1+kx+1:1+kx+1+kx)';
P1              = ones(t,1)*aux;
P1(1,:)         = P1(1,:) + [t0(1) t0(1+1:1+kx)'];
% ----------------------------------------------------------------------
o11     = diag(t0(1+kx+1+kx+1+1:1+kx+1+kx+1+t).^2);
o11     = o11 + (t0(1+kx+1+kx+1)^2)*ones(t);
% ----------------------------------------------------------------------
aux     = t0(1+kx+1+kx+1+t+1:1+kx+1+kx+1+t+kx*(t-1))';
o120    = ones(t,1)*aux;
ind     = 1+kx+1+kx+1+t+kx*(t-1)+1;
% ----------------------------------------------------------------------
for tt = 1:t-1
    o120(tt,kx*(tt-1)+1:end) = o120(tt,kx*(tt-1)+1:end) + t0(ind:ind+kx*(t-tt)-1)';
    ind = ind+kx*(t-tt);
end
% ----------------------------------------------------------------------
U1     = (B11*Y1'+B12*Y2'-P1*Z')';
H       = (Y2-U1*inv(o11)*o120)'*Q*(Y2-U1*inv(o11)*o120);
% ----------------------------------------------------------------------
G22_inv     = H/n;
o11_        = o11;
o12_        = o120;
o21_        = o12_';
o22_        = G22_inv + o21_*inv(o11_)*o12_;
o_          = [o11_ o12_; o21_ o22_];
% ----------------------------------------------------------------------
F12         = -inv(o11_)*o12_;
F21         = F12';
% ----------------------------------------------------------------------
P2_         = Y2'*Z*inv(Z'*Z) + F21*U1'*Z*inv(Z'*Z);
% ----------------------------------------------------------------------
A_      = B11;
% ----------------------------------------------------------------------
B_ = [];
aux = [ gamma0_ gamma1_ ];
for tt = 1:t
% ----------------------------------------------------------------------
    if tt == 1
        B_ = [B_; aux + [ alpha_ beta_ ] ];
    else
        B_ = [B_; aux + beta_*P2_( (tt-2)*kx+1:(tt-2)*kx+kx, :) ];
    end
% ----------------------------------------------------------------------
end
% ----------------------------------------------------------------------
C_ = [zeros(1,(t-1)*kx); kron( eye(t-1), beta_ )];
C_ = [eye(t) C_];
% ----------------------------------------------------------------------
var_ = inv(A_)*C_*o_*C_'*inv(A_');
res_ = ( Y1' - inv(A_)*B_*Z' );
% ----------------------------------------------------------------------
M = -(n/2)*log( det(var_) ) - (1/2)*trace( inv(var_)*(res_*res_'));
% ----------------------------------------------------------------------
end
% ----------------------------------------------------------------------




% ----------------------------------------------------------------------
function M = convertzbench(theta,t,kx,kz,n,Y1,Y2,Z,Q)
% ----------------------------------------------------------------------
alpha_  = theta(1);
delta_  = theta(1+1:1+kz)';
gamma0_ = theta(1+kz+1:1+kz+1);
% ----------------------------------------------------------------------
t0 = theta;
% ----------------------------------------------------------------------
B11 = diag(-ones(t-1,1)*t0(1),-1) + eye(t);
B12 = zeros(t,(t-1)*kx);
% ----------------------------------------------------------------------
aux             = [t0(1+kz+1)' t0(1+1:1+kz)'];
P1              = ones(t,1)*aux;
P1(1,1)         = P1(1,1) + t0(1);
% ----------------------------------------------------------------------
o11     = diag(t0(1+kz+1+1+1:1+kz+1+1+t).^2);
o11     = o11 + (t0(1+kz+1+1)^2)*ones(t);
% ----------------------------------------------------------------------
aux     = t0(1+kz+1+1+t+1:1+kz+1+1+t+kx*(t-1))';
o120    = ones(t,1)*aux;
ind     = 1+kz+1+1+t+kx*(t-1)+1;
% ----------------------------------------------------------------------
for tt = 1:t-1
    o120(tt,kx*(tt-1)+1:end) = o120(tt,kx*(tt-1)+1:end) + t0(ind:ind+kx*(t-tt)-1)';
    ind = ind+kx*(t-tt);
end
% ----------------------------------------------------------------------
U1     = (B11*Y1'+B12*Y2'-P1*Z')';
H       = (Y2-U1*inv(o11)*o120)'*Q*(Y2-U1*inv(o11)*o120);
% ----------------------------------------------------------------------
G22_inv     = H/n;
o11_        = o11;
o12_        = o120;
o21_        = o12_';
o22_        = G22_inv + o21_*inv(o11_)*o12_;
o_          = [o11_ o12_; o21_ o22_];
% ----------------------------------------------------------------------
A_      = B11;
% ----------------------------------------------------------------------
B_ = [];
aux = [ gamma0_ delta_ ];
for tt = 1:t
% ----------------------------------------------------------------------
    if tt == 1
        B_ = [B_; aux + [ alpha_ zeros(size(delta_)) ] ];
    else
        B_ = [B_; aux ];
    end
% ----------------------------------------------------------------------
end
% ----------------------------------------------------------------------
C_ = [zeros(1,(t-1)*kx)];
C_ = [eye(t) C_];
% ----------------------------------------------------------------------
var_ = inv(A_)*C_*o_*C_'*inv(A_');
res_ = ( Y1' - inv(A_)*B_*Z' );
% ----------------------------------------------------------------------
M = -(n/2)*log( det(var_) ) - (1/2)*trace( inv(var_)*(res_*res_'));
% ----------------------------------------------------------------------
end
% ----------------------------------------------------------------------




% ----------------------------------------------------------------------
function M = likbench(t0,t,kx,kz,n,Y1,Y2,Z,Q)
% ----------------------------------------------------------------------
B11 = diag(-ones(t-1,1)*t0(1),-1) + eye(t);
B12 = zeros(t,(t-1)*kx);
% ----------------------------------------------------------------------
for tt = 2:t
    B12(tt,(tt-2)*kx+1:(tt-2)*kx+kx) =  -t0(1+1:1+kx)';
end
% ----------------------------------------------------------------------
aux             = [t0(1+kx+kz+1:1+kx+kz+1+kx)' t0(1+kx+1:1+kx+kz)'];
P1              = ones(t,1)*aux;
P1(1,:)         = P1(1,:) + [t0(1) t0(1+1:1+kx)' zeros(1,kz)];
% ----------------------------------------------------------------------
o11     = diag(t0(1+kx+kz+1+kx+1+1:1+kx+kz+1+kx+1+t).^2);
o11     = o11 + (t0(1+kx+kz+1+kx+1)^2)*ones(t);
% ----------------------------------------------------------------------
aux     = t0(1+kx+kz+1+kx+1+t+1:1+kx+kz+1+kx+1+t+kx*(t-1))';
o120    = ones(t,1)*aux;
ind     = 1+kx+kz+1+kx+1+t+kx*(t-1)+1;
% ----------------------------------------------------------------------
for tt = 1:t-1
    o120(tt,kx*(tt-1)+1:end) = o120(tt,kx*(tt-1)+1:end) + t0(ind:ind+kx*(t-tt)-1)';
    ind = ind+kx*(t-tt);
end
% ----------------------------------------------------------------------
if det(o11)<=0
     likf   = 0;
else
    U1     = (B11*Y1'+B12*Y2'-P1*Z')';
    H       = (Y2-U1*inv(o11)*o120)'*Q*(Y2-U1*inv(o11)*o120);
    likf    = -(n/2)*log(det(o11)) - (1/2)*sum(diag(inv(o11)*(U1'*U1))) - (n/2)*log(det(H/n));
end
% ----------------------------------------------------------------------
M = -likf;
% ----------------------------------------------------------------------
end
% ----------------------------------------------------------------------



% ----------------------------------------------------------------------
function [M] = likxbench(t0,t,kx,n,Y1,Y2,Z,Q)
% ----------------------------------------------------------------------
B11 = diag(-ones(t-1,1)*t0(1),-1) + eye(t);
B12 = zeros(t,(t-1)*kx);
% ----------------------------------------------------------------------
for tt = 2:t
    B12(tt,(tt-2)*kx+1:(tt-2)*kx+kx) =  -t0(1+1:1+kx)';
end
% ----------------------------------------------------------------------
aux             = t0(1+kx+1:1+kx+1+kx)';
P1              = ones(t,1)*aux;
P1(1,:)         = P1(1,:) + [t0(1) t0(1+1:1+kx)'];
% ----------------------------------------------------------------------
o11     = diag(t0(1+kx+1+kx+1+1:1+kx+1+kx+1+t).^2);
o11     = o11 + (t0(1+kx+1+kx+1)^2)*ones(t);
% ----------------------------------------------------------------------
aux     = t0(1+kx+1+kx+1+t+1:1+kx+1+kx+1+t+kx*(t-1))';
o120    = ones(t,1)*aux;
ind     = 1+kx+1+kx+1+t+kx*(t-1)+1;
% ----------------------------------------------------------------------
for tt = 1:t-1
    o120(tt,kx*(tt-1)+1:end) = o120(tt,kx*(tt-1)+1:end) + t0(ind:ind+kx*(t-tt)-1)';
    ind = ind+kx*(t-tt);
end
% ----------------------------------------------------------------------
if det(o11)<=0
     likf   = 0;
else
    U1     = (B11*Y1'+B12*Y2'-P1*Z')';
    H       = (Y2-U1*inv(o11)*o120)'*Q*(Y2-U1*inv(o11)*o120);
    likf    = -(n/2)*log(det(o11)) - (1/2)*sum(diag(inv(o11)*(U1'*U1))) - (n/2)*log(det(H/n));
end
% ----------------------------------------------------------------------
M = -likf;
% ----------------------------------------------------------------------
end
% ----------------------------------------------------------------------


% ----------------------------------------------------------------------
function [M] = likzbench(t0,t,kx,kz,n,Y1,Y2,Z,Q)
% ----------------------------------------------------------------------
B11 = diag(-ones(t-1,1)*t0(1),-1) + eye(t);
B12 = zeros(t,(t-1)*kx);
% ----------------------------------------------------------------------
aux             = [t0(1+kz+1)' t0(1+1:1+kz)'];
P1              = ones(t,1)*aux;
P1(1,1)         = P1(1,1) + t0(1);
% ----------------------------------------------------------------------
o11     = diag(t0(1+kz+1+1+1:1+kz+1+1+t).^2);
o11     = o11 + (t0(1+kz+1+1)^2)*ones(t);
% ----------------------------------------------------------------------
aux     = t0(1+kz+1+1+t+1:1+kz+1+1+t+kx*(t-1))';
o120    = ones(t,1)*aux;
ind     = 1+kz+1+1+t+kx*(t-1)+1;
% ----------------------------------------------------------------------
for tt = 1:t-1
    o120(tt,kx*(tt-1)+1:end) = o120(tt,kx*(tt-1)+1:end) + t0(ind:ind+kx*(t-tt)-1)';
    ind = ind+kx*(t-tt);
end
% ----------------------------------------------------------------------
if det(o11)<=0
     likf   = 0;
else
    U1     = (B11*Y1'+B12*Y2'-P1*Z')';
    H       = (Y2-U1*inv(o11)*o120)'*Q*(Y2-U1*inv(o11)*o120);
    likf    = -(n/2)*log(det(o11)) - (1/2)*sum(diag(inv(o11)*(U1'*U1))) - (n/2)*log(det(H/n));
end
% ----------------------------------------------------------------------
M = -likf;
% ----------------------------------------------------------------------
end
% ----------------------------------------------------------------------


% ----------------------------------------------------------------------
function [CorrPiFreq,CorrBayesFreq,RafteryLewis] = diagnostics(steps,m_model_m,m_model_post,m_model_prior)
% ----------------------------------------------------------------------
drop_T = floor( steps*0.1 );
% ----------------------------------------------------------------------
T1 = floor( (steps - drop_T)/2 );
T2 = T1;
% ----------------------------------------------------------------------
sampleF_m       = m_model_m(:,      end-(T1+T2)+1:end);
sampleF_post    = m_model_post(:,   end-(T1+T2)+1:end);
sampleF_prior   = m_model_prior(:,  end-(T1+T2)+1:end);
% ----------------------------------------------------------------------
chain_m     = zeros(1,T1+T2);
% ----------------------------------------------------------------------
pool_post   = [];
pool_prior  = [];
count_m     = [];
% ----------------------------------------------------------------------
a_ones      = ones(1,T1+T2);
% ----------------------------------------------------------------------

% ----------------------------------------------------------------------
iter        = 0;
% ----------------------------------------------------------------------
while nnz(chain_m)<T1+T2
% ----------------------------------------------------------------------
    iter    = iter + 1;
% ----------------------------------------------------------------------
    ind_select      = find( chain_m==0, 1, 'first');
    select_model    = sampleF_m(:,ind_select);
    aux0            = sum( abs(sampleF_m-select_model*a_ones ) );
    ind_m_chain     = find(aux0==0);
% ----------------------------------------------------------------------
    chain_m(ind_m_chain)  = iter;
% ----------------------------------------------------------------------
    pool_post   = [ pool_post   sampleF_post(ind_select)  ];
    pool_prior  = [ pool_prior  sampleF_prior(ind_select) ];
    count_m     = [ count_m     numel(ind_m_chain)];
% ----------------------------------------------------------------------
end
% ----------------------------------------------------------------------


% ----------------------------------------------------------------------
% Compute transition matrix
% ----------------------------------------------------------------------
nr_model    = max(chain_m);
chain_1     = chain_m(1:T1);
chain_2     = chain_m(T1+1:end);
max_chain_1 = max(chain_1);
% ----------------------------------------------------------------------
P1 = zeros(nr_model);
% ----------------------------------------------------------------------
for t = 1:T1-1
% ----------------------------------------------------------------------
    cur_model   = chain_1(t);
    trans_model = chain_1(t+1);
% ----------------------------------------------------------------------
    P1(cur_model,trans_model) = P1(cur_model,trans_model) + 1;
% ----------------------------------------------------------------------
end
% ----------------------------------------------------------------------
if max_chain_1 == chain_1(end)
% ----------------------------------------------------------------------
    add_ind     = find(chain_2<max_chain_1,1,'first');
    add_chain   = chain_2(1:add_ind);
% ----------------------------------------------------------------------
    for t = 1:add_ind-1
% ----------------------------------------------------------------------
        cur_model   = add_chain(t);
        trans_model = add_chain(t+1);
% ----------------------------------------------------------------------
        P1(cur_model,trans_model) = P1(cur_model,trans_model) + 1;
% ----------------------------------------------------------------------
    end
% ----------------------------------------------------------------------
end
% ----------------------------------------------------------------------

% ----------------------------------------------------------------------
P1      = sparse( P1 );
% ----------------------------------------------------------------------
aux_P1              = sum(P1,2);
aux_P1(aux_P1==0)   = 1;
aux_P1              = aux_P1*ones(1,nr_model);
% ----------------------------------------------------------------------
P1      = P1./aux_P1;
P1      = full(P1);
% ----------------------------------------------------------------------


% ----------------------------------------------------------------------
% Compute stationary distribution by iteration
% ----------------------------------------------------------------------
stat0 = zeros(1,nr_model);
% ----------------------------------------------------------------------
stat0(1)=0.5;
stat0(2)=0.5;
% ----------------------------------------------------------------------
metric = 1; 
% ----------------------------------------------------------------------
while metric > 1e-10
% ----------------------------------------------------------------------
    stat1   = stat0*P1;
    metric  = max(abs(stat1-stat0)./(1+stat1));
    stat0   = stat1;
% ----------------------------------------------------------------------
end
% ----------------------------------------------------------------------


% ----------------------------------------------------------------------
% Compute relative frequencies in sample 2
% ----------------------------------------------------------------------
freq_2 = zeros(1,nr_model);
% ----------------------------------------------------------------------
for t = 1:nr_model
% ----------------------------------------------------------------------
    freq_2(t) = nnz( chain_2==t );
% ----------------------------------------------------------------------
end
% ----------------------------------------------------------------------
freq_2 = freq_2./T2;
% ----------------------------------------------------------------------
corr0       = corrcoef([freq_2' stat0']);
CorrPiFreq  = corr0(1,2); 
% ----------------------------------------------------------------------



% ----------------------------------------------------------------------
% ---- Relative frequencies in full sample / and Bayes factor
% ----------------------------------------------------------------------
aux0    = pool_post + pool_prior;
% ----------------------------------------------------------------------
ind_max = find(aux0 == max(aux0));
% ----------------------------------------------------------------------
rel_freq    = count_m./count_m(ind_max);
rel_bayes   = exp(aux0)./exp(aux0(ind_max));
% ----------------------------------------------------------------------
corr0           = corrcoef([rel_freq' rel_bayes']);
CorrBayesFreq   = corr0(1,2);
% ----------------------------------------------------------------------



% ----------------------------------------------------------------------
post_   = m_model_post(1,end-(T1+T2)+1:end);
p       = size(post_,2);
% ----------------------------------------------------------------------
r   = 0.005;
ss  = 0.95;
eps = 0.001;
q   = 0.025;
% ----------------------------------------------------------------------
n_min   = ( norminv( (ss+1)/2)*sqrt(q*(1-q))/r )^2;
quant   = quantile(post_,q);
% ----------------------------------------------------------------------
z = post_<quant;
P = zeros(2);
% ----------------------------------------------------------------------
for t = 1:p-1
% ----------------------------------------------------------------------
    cur_state   = z(t)+1;
    trans_state = z(t+1)+1;
    P(cur_state,trans_state) = P(cur_state,trans_state) + 1;
% ----------------------------------------------------------------------
end
% ----------------------------------------------------------------------
P = P./( sum(P,2)*ones(1,2) );
a = P(1,2);  b = P(2,1);
% ----------------------------------------------------------------------
N_run = ( a*b*(2-a-b)/((a+b)^3) )/( ( r/normcdf((1+ss)/2) )^2 );
% ----------------------------------------------------------------------
M     = log(eps*(a+b)/max([a b]))/log(1-a-b);
% ----------------------------------------------------------------------
RafteryLewis  = N_run/n_min;
% ----------------------------------------------------------------------
end
% ----------------------------------------------------------------------
