# AIC as specified in PSS
AIC_pss <- function(model){
    # maximized log-likelihood value of the model
    LLp <- logLik(model)
    # number of freely estimated coefficients
    sp <- length(model$coefficients)
    LLp - sp
}

# BIC as specified in PSS
SBC_pss <- function(model){
    # maximized log-likelihood value of the model
    LLp <- logLik(model)
    # number of freely estimated coefficients
    sp <- length(model$coefficients)
    # natural logarithm (ln) of the sample size
    lnT <- log(length(model$residuals))
    LLp - (sp/2)*lnT
}

# function to replicate Table I. p. 311
Table_I <- function(uecm_model, LM_orders, latex_type) {
    Table_I <- sapply(1:length(uecm_model), function(i) {
        # LM test
        LM_list <- list()
        for (z in 1:length(LM_orders)){
            LM <- bgtest(uecm_model[[i]], order = LM_orders[z], type = "Chisq")
            LM_list[[z]] <- list(stat = LM$statistic,
                        star = ifelse(LM$p.value < 0.01, "*",
                                      ifelse(LM$p.value < 0.05, "**",
                                             ifelse(LM$p.value < 0.1, "***", ""))))
        }
        # replicate the Table I and round at 2 decimals
        table <- data.frame(
            AIC = AIC_pss(uecm_model[[i]]),
            SBC = SBC_pss(uecm_model[[i]])
        )
        for (z in 1:length(LM_orders)){
            table <- cbind(table, LM_list[[z]]$stat)
        }
        # round at 2 decimals
        table <- round(table, 2)
        if (latex_type == FALSE) {
            # add the statistical significance anotation
            for (z in 1:length(LM_orders)){
                table[[z+2]] <- paste0(table[[z+2]], LM_list[[z]]$star)
            }
        } else { # LaTeX format
            for (z in 1:length(LM_orders)){
                table[[z+2]] <- paste0("$", as.character(table[[z+2]]), "^{", LM_list[[z]]$star, "}$")
            }
        }
        colnames(table) <- c("AIC", "SBC", paste0("X2sc_", LM_orders))
        # return the table
        table
    })
    # final Table I
    Table_I <- data.frame(p = 1:length(uecm_model), t(Table_I))
    return(Table_I)
}

build_Table_II <- function(uecm_model_cases23, uecm_model_cases45, cases = c(4,5,3), replication, latex_type){
    table <- data.frame()
    if (latex_type == TRUE) {
        ltx1 <- "$"
        ltx2 <- "^"
    } else {
        ltx1 <- c()
        ltx2 <- c()
    }
    for(z in 1:(length(uecm_model_cases23))) {
        if (replication == "narrow") {
            fill_table <- c(z+3)
        } else if (replication == "wide") {
            fill_table <- paste0("ARDL(",
                paste0(paste0(uecm_model_cases23[[z]]$order[1:4], ",", collapse = ""),
                    uecm_model_cases23[[z]]$order[5]),
                ")")
        }
        for(j in cases) {
            if( j %in% c(4,5)) {
                correct_uecm <- uecm_model_cases45
            } else {
                correct_uecm <- uecm_model_cases23
            }
            test <- bounds_f_test(correct_uecm[[z]], case = j, alpha = 0.05)
            value <- ifelse(test$statistic < test$parameters[1], paste0(ltx1, round(test$statistic,2), ltx2, "a", ltx1),
                            ifelse(test$statistic > test$parameters[2], paste0(ltx1, round(test$statistic, 2), ltx2, "c", ltx1),
                                   paste0(ltx1, round(test$statistic,2), ltx2, "b", ltx1)))
            fill_table <- c(fill_table, value)
            if (j %in% c(3,5)) {
                test <- bounds_t_test(object = correct_uecm[[z]], case = j, alpha = 0.05)
                value <- ifelse(abs(test$statistic) < abs(test$parameters[1]), paste0(ltx1, round(test$statistic,2), ltx2, "a", ltx1),
                                ifelse(abs(test$statistic) > abs(test$parameters[2]), 
                                       paste0(ltx1, ifelse(round(test$statistic, 2) %% 2 != 0, round(test$statistic, 2), paste0(round(test$statistic, 2), ".00")), ltx2, "c", ltx1),
                                       paste0(ltx1, round(test$statistic,2), ltx2, "b", ltx1)))
                fill_table <- c(fill_table, value)
            }
        }
        fill_table <- data.frame(matrix(fill_table, nrow = 1), stringsAsFactors = FALSE)
        if (replication == "narrow") {
            colnames(fill_table) <- c("p", "F_IV", "F_V", "t_V", "F_III", "t_III")
        } else if (replication == "wide") {
            colnames(fill_table) <- c("ARDL(p, q1, q2, q3, q4)", "F_II", "F_III", "t_III", "F_IV", "F_V", "t_V")
        }
        table <- data.frame(rbind(table, fill_table))
        if (replication == "narrow") {
            colnames(table) <- c("p", "F_IV", "F_V", "t_V", "F_III", "t_III")
        } else if (replication == "wide") {
            colnames(table) <- c("ARDL(p, q1, q2, q3, q4)", "F_II", "F_III", "t_III", "F_IV", "F_V", "t_V")
        }
    }
    if (latex_type == TRUE) {
        if (replication == "narrow") {
            colnames(table) <- c("p", "$F_{IV}$", "$F_V$", "$t_V$", "$F_{III}$", "$t_{III}$")
        } else if (replication == "wide") {
            colnames(table) <- c("ARDL(p, $q_1$, $q_2$, $q_3$, $q_4$)", "$F_{II}$", "$F_{III}$", "$t_{III}$", "$F_{IV}$", "$F_{V}$", "$t_{V}$")
        }
    }
    return(table)
}

build_Table_III <- function(model){
    tableIII <- data.frame(summary(model)$coefficients[,-3])
    if("recm" %in% class(model)){
        tableIII <- rbind(tableIII[nrow(tableIII),], tableIII[2:(nrow(tableIII)-3),], 
                          tableIII[1,], tableIII[(nrow(tableIII)-2):(nrow(tableIII)-1),])
        regressor <- c("$\\hat{u}_{t-1}$", paste0("$\\Delta w_{t-", 1:5,"}$"), 
                       "$\\Delta Prod_t$", "$\\Delta UR_t$", paste0("$\\Delta UR_{t-", 1:4,"}$"),
                       "$\\Delta Wedge_t$", paste0("$\\Delta Wedge_{t-", 1:3,"}$"),
                       "$\\Delta Union_t$", paste0("$\\Delta Union_{t-", 1:4,"}$"),
                       "Intercept", "$D7475_t$", "$D7579_t$")
    } else if ("uecm" %in% class(model)) {
        tableIII <- rbind(tableIII[2,], tableIII[7:(nrow(tableIII)-2),], tableIII[1,],
                          tableIII[(nrow(tableIII)-1):nrow(tableIII),], tableIII[3:6,])
        regressor <- c("$w_{t-1}$", paste0("$\\Delta w_{t-", 1:5,"}$"),
                       "$\\Delta Prod_t$", "$\\Delta UR_t$", paste0("$\\Delta UR_{t-", 1:4,"}$"),
                       "$\\Delta Wedge_t$", paste0("$\\Delta Wedge_{t-", 1:3,"}$"),
                       "$\\Delta Union_t$", paste0("$\\Delta Union_{t-", 1:4,"}$"),
                       "Intercept", "$D7475_t$", "$D7579_t$", "$Prod_{t-1}$",
                       "$UR_{t-1}$", "$Wedge_{t-1}$", "$Union_{t-1}$")
    }
    tableIII <- cbind(regressor, tableIII)
    colnames(tableIII) <- c("Regressor", "Coefficient", "Standard error", "p-value")
    return(tableIII)
}

build_Table_III_wide <- function(model){
    tableIII <- data.frame(summary(model)$coefficients[,-3])
    regressor <- c("Intercept", "$w_{t-1}$", "$Prod_{t-1}$", "$UR$", "$Wedge_{t-1}$", "$Union_{t-1}$",
                    paste0("$\\Delta w_{t-", 1:3,"}$"), "$\\Delta Prod_t$", "$\\Delta Wedge_t$",
                    paste0("$\\Delta Wedge_{t-", 1:4,"}$"), "$D7475_t$", "$D7579_t$")
    tableIII <- cbind(regressor, tableIII)
    colnames(tableIII) <- c("Regressor", "Coefficient", "Standard error", "p-value")
    return(tableIII)
}

ecm_tests <- function(model){
    # Adjusted R^2
    r2 <- summary(model)$adj.r.squared
    
    # Standard Error
    sig <- summary(model)$sigma
    
    # AIC
    aic <- AIC_pss(model)
    
    # SBC
    sbc <- SBC_pss(model)
    
    # serial Correlation
    xsc <- bgtest(model, order = 4)
    
    # normality
    N <- jarque.bera.test(residuals(model))
    
    # homoscedasticity
    uecm_lm <- model
    class(uecm_lm) <- "lm"
    H <- ols_test_breusch_pagan(uecm_lm)
    
    # functional form
    # 1 df so checks only for a single power
    # for any power it passes the test
    # Also for 2:3 or higher e.g. 2:6 it passes
    if ("recm" %in% class(model)){
        library(dynlm)
        model <- dynlm(d(w) ~ d(L(w, 1)) + d(L(w, 2)) + d(L(w, 3)) + d(L(w, 4)) + 
                           d(L(w, 5)) + d(Prod) + d(UR) + d(L(UR, 1)) + d(L(UR, 2)) + 
                           d(L(UR, 3)) + d(L(UR, 4)) + d(Wedge) + d(L(Wedge, 1)) + 
                           d(L(Wedge,2)) + d(L(Wedge, 3)) + d(Union) + d(L(Union, 1)) + 
                           d(L(Union, 2)) + d(L(Union, 3)) + d(L(Union, 4)) + D7475 + D7579 + ect,
                       data = ts(model$data))
    }
    ff <- resettest(model, power = 2, type = c("fitted"))
    
    return(list(r2=r2, sig=sig, aic=aic, sbc=sbc, xsc=xsc, ff=ff, N=N, H=H))
}

unit_root_table <- function(data, levels) {
    if(levels == TRUE){
        data0 <- data
    } else {
        data0 <- diff(data)
    }
    
    adf_w_n <- CADFtest(data0[,'w'], type = "none", max.lag.y = 7, criterion = "AIC")
    adf_w_c <- CADFtest(data0[,'w'], type = "drift", max.lag.y = 7, criterion = "AIC")
    adf_w_ct <- CADFtest(data0[,'w'], type = "trend", max.lag.y = 7, criterion = "AIC")
    adf_prod_n <- CADFtest(data0[,'Prod'], type = "none", max.lag.y = 7, criterion = "AIC")
    adf_prod_c <- CADFtest(data0[,'Prod'], type = "drift", max.lag.y = 7, criterion = "AIC")
    adf_prod_ct <- CADFtest(data0[,'Prod'], type = "trend", max.lag.y = 7, criterion = "AIC")
    adf_ur_n <- CADFtest(data0[,'UR'], type = "none", max.lag.y = 7, criterion = "AIC")
    adf_ur_c <- CADFtest(data0[,'UR'], type = "drift", max.lag.y = 7, criterion = "AIC")
    adf_ur_ct <- CADFtest(data0[,'UR'], type = "trend", max.lag.y = 7, criterion = "AIC")
    adf_wedge_n <- CADFtest(data0[,'Wedge'], type = "none", max.lag.y = 7, criterion = "AIC")
    adf_wedge_c <- CADFtest(data0[,'Wedge'], type = "drift", max.lag.y = 7, criterion = "AIC")
    adf_wedge_ct <- CADFtest(data0[,'Wedge'], type = "trend", max.lag.y = 7, criterion = "AIC")
    adf_union_n <- CADFtest(data0[,'Union'], type = "none", max.lag.y = 7, criterion = "AIC")
    adf_union_c <- CADFtest(data0[,'Union'], type = "drift", max.lag.y = 7, criterion = "AIC")
    adf_union_ct <- CADFtest(data0[,'Union'], type = "trend", max.lag.y = 7, criterion = "AIC")
    adf_unionr_n <- CADFtest(data0[,'UnionR'], type = "none", max.lag.y = 7, criterion = "AIC")
    adf_unionr_c <- CADFtest(data0[,'UnionR'], type = "drift", max.lag.y = 7, criterion = "AIC")
    adf_unionr_ct <- CADFtest(data0[,'UnionR'], type = "trend", max.lag.y = 7, criterion = "AIC")
    
    pp_w <- aTSA::pp.test(data0[,'w'], output = FALSE)
    pp_prod <- aTSA::pp.test(data0[,'Prod'], output = FALSE)
    pp_ur <- aTSA::pp.test(data0[,'UR'], output = FALSE)
    pp_wedge <- aTSA::pp.test(data0[,'Wedge'], output = FALSE)
    pp_union <- aTSA::pp.test(data0[,'Union'], output = FALSE)
    pp_unionr <- aTSA::pp.test(data0[,'UnionR'], output = FALSE)
    
    round_dec_stat = 2
    round_dec_pval = 3
    tab_unit_root <- data.frame(none = c(paste0(round(adf_w_n$statistic, round_dec_stat), " (", adf_w_n$max.lag.y, ")"),
                                         paste0("(", round(adf_w_n$p.value, round_dec_pval), ")"),
                                         paste0(round(adf_prod_n$statistic, round_dec_stat), " (", adf_prod_n$max.lag.y, ")"),
                                         paste0("(", round(adf_prod_n$p.value, round_dec_pval), ")"),
                                         paste0(round(adf_ur_n$statistic, round_dec_stat), " (", adf_ur_n$max.lag.y, ")"),
                                         paste0("(", round(adf_ur_n$p.value, round_dec_pval), ")"),
                                         paste0(round(adf_wedge_n$statistic, round_dec_stat), " (", adf_wedge_n$max.lag.y, ")"),
                                         paste0("(", round(adf_wedge_n$p.value, round_dec_pval), ")"),
                                         paste0(round(adf_union_n$statistic, round_dec_stat), " (", adf_union_n$max.lag.y, ")"),
                                         paste0("(", round(adf_union_n$p.value, round_dec_pval), ")"),
                                         paste0(round(adf_unionr_n$statistic, round_dec_stat), " (", adf_unionr_n$max.lag.y, ")"),
                                         paste0("(", round(adf_unionr_n$p.value, round_dec_pval), ")")),
                                intercept = c(paste0(round(adf_w_c$statistic, round_dec_stat), " (", adf_w_c$max.lag.y, ")"),
                                              paste0("(", round(adf_w_c$p.value, round_dec_pval), ")"),
                                              paste0(round(adf_prod_c$statistic, round_dec_stat), " (", adf_prod_c$max.lag.y, ")"),
                                              paste0("(", round(adf_prod_c$p.value, round_dec_pval), ")"),
                                              paste0(round(adf_ur_c$statistic, round_dec_stat), " (", adf_ur_c$max.lag.y, ")"),
                                              paste0("(", round(adf_ur_c$p.value, round_dec_pval), ")"),
                                              paste0(round(adf_wedge_c$statistic, round_dec_stat), " (", adf_wedge_c$max.lag.y, ")"),
                                              paste0("(", round(adf_wedge_c$p.value, round_dec_pval), ")"),
                                              paste0(round(adf_union_c$statistic, round_dec_stat), " (", adf_union_c$max.lag.y, ")"),
                                              paste0("(", round(adf_union_c$p.value, round_dec_pval), ")"),
                                              paste0(round(adf_unionr_c$statistic, round_dec_stat), " (", adf_unionr_c$max.lag.y, ")"),
                                              paste0("(", round(adf_unionr_c$p.value, round_dec_pval), ")")),
                                "intercept and trend" = c(paste0(round(adf_w_ct$statistic, round_dec_stat), " (", adf_w_ct$max.lag.y, ")"),
                                                          paste0("(", round(adf_w_ct$p.value, round_dec_pval), ")"),
                                                          paste0(round(adf_prod_ct$statistic, round_dec_stat), " (", adf_prod_ct$max.lag.y, ")"),
                                                          paste0("(", round(adf_prod_ct$p.value, round_dec_pval), ")"),
                                                          paste0(round(adf_ur_ct$statistic, round_dec_stat), " (", adf_ur_ct$max.lag.y, ")"),
                                                          paste0("(", round(adf_ur_ct$p.value, round_dec_pval), ")"),
                                                          paste0(round(adf_wedge_ct$statistic, round_dec_stat), " (", adf_wedge_ct$max.lag.y, ")"),
                                                          paste0("(", round(adf_wedge_ct$p.value, round_dec_pval), ")"),
                                                          paste0(round(adf_union_ct$statistic, round_dec_stat), " (", adf_union_ct$max.lag.y, ")"),
                                                          paste0("(", round(adf_union_ct$p.value, round_dec_pval), ")"),
                                                          paste0(round(adf_unionr_ct$statistic, round_dec_stat), " (", adf_unionr_ct$max.lag.y, ")"),
                                                          paste0("(", round(adf_unionr_ct$p.value, round_dec_pval), ")")),
                                c(rep("", 12)),
                                none = c(round(pp_w[1,2], round_dec_stat),
                                         paste0("(", round(pp_w[1,3], round_dec_pval), ")"),
                                         round(pp_prod[1,2], round_dec_stat),
                                         paste0("(", round(pp_prod[1,3], round_dec_pval), ")"),
                                         round(pp_ur[1,2], round_dec_stat),
                                         paste0("(", round(pp_ur[1,3], round_dec_pval), ")"),
                                         round(pp_wedge[1,2], round_dec_stat),
                                         paste0("(", round(pp_wedge[1,3], round_dec_pval), ")"),
                                         round(pp_union[1,2], round_dec_stat),
                                         paste0("(", round(pp_union[1,3], round_dec_pval), ")"),
                                         round(pp_unionr[1,2], round_dec_stat),
                                         paste0("(", round(pp_unionr[1,3], round_dec_pval), ")")),
                                intercept = c(round(pp_w[2,2], round_dec_stat),
                                              paste0("(", round(pp_w[2,3], round_dec_pval), ")"),
                                              round(pp_prod[2,2], round_dec_stat),
                                              paste0("(", round(pp_prod[2,3], round_dec_pval), ")"),
                                              round(pp_ur[2,2], round_dec_stat),
                                              paste0("(", round(pp_ur[2,3], round_dec_pval), ")"),
                                              round(pp_wedge[2,2], round_dec_stat),
                                              paste0("(", round(pp_wedge[2,3], round_dec_pval), ")"),
                                              round(pp_union[2,2], round_dec_stat),
                                              paste0("(", round(pp_union[2,3], round_dec_pval), ")"),
                                              round(pp_unionr[2,2], round_dec_stat),
                                              paste0("(", round(pp_unionr[2,3], round_dec_pval), ")")),
                                "intercept and trend" = c(round(pp_w[3,2], round_dec_stat),
                                                          paste0("(", round(pp_w[3,3], round_dec_pval), ")"),
                                                          round(pp_prod[3,2], round_dec_stat),
                                                          paste0("(", round(pp_prod[3,3], round_dec_pval), ")"),
                                                          round(pp_ur[3,2], round_dec_stat),
                                                          paste0("(", round(pp_ur[3,3], round_dec_pval), ")"),
                                                          round(pp_wedge[3,2], round_dec_stat),
                                                          paste0("(", round(pp_wedge[3,3], round_dec_pval), ")"),
                                                          round(pp_union[3,2], round_dec_stat),
                                                          paste0("(", round(pp_union[3,3], round_dec_pval), ")"),
                                                          round(pp_unionr[3,2], round_dec_stat),
                                                          paste0("(", round(pp_unionr[3,3], round_dec_pval), ")"))
    )
    
    if (levels == TRUE) {
        tab_unit_root <- cbind(variable = c("w", "", "Prod", "", "UR", "", "Wedge", "", "Union", "", "UnionR", ""), tab_unit_root)
    } else {
        tab_unit_root <- cbind(variable = c("$\\Delta w$", "", "$\\Delta Prod$", "", "$\\Delta UR$", "", "$\\Delta Wedge$", "", "$\\Delta Union$", "", "$\\Delta UnionR$", ""), tab_unit_root)
    }
    colnames(tab_unit_root) <- c("variable", "none", "intercept", "int. and trend", " ", "none", "intercept", "int. and trend")
    return(tab_unit_root)
}
