/*
AUTHOR:   Todd Clark, Federal Reserve Bank of Cleveland, todd.clark@researchfed.org
Code associated with Andrea Carriero, Todd Clark and Massimiliano Marcellino, 
"Nowcasting Tail Risks to Economic Activity at a Weekly Frequency"

Program for reading in forecast draws to compile forecasts and statistics:  BQR and PQR specifications (which one controlled by switch at the top)
BQR and PQR inputs and code differ a bit from the SV case because we have draws for multiple quantiles, unlike the SV case.
The program reads in 5k draws for all 19 forecast quantiles, thins using an interval of 1/5,
fits an empirical density (calling a procedure to do so), takes 19k draws from this density,
and then computes the forecast metrics of interest (quantile forecasts, QS, VaR-ES score, etc.)
Results are written to Excel file for compilation of results by other programs.  Note that while
the program writes both the "direct" quantile estimates that are posterior means of the 5k
draws for each quantile's regression, the results in the paper use the quantile estimates obtained
from the smoothed densities obtained using 19k draws from all 19 quantiles.
*/

dis %dateandtime()
dis %ratsversion()
comp overallstart = %cputime()

*********************
********************* BASIC SETUP & PARAMETER ENTRY
*********************

******* case-specific setup
comp varset = 1  ;* M=1, M-F=2, M+small=3, M+large=4, M-F+small=5, M-F+large=6
comp modset = 1  ;* BQR=1, PQR=2
comp ndraws =  5000	     ;*total number of Gibbs draws

******** setup of sample
comp useactual = 1          ;* adjustment for obs lost of last vintage relative to eval sample end date (used to set this at 2)
comp styr = 1947
cal(q) styr:1
comp stsmpl = styr:1	;*earliest period with data

comp endvint = 2021:1      ;* last quarter of data vintages considered
comp endsmpl = endvint - useactual
comp endsmpl19 = 2019:4

all endvint
smpl stsmpl endvint

******** setup of forecast specs and actual GDP series
comp npredmo = 4                 ;* number of months at which we form forecasts: months 1-3 of quarter t plus month 1 of quarter t+1
comp nweek = 4                   ;* number of weeks used in month
comp totweek = npredmo*nweek-1     ;* total number of predictions.  -1 because we don't form a forecast for a 16th week, by which time gdp(t+1) becomes available

** read in GDP estimates used as actuals in evaluation of forecasts:  these are 2nd avail in the quarterly RTDSM
open data ../data/GDPactuals.secondrelease.xls
data(format=xls,org=col) / actualGDP
close

******** setup of forecast input options among models and variable sets, as well as label
comp [vec[str]] filelabel = ||"BQR_M",$
"BQR_MF",$
"BQR_M_smallwkly",$
"BQR_M_largewkly",$
"BQR_MF_smallwkly",$
"BQR_MF_largewkly",$
"PQR_M",$
"PQR_MF",$
"PQR_M_smallwkly",$
"PQR_M_largewkly",$
"PQR_MF_smallwkly",$
"PQR_MF_largewkly"||

comp [vec[str]] methlabel = ||"base M:  BQR",$
"base M-F:  BQR",$
"base M + small weekly:  BQR",$
"base M + large weekly:  BQR",$
"base M-F + small weekly:  BQR",$
"base M-F + large weekly:  BQR",$
"base M:  PQR",$
"base M-F:  PQR",$
"base M + small weekly:  PQR",$
"base M + large weekly:  PQR",$
"base M-F + small weekly:  PQR",$
"base M-F + large weekly:  PQR"||

comp [vec[str]] chartfnames = ||"BQR_M_*.pdf",$
"BQR_MF_*.pdf",$
"BQR_M_smallwkly_*.pdf",$
"BQR_M_largewkly_*.pdf",$
"BQR_MF_smallwkly_*.pdf",$
"BQR_MF_largewkly_*.pdf",$
"PQR_M_*.pdf",$
"PQR_MF_*.pdf",$
"PQR_M_smallwkly_*.pdf",$
"PQR_M_largewkly_*.pdf",$
"PQR_MF_smallwkly_*.pdf",$
"PQR_MF_largewkly_*.pdf"||

comp nvarset = 6  ;* M=1, M-F=2, M+small=3, M+large=4, M-F+small=5, M-F+large=6
comp nmodset = 2  ;* BQR=1, PQR=2
comp thispos = (modset-1)*nvarset+varset
comp [str] basefilename = "fcdraws/"+filelabel(thispos)+"/"
comp [str] modeldesc = methlabel(thispos)
comp chartfilename = filelabel(thispos)+"_*.pdf"
gsave(format=pdf) &chartfilename

** setting start date of forecast sample
if varset<=2
 comp stvint = 1985:1 
else if varset==3.or.varset==5
 comp stvint = 2000:1 
else if varset==4.or.varset==6
 comp stvint = 2007:1 
endif

******************************** stuff for forecast processing and storage
sou(noecho) ../procedures/fcmoments_kernelsmoothed.src

comp nquant = 19
dec vec pctiles(nquant) quantfc(nquant) qscorevec(nquant)
ewise pctiles(i) = float(i)/(nquant+1)
dis ####.## pctiles(1) pctiles(2) pctiles(nquant-1) pctiles(nquant)

dec vec cwCRPSweights(nquant)
ewise cwCRPSweights(i) = (1./nquant)*(1.-pctiles(i))^2.  ;* left tail-weighted version

dec vec[ser] directpct50(totweek) directpct05(totweek) directpct10(totweek) directpct90(totweek) directpct95(totweek)
do mm = 1,totweek                   ;* create series labels to facilitate later reading results from Excel file created below
 labels directpct50(mm)
 # 'directpct50_m'+%string(mm)
 labels directpct05(mm)
 # 'directpct05_m'+%string(mm)
 labels directpct10(mm)
 # 'directpct10_m'+%string(mm)
 labels directpct90(mm)
 # 'directpct90_m'+%string(mm)
 labels directpct95(mm)
 # 'directpct95_m'+%string(mm)
end do mm
clear directpct50 directpct05 directpct10 directpct90 directpct95

*** additional storage of results from empirical distributions computed with draws of quantile forecasts
dec vec[ser] fcser(totweek) pct05ser(totweek) pct10ser(totweek) pct90ser(totweek) pct95ser(totweek) crpsser(totweek) qwcrpsser(totweek) $
 qs05ser(totweek) qs10ser(totweek) es05ser(totweek) es10ser(totweek) fzg05ser(totweek) fzg10ser(totweek)

do mm = 1,totweek                   ;* create ser labels to facilitate later reading results from Excel file created below
 labels fcser(mm)
 # 'fcseries_m'+%string(mm)
 labels pct05ser(mm)
 # 'pct05_m'+%string(mm)
 labels pct10ser(mm)
 # 'pct10_m'+%string(mm)
 labels pct90ser(mm)
 # 'pct90_m'+%string(mm)
 labels pct95ser(mm)
 # 'pct95_m'+%string(mm)
 labels crpsser(mm)
 # 'crps_m'+%string(mm)
 labels qwcrpsser(mm)
 # 'qwcrps_m'+%string(mm)
 labels qs05ser(mm)
 # 'qs05_m'+%string(mm)
 labels qs10ser(mm)
 # 'qs10_m'+%string(mm)
 labels es05ser(mm)
 # 'es05_m'+%string(mm)
 labels es10ser(mm)
 # 'es10_m'+%string(mm)
 labels fzg05ser(mm)
 # 'fzg05_m'+%string(mm)
 labels fzg10ser(mm)
 # 'fzg10_m'+%string(mm)
end do mm
clear  fcser pct05ser pct10ser pct90ser pct95ser crpsser qwcrpsser qs05ser qs10ser es05ser es10ser fzg05ser fzg10ser

*** for reading in forecast draws
comp skipint = 5             ;* will thin 5k draws for each quantile by 1/5, to avoid RAM computational problems
comp ndraws2=ndraws/skipint
dis ndraws2 (ndraws2*nquant)
dec vec[ser] fcdraws(ndraws2*nquant) rawdraws(nquant)
clear(length=endvint) fcdraws
clear(length=ndraws) rawdraws

**** NBER dates, quarterly
clear(length=endvint,zeroes) cycle
set cycle 1948:4 1949:4 = 1.0
set cycle 1953:2 1954:2 = 1.0
set cycle 1957:3 1958:2 = 1.0
set cycle 1960:2 1961:1 = 1.0
set cycle 1969:4 1970:4 = 1.0
set cycle 1973:4 1975:1 = 1.0
set cycle 1980:1 1980:3 = 1.0
set cycle 1981:3 1982:4 = 1.0
set cycle 1990:3 1991:1 = 1.0
set cycle 2001:1 2001:4 = 1.0
set cycle 2007:4 2009:2 = 1.0
set cycle 2019:4 2020:2 = 1.0

********************************
******************************** now loop over time to read monthly predictors, determine data avail, set up model, and forecast
********************************

do time = stvint,endvint-1
 dis ''
 dis '*******************************************************'
 dis '******************************************************* forecast date = ' %datelabel(time)
 dis '*******************************************************'
 dis ''
 comp quarter = %month(time)
 comp originst = %cputime()

 do ww = 1,totweek
  smpl stsmpl time
  comp eststpt = stsmpl
  comp estendpt = time
  ****
  comp lastmoposs = ((ww-1)/4)
  comp nn = ww - lastmoposs*4
  comp yy = %year(time)
  comp month = lastmoposs+1+(quarter-1)*3
  comp mml = 1+(ww-1)/4     ;* month number in quarter (1-3 in quarter and 4 in next quarter)
  comp nn = ww - (mml-1)*4  ;* week number in month (1-4)
  if ww<=3
   comp horz = 2
  else
   comp horz = 1

  dis ""
  dis "****************************** totweek, month, week = " ww mml nn
  dis 'year, quarter, week = ' (%string(yy)+"qtr"+quarter+"wk"+ww)
  
  comp outfilename = basefilename+"draws."+%string(yy)+"qtr"+quarter+"wk"+ww+".csv"
  dis outfilename
  open data &outfilename
  data(org=col,for=cdf,skiplines=1,nolabels) 1 ndraws rawdraws
  close data
  
  *** estimate and forecast and store results
  do q = 1,nquant
    dis '******************************* quantile = ' pctiles(q)
    *
    sstats(mean) 1 ndraws rawdraws(q)(t)>>quantfc(q)
    comp qscorevec(q) = (actualGDP(time)-quantfc(q))*(pctiles(q)-%if(actualGDP(time)<=quantfc(q),1.,0.))
    if pctiles(q)==0.05
      comp directpct05(ww)(time) = quantfc(q)
    else if pctiles(q)==0.1
      comp directpct10(ww)(time) = quantfc(q)
    else if pctiles(q)==0.5
     comp directpct50(ww)(time) = quantfc(q)
    else if pctiles(q)==0.90
     comp directpct90(ww)(time) = quantfc(q)
    else if pctiles(q)==0.95
     comp directpct95(ww)(time) = quantfc(q)
    *
    do draws = skipint,ndraws,skipint
     comp drawcount = %idiv(draws,skipint)
     comp fcdraws((q-1)*ndraws2+drawcount)(time) = rawdraws(q)(draws)
    end do draws
  end do q

  *** results using empirical distribution of raw draws of quantile forecasts
  @FCMOMENTSkernelsmoothed(kernel=gaussian) fcdraws actualGDP time 1 pctiles meanres pctileres crps qwcrps qscoreres esres fzgres
  comp fcser(ww)(time) = meanres
  comp crpsser(ww)(time) = crps
  comp qwcrpsser(ww)(time) = qwcrps
  comp pct05ser(ww)(time) = pctileres(1)
  comp pct10ser(ww)(time) = pctileres(2)
  comp pct90ser(ww)(time) = pctileres(nquant-1)
  comp pct95ser(ww)(time) = pctileres(nquant)
  comp qs05ser(ww)(time) = qscoreres(1)
  comp qs10ser(ww)(time) = qscoreres(2)
  comp es05ser(ww)(time) = esres(1)
  comp es10ser(ww)(time) = esres(2)
  comp fzg05ser(ww)(time) = fzgres(1)
  comp fzg10ser(ww)(time) = fzgres(2)

 end do ww

 dis 'total time for this forecast origin = ' (%cputime()-originst)/60.
 
end do time

dis %dateandtime()
dis 'total estimation run time in minutes = ' (%cputime()-overallstart)/60.

********************************
******************************** tabulate and display results:  full sample, ending in 2019:4
********************************
smpl stvint endsmpl19
dis 'nobs = ' (endsmpl19-stvint+1)

do mm = 1,totweek
 dis ''
 dis '************************************** forecasts using ' mm ' weeks of data'
 dis ''
 set fcerror = actualgdp - fcser(mm){0}
 set fcerrorsq = fcerror^2.

 stats(noprint) fcerror      ;* don't need results from this, but it provides a useful check/verification of the sample we're using at each horizon
 if %nobs<>(endsmpl19-stvint+1)
  dis 'missing data at month ' mm

 sstats(mean) / fcerror>>meanerror
 sstats(mean) / fcerrorsq>>mse

 sstats(mean) / crpsser(mm)>>avgcrps
 sstats(mean) / qwcrpsser(mm)>>avgqwcrps

 sstats(mean) / (actualgdp{0}<=pct05ser(mm){0})>>covrate05
 sstats(mean) / (actualgdp{0}<=pct10ser(mm){0})>>covrate10
 
 sstats(mean) / qs05ser(mm)>>avgqs05
 sstats(mean) / qs10ser(mm)>>avgqs10

 sstats(mean) / fzg05ser(mm)>>avgfzg05
 sstats(mean) / fzg10ser(mm)>>avgfzg10

 dis 'mean error, rmse = ' @45 #####.### meanerror (mse^0.5)
 dis 'CRPS = ' @45 #####.### avgcrps
 dis 'qwCRPS = ' @45 #####.### avgqwcrps
 dis '5% coverage rate = ' @45 #####.### covrate05
 dis '10% coverage rate = ' @45 #####.### covrate10
 dis '5% quantile score = ' @45 #####.### avgqs05
 dis '10% quantile score = ' @45 #####.### avgqs10
 dis '5% FZG score = ' @45 #####.### avgfzg05
 dis '10% FZG score = ' @45 #####.### avgfzg10
 
end do mm

********************************
******************************** write time series of forecast, forecast error, and pred. likelihoods/scores to Excel file
********************************
comp filename = "fcresults_direct_"+filelabel(thispos)+".xls"
open copy &filename
copy(for=xls,dates,org=col) stvint endsmpl directpct50 directpct05 directpct10 directpct90 directpct95
close copy

comp filename = "fcresults_"+filelabel(thispos)+"_empiricalpctiles.xls"
open copy &filename
copy(for=xls,dates,org=col) stvint endsmpl fcser crpsser qwcrpsser pct05ser pct10ser pct90ser pct95ser qs05ser qs10ser es05ser es10ser fzg05ser fzg10ser
close copy

if stvint==1985:1
{
********************************
******************************** tabulate and display results:  2000-2019 sample
********************************
comp altstart = 2000:1
smpl altstart endsmpl19
dis 'nobs = ' (endsmpl19-altstart+1)

do mm = 1,totweek
 dis ''
 dis '************************************** forecasts using ' mm ' weeks of data'
 dis ''
 set fcerror = actualgdp - fcser(mm){0}
 set fcerrorsq = fcerror^2.

 stats(noprint) fcerror      ;* don't need results from this, but it provides a useful check/verification of the sample we're using at each horizon
 if %nobs<>(endsmpl19-altstart+1)
  dis 'missing data at month ' mm

 sstats(mean) / fcerror>>meanerror
 sstats(mean) / fcerrorsq>>mse

 sstats(mean) / crpsser(mm)>>avgcrps
 sstats(mean) / qwcrpsser(mm)>>avgqwcrps

 sstats(mean) / (actualgdp{0}<=pct05ser(mm){0})>>covrate05
 sstats(mean) / (actualgdp{0}<=pct10ser(mm){0})>>covrate10
 
 sstats(mean) / qs05ser(mm)>>avgqs05
 sstats(mean) / qs10ser(mm)>>avgqs10

 sstats(mean) / fzg05ser(mm)>>avgfzg05
 sstats(mean) / fzg10ser(mm)>>avgfzg10

 dis 'mean error, rmse = ' @45 #####.### meanerror (mse^0.5)
 dis 'CRPS = ' @45 #####.### avgcrps
 dis 'qwCRPS = ' @45 #####.### avgqwcrps
 dis '5% coverage rate = ' @45 #####.### covrate05
 dis '10% coverage rate = ' @45 #####.### covrate10
 dis '5% quantile score = ' @45 #####.### avgqs05
 dis '10% quantile score = ' @45 #####.### avgqs10
 dis '5% FZG score = ' @45 #####.### avgfzg05
 dis '10% FZG score = ' @45 #####.### avgfzg10
 
end do mm
}
endif

********************************
******************************** charting actual growth with upper and lower tails
********************************
smpl stvint endsmpl19

comp vf = 2
comp hf = 2
comp perpage = 4

comp [vec[string]] mykey = ||'actual','mean','10th-%ile','90th-%ile'||

do mm = 1,totweek,perpage
 grparm(bold) header 14
 spgraph(vfields=vf,hfields=hf,header=modeldesc)
 grparm(bold) header 22
 do j = mm,%imin(mm+perpage-1,totweek)
  comp header = "Forecasts using " + %string(j) + " weeks of data"
  graph(header=header,dates,key=below,klab=mykey,shading=cycle) 4
  # actualGDP / 1
  # fcser(j) / 2
  # pct10ser(j) / 4
  # pct90ser(j) / 4
 end do j
 spgraph(done)
end do mm
