/*
AUTHOR:   Todd Clark, Federal Reserve Bank of Cleveland, todd.clark@researchfed.org
Code associated with Andrea Carriero, Todd Clark and Massimiliano Marcellino, 
"Nowcasting Tail Risks to Economic Activity at a Weekly Frequency"

Program for reading in forecasts to compute accuracy measures reported in the paper's Table 7 and corresponding
appendix tables.  For the sample of 2020, this program omits tests.  It also replaces BMF-SV forecasts with
BMF-SVO forecasts that add an outlier treatment to the SV specification. 

Note re measures:  The program includes expected shortfall (ES) for the purpose of computing the VaR-ES score
for the average forecast. The VaR-ES score is denoted below with "FZG".  Results are produced for both
the paper's 10% tail and the 5% tail covered in the appendix.

Note re dating and samples:  Forecasts are dated by the forecast outcome date (current quarter, given nowcast setup)
*/

comp styr = 1947
cal(q) styr:1
comp stsmpl = styr:1	;*earliest period with data

comp stvint = 1985:1       ;* stvint is first vintage, which corresponds to the start of forecasting
comp endvint = 2021:1      ;* last quarter of data vintages considered
comp useactual = 1          ;* adjustment for obs lost of last vintage relative to eval sample end date (used to set this at 2)
comp endsmpl = endvint-useactual

comp stpt = 1985:1   ;* starting point for forecasts
comp svostart = 2019:1   ;* BMF-SVO forecasts do not start until 2019:Q1

all endvint
smpl stsmpl endvint

******************** actual GDP used as actuals
** read in GDP estimates used as actuals in evaluation of forecasts:  these are 2nd avail in the quarterly RTDSM
open data ../data/GDPactuals.secondrelease.xls
data(format=xls,org=col) / actualGDP
close
set gdp = actualGDP

**** NBER dates, quarterly
clear(length=endvint,zeroes) cycle
set cycle 1948:4 1949:4 = 1.0
set cycle 1953:2 1954:2 = 1.0
set cycle 1957:3 1958:2 = 1.0
set cycle 1960:2 1961:1 = 1.0
set cycle 1969:4 1970:4 = 1.0
set cycle 1973:4 1975:1 = 1.0
set cycle 1980:1 1980:3 = 1.0
set cycle 1981:3 1982:4 = 1.0
set cycle 1990:3 1991:1 = 1.0
set cycle 2001:1 2001:4 = 1.0
set cycle 2007:4 2009:2 = 1.0
set cycle 2019:4 2020:2 = 1.0

******************** dimensioning for reading in results
******************** note:  in the read loop, we'll overwrite SV forecasts for 2019-20 with SVO forecasts
comp [vec[int]] svindicator = ||1,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0||

comp [vec[str]] methlabel = ||"AR-SV",$
"base M:  BMF-SV",$
"base M:  BQR",$
"base M:  PQR",$
"base M-F:  BMF-SV",$
"base M-F:  BQR",$
"base M-F:  PQR",$
"base M + small weekly:  BMF-SV",$
"base M + small weekly:  BQR",$
"base M + small weekly:  PQR",$
"base M-F + small weekly:  BMF-SV",$
"base M-F + small weekly:  BQR",$
"base M-F + small weekly:  PQR",$
"base M + large weekly:  BMF-SV",$
"base M + large weekly:  BQR",$
"base M + large weekly:  PQR",$
"base M-F + large weekly:  BMF-SV",$
"base M-F + large weekly:  BQR",$
"base M-F + large weekly:  PQR"||

comp [vec[str]] filestouse = ||"out.forecastruns/AR_SV",$
"out.forecastruns/basemacro_SV",$
"out.fcprocessingfromdraws/fcresults_BQR_M_empiricalpctiles",$
"out.fcprocessingfromdraws/fcresults_PQR_M_empiricalpctiles",$
"out.forecastruns/basemacro_basefinance_SV",$
"out.fcprocessingfromdraws/fcresults_BQR_MF_empiricalpctiles",$
"out.fcprocessingfromdraws/fcresults_PQR_MF_empiricalpctiles",$
"out.forecastruns/basemacro_smallwkly_SV",$
"out.fcprocessingfromdraws/fcresults_BQR_M_smallwkly_empiricalpctiles",$
"out.fcprocessingfromdraws/fcresults_PQR_M_smallwkly_empiricalpctiles",$
"out.forecastruns/basemacro_basefinance_smallwkly_SV",$
"out.fcprocessingfromdraws/fcresults_BQR_MF_smallwkly_empiricalpctiles",$
"out.fcprocessingfromdraws/fcresults_PQR_MF_smallwkly_empiricalpctiles",$
"out.forecastruns/basemacro_largewkly_SV",$
"out.fcprocessingfromdraws/fcresults_BQR_M_largewkly_empiricalpctiles",$
"out.fcprocessingfromdraws/fcresults_PQR_M_largewkly_empiricalpctiles",$
"out.forecastruns/basemacro_basefinance_largewkly_SV",$
"out.fcprocessingfromdraws/fcresults_BQR_MF_largewkly_empiricalpctiles",$
"out.fcprocessingfromdraws/fcresults_PQR_MF_largewkly_empiricalpctiles"||
comp nmeth = %rows(filestouse)
if %rows(filestouse)<>nmeth
 dis "mismatch in files listed versus count of labels specified"
endif

******************** setting up average forecasts
******************** note:  some of these indicators are used in creating charts
******************** we deliberately put the BMF-SV last to make it easier to form an FZG avg result
open data comboweights.xlsx
data(org=col,left=3,for=xlsx) 1 nmeth basemf
close data

make avgwts 1 nmeth
# basemf
comp [vec[str]] avgmethlabel = ||"avg.\ base M-F"||

comp navg = %cols(avgwts)       ;* number of average forecasts considered
comp totmeth = nmeth + navg

comp [vec] sumw = %sumc(avgwts)  ;* should be %sumr, but it us %sumc applied here that works right
ewise avgwts(i,j) = avgwts(i,j)/sumw(j)
dis ###.#### avgwts

comp [vec[str]] methlabel = methlabel+avgmethlabel

******************** setting up storage
comp totweek = 15

*** what we use to read in, method by method
dec vec[ser] fcseries(totweek) pct05series(totweek) pct10series(totweek) pct90series(totweek) pct95series(totweek) qs05series(totweek) qs10series(totweek) $ 
 fzg05series(totweek) fzg10series(totweek) es05series(totweek) es10series(totweek) crpsseries(totweek) qwcrpsseries(totweek)
do mm = 1,totweek 
 labels fcseries(mm)
 # 'fcseries_m'+%string(mm)
 labels pct05series(mm)
 # 'pct05_m'+%string(mm)
 labels pct10series(mm)
 # 'pct10_m'+%string(mm)
 labels pct90series(mm)
 # 'pct90_m'+%string(mm)
 labels pct95series(mm)
 # 'pct95_m'+%string(mm)
 labels qs05series(mm)
 # 'qs05_m'+%string(mm)
 labels qs10series(mm)
 # 'qs10_m'+%string(mm)
 labels fzg05series(mm)
 # 'fzg05_m'+%string(mm)
 labels fzg10series(mm)
 # 'fzg10_m'+%string(mm)
 labels es05series(mm)
 # 'es05_m'+%string(mm)
 labels es10series(mm)
 # 'es10_m'+%string(mm)
 labels crpsseries(mm)
 # 'crps_m'+%string(mm)
 labels qwcrpsseries(mm)
 # 'qwcrps_m'+%string(mm)
end do mm
clear fcseries pct05series pct10series pct90series pct95series qs05series qs10series fzg05series fzg10series es05series es10series crpsseries qwcrpsseries pct05

*** what we use to store, covering all methods
*** note that, in the hit05 and hit10 series, we define as empirical rate less nominal, rather than just the empirical rate 
dec rec[ser] sqerror(totmeth,totweek) hit05(totmeth,totweek) hit10(totmeth,totweek) pointfc(totmeth,totweek) q05fc(totmeth,totweek) q10fc(totmeth,totweek) q90fc(totmeth,totweek) q95fc(totmeth,totweek) $
 qs05(totmeth,totweek) qs10(totmeth,totweek) fzg05(totmeth,totweek) fzg10(totmeth,totweek) es05(totmeth,totweek) es10(totmeth,totweek) crps(totmeth,totweek) qwcrps(totmeth,totweek)

*** stuff for FZG score computation of average forecast, using quantiles of source base MF forecasts
comp fzgacoef = 4.0   ;* coef used in G_2 function of FZG score
comp fzgbcoef = 6.0   ;* intercept added to FZG score

******************** reading in forecasts from BMF, BQR, and PQR methods for each variable set
******************** Note that we first read in a full history of BMF-SV forecasts and then read in SVO forecasts for
******************** 2019-20 and overwrite the SV forecasts for these years
do nn = 1,%rows(filestouse)
 comp [str] filename = filestouse(nn)+".xls"
 dis filename
 open data &filename
 data(org=col,for=xls) / fcseries pct05series pct10series pct90series pct95series qs05series qs10series fzg05series fzg10series es05series es10series crpsseries qwcrpsseries
 close data
 if svindicator(nn)==1
 {
  comp [str] filename = filestouse(nn)+"O.xls"
  dis filename
  open data &filename
  data(org=col,for=xls) svostart endvint fcseries pct05series pct10series pct90series pct95series qs05series qs10series fzg05series fzg10series es05series es10series crpsseries qwcrpsseries
  close data
 }

 do mm = 1,totweek
  set pointfc(nn,mm) = fcseries(mm){0}
  set q05fc(nn,mm) = pct05series(mm){0} 
  set q10fc(nn,mm) = pct10series(mm){0} 
  set q90fc(nn,mm) = pct90series(mm){0} 
  set q95fc(nn,mm) = pct95series(mm){0} 
  set qs05(nn,mm) = qs05series(mm){0} 
  set qs10(nn,mm) = qs10series(mm){0}
  set fzg05(nn,mm) = fzg05series(mm){0} 
  set fzg10(nn,mm) = fzg10series(mm){0}
  set es05(nn,mm) = es05series(mm){0} 
  set es10(nn,mm) = es10series(mm){0}
  set crps(nn,mm) = crpsseries(mm){0}
  set qwcrps(nn,mm) = qwcrpsseries(mm){0}
  *
  set sqerror(nn,mm) = (gdp - fcseries(mm){0})^2.
  set hit05(nn,mm) = %if(gdp{0}<=pct05series(mm){0},1.0,0.0) - 0.05
  set hit10(nn,mm) = %if(gdp{0}<=pct10series(mm){0},1.0,0.0) - 0.10
 end do mm

end do nn

** figure out common start point of sample
do i = 1,nmeth
 inquire(series=q05fc(i,1)) tempstpt tempendpt
 comp stpt = %imax(stpt,tempstpt)
 comp endsmpl = %imin(endsmpl,tempendpt)
end do i
dis %datelabel(stpt) %datelabel(endsmpl)

******************** forming average forecasts:  point and quantile
******************** note:  For point and quantile forecasts and ES, we take a simple average of the forecasts.  The average 
******************** quantile and ES forecasts are then used to compute the FZG score.
******************** for the CRPS and qwCRPS results, we compute them from the linear pooled combination density from BMF, BQR, and PQR
** first compute combos of point and quantile fcs
do ii = 1,navg
 do mm = 1,totweek
  clear(zeros) pointfc(nmeth+ii,mm) q05fc(nmeth+ii,mm) q10fc(nmeth+ii,mm) q90fc(nmeth+ii,mm) q95fc(nmeth+ii,mm) es05(nmeth+ii,mm) es10(nmeth+ii,mm) 
  do nn = 1,nmeth
   set pointfc(nmeth+ii,mm) = pointfc(nmeth+ii,mm){0} + avgwts(nn,ii)*pointfc(nn,mm){0}
   set q05fc(nmeth+ii,mm) = q05fc(nmeth+ii,mm){0} + avgwts(nn,ii)*q05fc(nn,mm){0} 
   set q10fc(nmeth+ii,mm) = q10fc(nmeth+ii,mm){0} + avgwts(nn,ii)*q10fc(nn,mm){0} 
   set q90fc(nmeth+ii,mm) = q90fc(nmeth+ii,mm){0} + avgwts(nn,ii)*q90fc(nn,mm){0} 
   set q95fc(nmeth+ii,mm) = q95fc(nmeth+ii,mm){0} + avgwts(nn,ii)*q95fc(nn,mm){0} 
   set es05(nmeth+ii,mm) = es05(nmeth+ii,mm){0} + avgwts(nn,ii)*es05(nn,mm){0} 
   set es10(nmeth+ii,mm) = es10(nmeth+ii,mm){0} + avgwts(nn,ii)*es10(nn,mm){0} 
  end do nn
 end do mm
end do ii

** now compute squared errors, hit rates, quantile scores, and FZG scores for combo fcs
do ii = 1,navg
 do mm = 1,totweek
  set sqerror(nmeth+ii,mm) = (gdp - pointfc(nmeth+ii,mm){0})^2.
  set hit05(nmeth+ii,mm) = %if(gdp{0}<=q05fc(nmeth+ii,mm){0},1.0,0.0) - 0.05
  set hit10(nmeth+ii,mm) = %if(gdp{0}<=q10fc(nmeth+ii,mm){0},1.0,0.0) - 0.10
  *
  comp useq = 0.05
  set qs05(nmeth+ii,mm) = (GDP{0}-q05fc(nmeth+ii,mm){0})*(useq-%if(GDP{0}<=q05fc(nmeth+ii,mm){0},1.,0.))
  set adjhitrate = hit05(nmeth+ii,mm){0}+useq
  set fzg05(nmeth+ii,mm) = fzgbcoef + (adjhitrate{0}-useq)*(q05fc(nmeth+ii,mm){0}-gdp{0}) + $
    (1./useq)*exp(es05(nmeth+ii,mm)/fzgacoef)*(adjhitrate{0}*(q05fc(nmeth+ii,mm){0}-gdp{0})+useq*(es05(nmeth+ii,mm)-q05fc(nmeth+ii,mm){0}-fzgacoef))
  *
  comp useq = 0.10
  set qs10(nmeth+ii,mm) = (GDP{0}-q10fc(nmeth+ii,mm){0})*(useq-%if(GDP{0}<=q10fc(nmeth+ii,mm){0},1.,0.))
  set adjhitrate = hit10(nmeth+ii,mm){0}+useq
  set fzg10(nmeth+ii,mm) = fzgbcoef + (adjhitrate{0}-useq)*(q10fc(nmeth+ii,mm){0}-gdp{0}) + $
    (1./useq)*exp(es10(nmeth+ii,mm)/fzgacoef)*(adjhitrate{0}*(q10fc(nmeth+ii,mm){0}-gdp{0})+useq*(es10(nmeth+ii,mm)-q10fc(nmeth+ii,mm){0}-fzgacoef))
 end do mm
end do ii

** now read in CRPS for average forecast:  these results come from linear pooling of densities for BMF, BQR, PQR
comp nn = nmeth+1
comp [str] filename = "out.fcprocessingfromdraws/averageforecast_SVO_1920_empiricaldensity.xls"
open data &filename
data(org=col,for=xls) / crpsseries qwcrpsseries
close data
do mm = 1,totweek
 set crps(nn,mm) = crpsseries(mm){0}
 set qwcrps(nn,mm) = qwcrpsseries(mm){0}
end do mm

******************** computing statistics
*** reset sample to 2020
comp stpt = 2020:1
comp endsmpl = endsmpl  ;* shortening sample to drop pandemic period
comp [rec[int]] evalranges = ||stpt,endsmpl||
comp nsamples = %rows(evalranges)
comp basepos = 1  ;* which method will be the baseline, which is base macro variable set, QR (avg) here
dis methlabel(basepos)

dec vec[rec] rmseres(nsamples) cover05(nsamples) cover10(nsamples) qscore05res(nsamples) qscore10res(nsamples) $
 msetstat(nsamples) cover05tstat(nsamples) qscore05tstat(nsamples) cover10tstat(nsamples) qscore10tstat(nsamples)  $
 msepvalue(nsamples) cover05pvalue(nsamples) qscore05pvalue(nsamples) cover10pvalue(nsamples) qscore10pvalue(nsamples)  $
 msepindic(nsamples) cover05pindic(nsamples) qscore05pindic(nsamples) cover10pindic(nsamples) qscore10pindic(nsamples) $
 rawrmseres(nsamples) rawqscore05res(nsamples) rawqscore10res(nsamples) $
 mseboth(nsamples) cover05both(nsamples) cover10both(nsamples) qscore05both(nsamples) qscore10both(nsamples) $
 fzg05res(nsamples) fzg10res(nsamples) fzg05tstat(nsamples) fzg10tstat(nsamples) fzg05pvalue(nsamples) fzg10pvalue(nsamples) $
 fzg05pindic(nsamples) fzg10pindic(nsamples) fzg05both(nsamples) fzg10both(nsamples) rawfzg05res(nsamples) rawfzg10res(nsamples) $
 condcov1(nsamples) condcov2(nsamples) condcov3(nsamples) christ10pindic(nsamples) christ10both(nsamples) $
 crpsres(nsamples) crpststat(nsamples) crpspvalue(nsamples) crpspindic(nsamples) rawcrpsres(nsamples) crpsboth(nsamples) $
 qwcrpsres(nsamples) qwcrpststat(nsamples) qwcrpspvalue(nsamples) qwcrpspindic(nsamples) rawqwcrpsres(nsamples) qwcrpsboth(nsamples) 
 
do i = 1,nsamples
 dim  rmseres(i)(totmeth,totweek) cover05(i)(totmeth,totweek) cover10(i)(totmeth,totweek) qscore05res(i)(totmeth,totweek) qscore10res(i)(totmeth,totweek) $
 msetstat(i)(totmeth,totweek) cover05tstat(i)(totmeth,totweek) qscore05tstat(i)(totmeth,totweek) cover10tstat(i)(totmeth,totweek) qscore10tstat(i)(totmeth,totweek)  $
 msepvalue(i)(totmeth,totweek) cover05pvalue(i)(totmeth,totweek) qscore05pvalue(i)(totmeth,totweek) cover10pvalue(i)(totmeth,totweek) qscore10pvalue(i)(totmeth,totweek)  $
 msepindic(i)(totmeth,totweek) cover05pindic(i)(totmeth,totweek) qscore05pindic(i)(totmeth,totweek) cover10pindic(i)(totmeth,totweek) qscore10pindic(i)(totmeth,totweek) $
 fzg05res(i)(totmeth,totweek) fzg10res(i)(totmeth,totweek) fzg05tstat(i)(totmeth,totweek) fzg10tstat(i)(totmeth,totweek) fzg05pvalue(i)(totmeth,totweek) fzg10pvalue(i)(totmeth,totweek) $
 fzg05pindic(i)(totmeth,totweek) fzg10pindic(i)(totmeth,totweek) condcov1(i)(totmeth,totweek) condcov2(i)(totmeth,totweek) condcov3(i)(totmeth,totweek) christ10pindic(i)(totmeth,totweek) $
 crpsres(i)(totmeth,totweek) crpststat(i)(totmeth,totweek) crpspvalue(i)(totmeth,totweek) crpspindic(i)(totmeth,totweek) rawcrpsres(i)(totmeth,totweek)  $
 qwcrpsres(i)(totmeth,totweek) qwcrpststat(i)(totmeth,totweek) qwcrpspvalue(i)(totmeth,totweek) qwcrpspindic(i)(totmeth,totweek) rawqwcrpsres(i)(totmeth,totweek)  
end do i
dec rec both(totmeth,2*totweek)  ;* for storing relative performance measures and p-value indicators together in one table

do nn = 1,totmeth
 do mm = 1,totweek
  do n = 1,nsamples
    if nn==1
     dis 'forecast origins, for sample ' n +':  ' %datelabel(evalranges(n,1))  %datelabel(evalranges(n,2))
    *
    dis "nn = " methlabel(nn)
    dis "week = " mm
    
    sstats(mean) evalranges(n,1) evalranges(n,2) sqerror(nn,mm){0}>>mse
    comp rmseres(n)(nn,mm) = mse^0.5
    sstats(mean) evalranges(n,1) evalranges(n,2) hit05(nn,mm){0}>>cover05(n)(nn,mm)
    sstats(mean) evalranges(n,1) evalranges(n,2) hit10(nn,mm){0}>>cover10(n)(nn,mm)
    sstats(mean) evalranges(n,1) evalranges(n,2) qs05(nn,mm){0}>>qscore05res(n)(nn,mm)
    sstats(mean) evalranges(n,1) evalranges(n,2) qs10(nn,mm){0}>>qscore10res(n)(nn,mm)
    sstats(mean) evalranges(n,1) evalranges(n,2) fzg05(nn,mm){0}>>fzg05res(n)(nn,mm)
    sstats(mean) evalranges(n,1) evalranges(n,2) fzg10(nn,mm){0}>>fzg10res(n)(nn,mm)
    sstats(mean) evalranges(n,1) evalranges(n,2) crps(nn,mm){0}>>crpsres(n)(nn,mm)
    sstats(mean) evalranges(n,1) evalranges(n,2) qwcrps(nn,mm){0}>>qwcrpsres(n)(nn,mm)
  end do n
 end do hh
end do nn

do n = 1,nsamples
 ** create copies to store raw rather than relative results
 comp rawrmseres(n) = rmseres(n)
 comp rawqscore05res(n) = qscore05res(n)
 comp rawqscore10res(n) = qscore10res(n) 
 comp rawfzg05res(n) = fzg05res(n)
 comp rawfzg10res(n) = fzg10res(n) 
 comp rawcrpsres(n) = crpsres(n)
 comp rawqwcrpsres(n) = qwcrpsres(n)

 ** adjust raw coverage tables to show empirical rates and not gaps relative to 0.05 nominal and .10 nominal
 ewise cover05(n)(i,j) = cover05(n)(i,j) + 0.05
 ewise cover10(n)(i,j) = cover10(n)(i,j) + 0.10

 ** now convert to relative results
 ewise rmseres(n)(i,j) = %if(i==basepos,rmseres(n)(i,j),rmseres(n)(i,j)/rmseres(n)(basepos,j))
 ewise qscore05res(n)(i,j) = %if(i==basepos,qscore05res(n)(i,j),qscore05res(n)(i,j)/qscore05res(n)(basepos,j))
 ewise qscore10res(n)(i,j) = %if(i==basepos,qscore10res(n)(i,j),qscore10res(n)(i,j)/qscore10res(n)(basepos,j))
 ewise fzg05res(n)(i,j) = %if(i==basepos,fzg05res(n)(i,j),fzg05res(n)(basepos,j)-fzg05res(n)(i,j))
 ewise fzg10res(n)(i,j) = %if(i==basepos,fzg10res(n)(i,j),fzg10res(n)(basepos,j)-fzg10res(n)(i,j))
 ewise crpsres(n)(i,j) = %if(i==basepos,crpsres(n)(i,j),crpsres(n)(i,j)/crpsres(n)(basepos,j))
 ewise qwcrpsres(n)(i,j) = %if(i==basepos,qwcrpsres(n)(i,j),qwcrpsres(n)(i,j)/qwcrpsres(n)(basepos,j))

end do n

******************** display results:  raw results (not relatives)

do n = 1,nsamples
 if n==1
  {
   dis ''
   dis '**************   10% quantile scores     ****************'
  }
 dis ''
 dis 'sample: ' %datelabel(evalranges(n,1))  %datelabel(evalranges(n,2))
 dis ''
 do i = 1,totmeth
  comp [vec] resultsvec = %xrow(rawqscore10res(n),i)
  dis methlabel(i) @56 ###.### resultsvec
 end do i
end do n

do n = 1,nsamples
 if n==1
  {
   dis ''
   dis '**************   10% FZG scores     ****************'
  }
 dis ''
 dis 'sample: ' %datelabel(evalranges(n,1))  %datelabel(evalranges(n,2))
 dis ''
 do i = 1,totmeth
  comp [vec] resultsvec = %xrow(rawfzg10res(n),i)
  dis methlabel(i) @56 ###.### resultsvec
 end do i
end do n

do n = 1,nsamples
 if n==1
  {
   dis ''
   dis '**************   10% coverage rates     ****************'
  }
 dis ''
 dis 'sample: ' %datelabel(evalranges(n,1))  %datelabel(evalranges(n,2))
 dis ''
 do i = 1,totmeth
  comp [vec] resultsvec = %xrow(cover10(n),i)
  dis methlabel(i) @56 ###.### resultsvec
 end do i
end do n

do n = 1,nsamples
 if n==1
  {
   dis ''
   dis '**************   CRPS     ****************'
  }
 dis ''
 dis 'sample: ' %datelabel(evalranges(n,1))  %datelabel(evalranges(n,2))
 dis ''
 do i = 1,totmeth
  comp [vec] resultsvec = %xrow(rawcrpsres(n),i)
  dis methlabel(i) @56 ###.### resultsvec
 end do i
end do n

do n = 1,nsamples
 if n==1
  {
   dis ''
   dis '**************   qwCRPS     ****************'
  }
 dis ''
 dis 'sample: ' %datelabel(evalranges(n,1))  %datelabel(evalranges(n,2))
 dis ''
 do i = 1,totmeth
  comp [vec] resultsvec = %xrow(rawqwcrpsres(n),i)
  dis methlabel(i) @56 ###.### resultsvec
 end do i
end do n

do n = 1,nsamples
 if n==1
  {
   dis ''
   dis '**************   5% quantile scores     ****************'
  }
 dis ''
 dis 'sample: ' %datelabel(evalranges(n,1))  %datelabel(evalranges(n,2))
 dis ''
 do i = 1,totmeth
  comp [vec] resultsvec = %xrow(rawqscore05res(n),i)
  dis methlabel(i) @56 ###.### resultsvec
 end do i
end do n

do n = 1,nsamples
 if n==1
  {
   dis ''
   dis '**************   5% FZG scores     ****************'
  }
 dis ''
 dis 'sample: ' %datelabel(evalranges(n,1))  %datelabel(evalranges(n,2))
 dis ''
 do i = 1,totmeth
  comp [vec] resultsvec = %xrow(rawfzg05res(n),i)
  dis methlabel(i) @56 ###.### resultsvec
 end do i
end do n

do n = 1,nsamples
 if n==1
  {
   dis ''
   dis '**************   5% coverage rates     ****************'
  }
 dis ''
 dis 'sample: ' %datelabel(evalranges(n,1))  %datelabel(evalranges(n,2))
 dis ''
 do i = 1,totmeth
  comp [vec] resultsvec = %xrow(cover05(n),i)
  dis methlabel(i) @56 ###.### resultsvec
 end do i
end do n

do n = 1,nsamples
 if n==1
  {
   dis ''
   dis '**************   RMSEs     ****************'
  }
 dis ''
 dis 'sample: ' %datelabel(evalranges(n,1))  %datelabel(evalranges(n,2))
 dis ''
 do i = 1,totmeth
  comp [vec] resultsvec = %xrow(rawrmseres(n),i)
  dis methlabel(i) @56 ###.### resultsvec
 end do i
end do n

******************** display results:  relative results

do n = 1,nsamples
 if n==1
  {
   dis ''
   dis '**************   CRPS     ****************'
  }
 dis ''
 dis 'sample: ' %datelabel(evalranges(n,1))  %datelabel(evalranges(n,2))
 dis ''
 do i = 1,totmeth
  comp [vec] resultsvec = %xrow(crpsres(n),i)
  dis methlabel(i) @56 ####.## resultsvec
 end do i
end do n

do n = 1,nsamples
 if n==1
  {
   dis ''
   dis '**************   10% quantile scores     ****************'
  }
 dis ''
 dis 'sample: ' %datelabel(evalranges(n,1))  %datelabel(evalranges(n,2))
 dis ''
 do i = 1,totmeth
  comp [vec] resultsvec = %xrow(qscore10res(n),i)
  dis methlabel(i) @56 ####.## resultsvec
 end do i
end do n

do n = 1,nsamples
 if n==1
  {
   dis ''
   dis '**************   10% FZG scores     ****************'
  }
 dis ''
 dis 'sample: ' %datelabel(evalranges(n,1))  %datelabel(evalranges(n,2))
 dis ''
 do i = 1,totmeth
  comp [vec] resultsvec = %xrow(fzg10res(n),i)
  dis methlabel(i) @56 #####.# resultsvec
 end do i
end do n

do n = 1,nsamples
 if n==1
  {
   dis ''
   dis '**************   qwcrps     ****************'
  }
 dis ''
 dis 'sample: ' %datelabel(evalranges(n,1))  %datelabel(evalranges(n,2))
 dis ''
 do i = 1,totmeth
  comp [vec] resultsvec = %xrow(qwcrpsres(n),i)
  dis methlabel(i) @56 ####.## resultsvec
 end do i
end do n

do n = 1,nsamples
 if n==1
  {
   dis ''
   dis '**************   5% quantile scores     ****************'
  }
 dis ''
 dis 'sample: ' %datelabel(evalranges(n,1))  %datelabel(evalranges(n,2))
 dis ''
 do i = 1,totmeth
  comp [vec] resultsvec = %xrow(qscore05res(n),i)
  dis methlabel(i) @56 ####.## resultsvec
 end do i
end do n

do n = 1,nsamples
 if n==1
  {
   dis ''
   dis '**************   5% FZG scores     ****************'
  }
 dis ''
 dis 'sample: ' %datelabel(evalranges(n,1))  %datelabel(evalranges(n,2))
 dis ''
 do i = 1,totmeth
  comp [vec] resultsvec = %xrow(fzg05res(n),i)
  dis methlabel(i) @56 #####.# resultsvec
 end do i
end do n

do n = 1,nsamples
 if n==1
  {
   dis ''
   dis '**************   RMSEs     ****************'
  }
 dis ''
 dis 'sample: ' %datelabel(evalranges(n,1))  %datelabel(evalranges(n,2))
 dis ''
 do i = 1,totmeth
  comp [vec] resultsvec = %xrow(rmseres(n),i)
  dis methlabel(i) @56 ####.## resultsvec
 end do i
end do n
