#!/usr/bin/env python3
# Author(s): Nurzhan Sapargali <sapargalin95@gmail.com>
#            Michael E. Rose <michael.ernst.rose@gmail.com>
"""Plots local network of Joseph E. Stiglitz in 1990s"""

import networkx as nx
import matplotlib.pyplot as plt
from pybliometrics.scopus import AuthorRetrieval

ECONLIT_FOLDER = './100_EconLit_networks/'
SCOPUS_FOLDER = './200_Scopus_networks/'
OUTPUT_FOLDER = './990_output/'


def main():
    folders = (ECONLIT_FOLDER, SCOPUS_FOLDER)
    samples = ("Econ", "Scopus")
    keys = ("stiglitz je", "57208457063")
    for key, label, folder in zip(keys, samples, folders):
        G = nx.read_gexf(folder + r'1990.gexf')
        # Create subgraph with all nodes within distance of 1 and 2
        coauth_1 = [au for au in G.neighbors(key)]
        coauth_2 = [au for coauthor in coauth_1 for au in G.neighbors(coauthor)]
        H = G.subgraph(coauth_1 + coauth_2)
        # Label nodes of Stiglitz and co-authors within distance 1
        if key.isnumeric():
            labels = {au: AuthorRetrieval(au).indexed_name for au in coauth_1}
            labels[key] = AuthorRetrieval(key).indexed_name
        else:
            labels = {c: c for c in coauth_1}
            labels[key] = key
        # Draw network
        pos = nx.spring_layout(H, k=0.35)
        nx.draw(H, with_labels=False, pos=pos, node_size=180,
                node_color='#b0b0b0')
        # Draw labels above nodes
        for k in pos:
            pos[k][1] = 0.075 + pos[k][1]
        nx.draw_networkx_labels(H, labels=labels, pos=pos)
        # Write out
        figname = f'{OUTPUT_FOLDER}/Figures/{label}_2.pdf'
        plt.savefig(figname, bbox_inches='tight')
        plt.close()


if __name__ == '__main__':
    main()
