% Michael T. Owyang, Jeremy Piger, and Daniel Soques, 
% "Contagious Switching", Journal of Applied Econometrics, forthcoming

% This code runs all the specifications from the paper and saves the
% results as mat files. To get the tables and figures, run
% "OPS_Tables_Figures.m" after running this file.

% Note: This code uses the following MATLAB toolboxes:
% "Robust Control"
% "Optimization"
% "Statistics and Machine Learning"

clear all; clc;

% Set working directory
cd(fileparts(matlab.desktop.editor.getActiveFilename))
addpath(genpath(fileparts(matlab.desktop.editor.getActiveFilename)))

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Model Options
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Data filename
model.loadfile = 'OPS_Data.xlsx';  

% Set seed
rand('state', 1234) ;
randn('state', 5678);

% MCMC parameters
model.Nburn = 20000;                  % Number of Gibbs Iterations to Toss
model.Nkeep = 10000;                  % Number of Gibbs Iterations to Keep
model.lags = 1;                     % Number of Lags in the VAR
model.disp = 1;                     % Display iterations in command window? (Yes = 1; No = 0)
model.HPI = 68;                     % Interval for Posterior Coverage
model.IRF_length = 16;              % Length of GIRF
model.rho0 = 1;                     % Set rho in GDP equation = 0 (Yes = 1; No = 0)
model.Mex_break = 0;                % Allow for break in Mexico switching dynamics? (Yes = 1; No = 0)
model.start_1980 = 1;               % Start sample in 1980?(Yes = 1; No = 0)
model.EPUI = 0;                     % Include EPUI in VAR? (Yes = 1; No = 0)
model.Smin = 0.10;                  % Minimum portion of the sample in each regime

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Estimate model for full sample when S observed
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
model.NAFTA = 0;                    % NAFTA sample? (0 = All Data; 1 = pre-NAFTA; 2 = post-NAFTA)
model.S_known = 1;                  % Regime variable known? (Yes = 1; No = 0)

% Gibbs Sampler
[priors,data,results,post,means] = OPS_mcmc(model);

% Compute Marginal Effects
[MEall,girfz1,girfz0,s1,s0] = OPS_MargEff(priors,data,results);
post.ME = squeeze(prctile(MEall,[50-priors.HPI/2,50,50 + priors.HPI/2],4)); % Dimensions: Response Variable, Horizon, Shocked Variable, Percentiles

% Save
filename = ['OPS',sprintf('_Sknown%d',priors.S_known),sprintf('_NAFTA%d',priors.NAFTA)];
save(filename,'-v7.3')

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Estimate model for full sample when S unobserved
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
model.NAFTA = 0;
model.S_known = 0;

% Gibbs Sampler
[priors,data,results,post,means] = OPS_mcmc(model);

% Save
filename = ['OPS',sprintf('_Sknown%d',priors.S_known),sprintf('_NAFTA%d',priors.NAFTA)];
save(filename,'-v7.3')

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Estimate model for each subsample (Pre- and Post-NAFTA) for S unobserved
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
model.S_known = 0;

% Pre-NAFTA
model.NAFTA = 1;
[priors,data,results,post,means] = OPS_mcmc(model);

[MEpre] = OPS_MargEff(priors,data,results);
post.MEpre = squeeze(prctile(MEpre,[50-priors.HPI/2,50,50 + priors.HPI/2],4)); % Dimensions: Response Variable, Horizon, Shocked Variable, Percentiles

filename = ['OPS',sprintf('_Sknown%d',priors.S_known),sprintf('_NAFTA%d',priors.NAFTA)];
save(filename,'-v7.3')

% Post-NAFTA
model.NAFTA = 2;
[priors,data,results,post,means] = OPS_mcmc(model);

[MEpost] = OPS_MargEff(priors,data,results);
post.MEpost = squeeze(prctile(MEpost,[50-priors.HPI/2,50,50 + priors.HPI/2],4)); % Dimensions: Response Variable, Horizon, Shocked Variable, Percentiles

filename = ['OPS',sprintf('_Sknown%d',priors.S_known),sprintf('_NAFTA%d',priors.NAFTA)];
save(filename,'-v7.3')






