function [z] = OPS_z(draw,priors,data)
%Output:
%   z: (TxN)
% Observation equation: y_t = H*z_t + e_t
% Measurement equation: z_t = M_t + F*z_tm1 + v_t
% e_t ~ N(0,R)
% v_t ~ N(0,Q)

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Define the model
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

Y = data.y;
K = (priors.N + priors.J) * priors.P; % dimension of state vector
K_y = size(Y,1); % diminesion of observation vector
T = size(Y,2);

% H: (K_y x K) loadings
H = zeros(K_y,K);
H(1:+ priors.J,priors.N + 1:priors.N + priors.J) = eye(priors.J);

% M: (K x T) time-varying intercept in the measurement equation
Msmall = zeros(K,1);
Msmall(1:priors.N+priors.J,1) = draw.C;
M = repmat(Msmall,1,T); % Need to add X part

% F: (K x K) AR coefficient matrix in measurement equation
F = zeros(K,K);
F(1:priors.N+priors.J,:) = draw.B;
for p = 1:priors.P-1
    x_start = (priors.N + priors.J) * p + 1;
    x_stop = (priors.N + priors.J) * (p + 1);
    y_start = (priors.N + priors.J) * (p-1) + 1;
    y_stop = (priors.N + priors.J) * p;
    F(x_start:x_stop,y_start:y_stop) = eye(priors.N + priors.J);
end

R = zeros(priors.J,priors.J);

% Q: (K x K) covariance matrix for v_t
Q = zeros(K,K);
Q(1:priors.N+priors.J,1:priors.N+priors.J) = draw.omega;

% Initial Values based on unconditional mean
C0 =  M(:,1);
z_0_0 = (eye(K)-F)\C0;
P_0_0_vec = (eye(size(Q(:),1)) - kron(F,F))\Q(:);
P_0_0 = reshape(P_0_0_vec,K,K);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Forward Kalman Filter
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

[z_t_t,P_t_t] = OPS_Kalman(Y,H,M,F,R,Q,z_0_0,P_0_0);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Backwards Smoother & Draws
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Q is singular (with the first J x J block postive definite)
J = priors.N;
z = zeros(J,T);
% Draw state vector for t=T
z_t_T = z_t_t(:,end);
P_t_T = P_t_t(:,:,end);

lbnd = [-Inf; 1e-10]; %lower bounds for each regime
ubnd = [0; Inf]; %upper bounds for each regime
lbndT = lbnd(data.S+1);
ubndT = ubnd(data.S+1);
% Rejection sampler to ensure z matches S
accept = 0; wait = 0;
while accept == 0
    z_full = z_t_T(1:J,1) + chol(P_t_T(1:J,1:J))' * randn(J,1); % Alternative: z_full = mvnrnd(z_t_T(1:J,1), P_t_T(1:J,1:J))';
    z(:,end) = z_full(1:J,1);
    S_hat = double(z(:,end) > 0);
    if all(S_hat == data.S(:,end)) == 1
        accept = 1;
    else
        wait = wait + 1;
    end
    if wait > 1e4
        z(:,end) = z_t_T(1:J,1) + mvrandn(lbndT(:,end) - z_t_T(1:J,1),ubndT(:,end) - z_t_T(1:J,1),P_t_T(1:J,1:J),1);
        accept = 1;
        %         error('Draw for z not matching S')
    end
end
% Truncate the parameter matrices
Qstar = Q(1:J,1:J);
Fstar = F(1:J,:);
% Draw state vectors for t = T-1,...,1
for t = T-1:-1:1
    fstar = Fstar * P_t_t(:,:,t) * Fstar' + Qstar;
    eta = z(:,t+1) - M(1:J,t+1) - Fstar * z_t_t(:,t); % Forecast error
    z_t_T = z_t_t(:,t) + P_t_t(:,:,t) * Fstar' * pinv(fstar) * eta;
    P_t_T = P_t_t(:,:,t) - P_t_t(:,:,t) * Fstar' * pinv(fstar) * Fstar * P_t_t(:,:,t);
    % Rejection sampler to ensure z matches S
    accept = 0; wait = 0;
    while accept == 0
        z_full = z_t_T(1:J,1) + chol(P_t_T(1:J,1:J))' * randn(J,1);
%         % Alternative: 
%         z_full = mvnrnd(z_t_T(1:J,1), P_t_T(1:J,1:J))';
        z(:,t) = z_full(1:J,1);
        S_hat = double(z(:,t) > 0); % state implied by
        if all(S_hat == data.S(:,t)) == 1
            accept = 1;
        else
            wait = wait + 1;
        end
        if wait > 1e4
            z(:,t) = z_t_T(1:J,1) + mvrandn(lbndT(:,t) - z_t_T(1:J,1),ubndT(:,t) - z_t_T(1:J,1),P_t_T(1:J,1:J),1);
            accept = 1;
            %         error('Draw for z not matching S')
        end
    end
end


end

