function [z_t_t, P_t_t] = OPS_Kalman(Y,H,M,F,R,Q,z0,P0)
% Forward Kalman Filter for Owyang, Piger, Soques
%
% Outputs:
% z: (K x T) unobserved factor
%
% Inputs:
% Y: (N x T) observables variables
% H: (N x K) loadings
% M: (K x T) time-varying intercept in the measurement equation
% F: (K x K) AR coefficient matrix in measurement equation
% R: (N x N) covariance matrix for e_t
% Q: (K x K) covariance matrix for v_t
% z_0_0: (K x 1) initial value for state vector
% P_0_0: (K x K) initial value for covariance matrix of z_0_0

% Observation equation: y_t = H*z_t + e_t
% Measurement equation: z_t = M_t + F*z_tm1 + v_t
% e_t ~ N(0,R)
% v_t ~ N(0,Q)

T = size(Y,2);
z_t_t = zeros(size(z0,1),T);
P_t_t = zeros(size(z0,1),size(z0,1),T);

z_tm1_tm1 = z0;
P_tm1_tm1 = P0;

for t = 1:T
    % Prediction
    z_t_tm1 = M(:,t) + F * z_tm1_tm1;
    P_t_tm1 = F * P_tm1_tm1 * F' + Q;
    
    % Updating
    Y_t_tm1 = H * z_t_tm1; % Prediction
    n_t_tm1 = Y(:,t) - Y_t_tm1; % Predicition Error
    f_t_tm1 = H * P_t_tm1 * H' + R; % Predicition Error Covariance
    K_t = P_t_tm1 * H' / f_t_tm1; % Kalman gain
    
    z_t_t(:,t) = z_t_tm1 + K_t * n_t_tm1; % Updated state estimate
    P_t_t(:,:,t) = P_t_tm1 - K_t * H * P_t_tm1; % Updated covariance estimate
    
    z_tm1_tm1 = z_t_t(:,t);
    P_tm1_tm1 = P_t_t(:,:,t);
end

end

