% trading strategies on bubble interval estimates
% author: Simon Kwok
% date: 20210105

% input variables:
% dataname      name of the input dataset
% ma1           length of MA1 local window (in days)
% ma2           length of MA2 local window (in days)

% input dataset:
% dataname.mat (storing all variables generated from bubcp.m)

% required function:
% nw_cov.m              Newey-West variance

function bubcp_trade(dataname,ma1,ma2)

load(strcat(dataname,'.mat'))

plotind = 1;    % with plots if =1; no plots if =0

% trading parameters
tranr = 0.005;                  % transaction cost
vbuywin = [1,5,10,20];          % rebalance frequency (in days)
taucut = [0,0.25,0.5,1];        % endpoints for tau groups
ind_ts = 1;                     % momentum trading strategy indicator
                                % ind_ts=1 for S or Pi; =2 for S and Pi; =3 for Pi only 
dwin = 10;                      % step for returns in correl analysis

ntaugp = length(taucut)-1;
vlag = 1:1:5;

sbub_qcdfp = bubout.sbub_qcdfp;
sbub_qcdfc = bubout.sbub_qcdfc;
sbub_qcdf = bubout.sbub_qcdf;
sbub_qcdfp_se = bubout.sbub_qcdfp_se;
sbub_qcdfc_se = bubout.sbub_qcdfc_se;
sbub_qcdf_se = bubout.sbub_qcdf_se;
qcdfp_bias = bubout.qcdfp_bias;
qcdfc_bias = bubout.qcdfc_bias;
qcdf_bias = bubout.qcdf_bias;
qcdf_A_lb = bubout.qcdf_A_lb;
qcdf_A_ub = bubout.qcdf_A_ub;
qcdf_Ap_lb = bubout.qcdf_Ap_lb;
qcdf_Ap_ub = bubout.qcdf_Ap_ub;
qcdf_Ac_lb = bubout.qcdf_Ac_lb;
qcdf_Ac_ub = bubout.qcdf_Ac_ub;
qcdf_B1 = bubout.qcdf_B1;
qcdf_B21 = bubout.qcdf_B21;
qcdf_B22 = bubout.qcdf_B22;
qcdf_B23 = bubout.qcdf_B23;
qcdf_B3 = bubout.qcdf_B3;

qcdfp_lb = bubout.qcdfp_lb;
qcdfp_ub = bubout.qcdfp_ub;
qcdfc_lb = bubout.qcdfc_lb;
qcdfc_ub = bubout.qcdfc_ub;
lp = bubout.lp;
up = bubout.up;
lc = bubout.lc;
uc = bubout.uc;

nkc = bubout.nkc;
nkp = bubout.nkp;
sumvolc = bubout.sumvolc;
sumvolp = bubout.sumvolp;

filetype = setout.filetype;
modelname = setout.modelname;
yr = setout.yr;
pow = setout.pow;
step = setout.step;
opth = setout.opth;
hnumsd = setout.hnumsd;
nperiod = setout.nperiod;

sout = dataout.sout;
cp = dataout.cp;
tau = dataout.tau;
tr = dataout.tr;
da = dataout.da;

trate = zeros(nperiod,1);
for t=1:nperiod
    trate(t) = tr{t}(1);
end

% impute nonsensible values
semax = 1000;

% impute super big entry with value at t-1 
sbub_qcdf_se = real(sbub_qcdf_se);
sbub_qcdfp_se = real(sbub_qcdfp_se);
sbub_qcdfc_se = real(sbub_qcdfc_se);

ntau = size(sbub_qcdf,2);

for t=2:nperiod
    for i=1:ntau
        if sbub_qcdf_se(t,i) > semax
            sbub_qcdf_se(t,i) = sbub_qcdf_se(t-1,i);
        end
        if sbub_qcdfp_se(t,i) > semax
            sbub_qcdfp_se(t,i) = sbub_qcdfp_se(t-1,i);
        end
        if sbub_qcdfc_se(t,i) > semax
            sbub_qcdfc_se(t,i) = sbub_qcdfc_se(t-1,i);
        end
    end
end

% bub by gp

nem = 3; % num of estimation methods
bub_gp = zeros(nperiod,ntaugp,nem);
sbubcdf_mu = zeros(nperiod,ntaugp,3);
sbubcdf_se = zeros(nperiod,ntaugp,3);
sbubcdf_bc_mu = zeros(nperiod,ntaugp,3);
sbubcdf_bc_lb = zeros(nperiod,ntaugp,3);
sbubcdf_bc_ub = zeros(nperiod,ntaugp,3);

Awidth = zeros(nperiod,ntaugp,3);
sbubcdf_bias = zeros(nperiod,ntaugp,3);
B1 = zeros(nperiod,ntaugp);
B21 = zeros(nperiod,ntaugp);
B22 = zeros(nperiod,ntaugp);
B23 = zeros(nperiod,ntaugp);
B3 = zeros(nperiod,ntaugp);

qcdflb = zeros(nperiod,ntaugp,2);
qcdfub = zeros(nperiod,ntaugp,2);
allmass = zeros(nperiod,ntaugp,2);
nk = zeros(nperiod,ntaugp,2);
sumvol = zeros(nperiod,ntaugp,2);
klb = zeros(nperiod,ntaugp,2);
kub = zeros(nperiod,ntaugp,2);

% bub by tau group
for t=1:nperiod
    taulist = unique(tau{t}(cp{t}));
    for i=1:ntaugp
        gp = find((taulist>taucut(i))&(taulist<=taucut(i+1)));
        ngp = length(gp);
        if ngp>0
            bub_gp(t,i,1) = nanmean(sbub_qcdfp(t,gp) + qcdfp_bias(t,gp));
            bub_gp(t,i,2) = nanmean(sbub_qcdfc(t,gp) + qcdfc_bias(t,gp));
            bub_gp(t,i,3) = nanmean(sbub_qcdf(t,gp) + qcdf_bias(t,gp));

            qcdflb(t,i,1) = nanmean(qcdfp_lb(t,gp));
            qcdflb(t,i,2) = nanmean(qcdfc_lb(t,gp));
            qcdfub(t,i,1) = nanmean(qcdfp_ub(t,gp));
            qcdfub(t,i,2) = nanmean(qcdfc_ub(t,gp));
            allmass(t,i,1) = nanmean(qcdfp_ub(t,gp) - qcdfp_lb(t,gp));
            allmass(t,i,2) = nanmean(qcdfc_ub(t,gp) - qcdfc_lb(t,gp));
            nk(t,i,1) = nansum(nkp(t,gp));
            nk(t,i,2) = nansum(nkc(t,gp));
            klb(t,i,1) = nanmean(lp(t,gp));
            klb(t,i,2) = nanmean(lc(t,gp));
            kub(t,i,1) = nanmean(up(t,gp));
            kub(t,i,2) = nanmean(uc(t,gp));
            sumvol(t,i,1) = nansum(sumvolp(t,gp));
            sumvol(t,i,2) = nansum(sumvolc(t,gp));
            
            sbubcdf_bc_mu(t,i,1) = bub_gp(t,i,1);
            sbubcdf_bc_mu(t,i,2) = bub_gp(t,i,2);
            sbubcdf_bc_mu(t,i,3) = bub_gp(t,i,3);
            
            sbubcdf_se(t,i,1) = sqrt(nansum(sbub_qcdfp_se(t,gp).^2))/ngp;
            sbubcdf_se(t,i,2) = sqrt(nansum(sbub_qcdfc_se(t,gp).^2))/ngp;
            sbubcdf_se(t,i,3) = sqrt(nansum(sbub_qcdf_se(t,gp).^2))/ngp;        
                        
            sbubcdf_bc_ub(t,i,1) = bub_gp(t,i,1) - nanmean(qcdf_Ap_lb(t,gp));
            sbubcdf_bc_ub(t,i,2) = bub_gp(t,i,2) - nanmean(qcdf_Ac_lb(t,gp));
            sbubcdf_bc_ub(t,i,3) = bub_gp(t,i,3) - nanmean(qcdf_A_lb(t,gp));
            sbubcdf_bc_lb(t,i,1) = bub_gp(t,i,1) - nanmean(qcdf_Ap_ub(t,gp));
            sbubcdf_bc_lb(t,i,2) = bub_gp(t,i,2) - nanmean(qcdf_Ac_ub(t,gp));
            sbubcdf_bc_lb(t,i,3) = bub_gp(t,i,3) - nanmean(qcdf_A_ub(t,gp));
            
            Awidth(t,i,1) = nanmean(qcdf_Ap_ub(t,gp) - qcdf_Ap_lb(t,gp));
            Awidth(t,i,2) = nanmean(qcdf_Ac_ub(t,gp) - qcdf_Ac_lb(t,gp));
            Awidth(t,i,3) = nanmean(qcdf_A_ub(t,gp) - qcdf_A_lb(t,gp));
            
            sbubcdf_bias(t,i,1) = nanmean(qcdfp_bias(t,gp));
            sbubcdf_bias(t,i,2) = nanmean(qcdfc_bias(t,gp));
            sbubcdf_bias(t,i,3) = nanmean(qcdf_bias(t,gp));
            
            B1(t,i) = nanmean(qcdf_B1(t,gp));
            B21(t,i) = nanmean(qcdf_B21(t,gp));
            B22(t,i) = nanmean(qcdf_B22(t,gp));
            B23(t,i) = nanmean(qcdf_B23(t,gp));
            B3(t,i) = nanmean(qcdf_B3(t,gp));
            
        end
    end
end

% impute missing entry with time average
for i=1:ntaugp
    for k=1:nem
        bub_gp(isnan(bub_gp(:,i,k)),i,k)=nanmean(bub_gp(:,i,k));
    end
end


%% trading
nyr = length(yr);
nbuywin = length(vbuywin);
buy_bub = zeros(nperiod,ntaugp,nbuywin,nem);
buy_rbub = zeros(nperiod,nbuywin);
retur_rbub = zeros(nperiod,nbuywin);

rbubmu_gp = zeros(nperiod,ntaugp,nem);
r2bubmu_gp = zeros(nperiod,ntaugp,nem);
rsout = zeros(1,nperiod);
r2sout = zeros(1,nperiod);
rsbubcdf_mu = zeros(nperiod,ntaugp,3);
rsbubcdf_se = zeros(nperiod,ntaugp,3);
rsbubcdf_bc_mu = zeros(nperiod,ntaugp,3);
rsbubcdf_bc_lb = zeros(nperiod,ntaugp,3);
rsbubcdf_bc_ub = zeros(nperiod,ntaugp,3);

rnkp = zeros(nperiod,1);
rnkc = zeros(nperiod,1);
rlp = zeros(nperiod,1);
rup = zeros(nperiod,1);
rlc = zeros(nperiod,1);
ruc = zeros(nperiod,1);
rsumvolp = zeros(nperiod,1);
rsumvolc = zeros(nperiod,1);
rallmassp = zeros(nperiod,1);
rallmassc = zeros(nperiod,1);
rqcdfp_lb = zeros(nperiod,1);
rqcdfp_ub = zeros(nperiod,1);
rqcdfc_lb = zeros(nperiod,1);
rqcdfc_ub = zeros(nperiod,1);

rAwidth = zeros(nperiod,ntaugp,3);
rsewidth = zeros(nperiod,ntaugp,3);
rsbub_bias = zeros(nperiod,ntaugp,3);
rB1 = zeros(nperiod,ntaugp);
rB21 = zeros(nperiod,ntaugp);
rB22 = zeros(nperiod,ntaugp);
rB23 = zeros(nperiod,ntaugp);
rB3 = zeros(nperiod,ntaugp);

pbuy_bub = zeros(ntaugp,nbuywin,nem);
sumprofi_bub = zeros(ntaugp,nbuywin,nem);
sumnetprofi_bub = zeros(ntaugp,nbuywin,nem);
pbuy_rbub = zeros(1,nbuywin);
sumprofi_rbub = zeros(1,nbuywin);
sumnetprofi_rbub = zeros(1,nbuywin);

nbuy_bub = zeros(ntaugp,nbuywin,nem);
nsell_bub = zeros(ntaugp,nbuywin,nem);
sumpprofi_bub = zeros(ntaugp,nbuywin,nem);
sumnprofi_bub = zeros(ntaugp,nbuywin,nem);
yrretur_bub = zeros(ntaugp,nbuywin,nem);
yrnetr_bub = zeros(ntaugp,nbuywin,nem);
yrnetse_bub = zeros(ntaugp,nbuywin,nem);
yrnetexr_bub = zeros(ntaugp,nbuywin,nem);

nbuy_rbub = zeros(1,nbuywin);
nsell_rbub = zeros(1,nbuywin);
sumpprofi_rbub = zeros(1,nbuywin);
sumnprofi_rbub = zeros(1,nbuywin);
yrretur_rbub = zeros(1,nbuywin);
yrnetr_rbub = zeros(1,nbuywin);
yrnetse_rbub = zeros(1,nbuywin);
yrnetexr_rbub = zeros(1,nbuywin);
yrreturse_mkt = zeros(1,nbuywin);

% for graphing only
mc = ceil(ma1^0.25);
for t=ma1:nperiod
    vt = t-ma1+1:t;
    rbubmu_gp(t,:,:) = nanmean(bub_gp(vt,:,:));
    rsout(t) = nanmean(sout(vt));    
    for i=1:ntaugp
        for j=1:3
            rsbubcdf_mu(t,i,j) = nanmean(sbubcdf_mu(vt,i,j));
            rsbubcdf_se(t,i,j) = sqrt((nansum(sbubcdf_se(vt,i,j).^2) + 2*nw_cov(sbubcdf_mu(vt,i,j),mc)))/ma1;
            rsbubcdf_bc_mu(t,i,j) = nanmean(sbubcdf_bc_mu(vt,i,j));
            rsbubcdf_bc_lb(t,i,j) = nanmean(sbubcdf_bc_lb(vt,i,j)) - 1.96*rsbubcdf_se(t,i,j);
            rsbubcdf_bc_ub(t,i,j) = nanmean(sbubcdf_bc_ub(vt,i,j)) + 1.96*rsbubcdf_se(t,i,j);
                        
            rAwidth(t,i,j) = nanmean(Awidth(vt,i,j));
            rsewidth(t,i,j) = 2*1.96*rsbubcdf_se(t,i,j);
            rsbub_bias(t,i,j) = nanmean(sbubcdf_bias(vt,i,j));    
        end
        
        rB1(t,i) = nanmean(B1(vt,i));
        rB21(t,i) = nanmean(B21(vt,i));
        rB22(t,i) = nanmean(B22(vt,i));
        rB23(t,i) = nanmean(B23(vt,i));
        rB3(t,i) = nanmean(B3(vt,i));
    end
    rnkp(t)=sum(mean(nkp(vt,:)));
    rnkc(t)=sum(mean(nkc(vt,:)));
    rlp(t)=mean(mean(lp(vt,:)));
    rup(t)=mean(mean(up(vt,:)));
    rlc(t)=mean(mean(lc(vt,:)));
    ruc(t)=mean(mean(uc(vt,:)));
    rsumvolp(t)=sum(mean(sumvolp(vt,:)));
    rsumvolc(t)=sum(mean(sumvolc(vt,:)));
    rallmassp(t)=nanmean(nanmean(qcdfp_ub(vt,:) - qcdfp_lb(vt,:)));
    rallmassc(t)=nanmean(nanmean(qcdfc_ub(vt,:) - qcdfc_lb(vt,:)));
    rqcdfp_lb(t)=nanmean(nanmean(qcdfp_lb(vt,:)));
    rqcdfp_ub(t)=nanmean(nanmean(qcdfp_ub(vt,:)));
    rqcdfc_lb(t)=nanmean(nanmean(qcdfc_lb(vt,:)));
    rqcdfc_ub(t)=nanmean(nanmean(qcdfc_ub(vt,:)));
end

for t=ma2+1:nperiod
    r2bubmu_gp(t,:,:) = nanmean(bub_gp(max(1,t-ma2):t,:,:));  
    r2sout(t) = nanmean(sout(max(1,t-ma2):t));
end

bub_gp_dwin = bub_gp(1:dwin:end-dwin,:,:);
rbubmu_gp_dwin = rbubmu_gp(ma1:dwin:end-dwin,:,:);
sout_dwin = sout(1:dwin:nperiod-dwin);
sout_ma1 = sout(ma1:dwin:nperiod-dwin);

for j=1:nbuywin
    hdwin = vbuywin(j);

    vtr = ma2+1:hdwin:nperiod-hdwin;
    nvtr = length(vtr);
    profi_bub = zeros(nvtr,ntaugp,nem);
    netprofi_bub = zeros(nvtr,ntaugp,nem);
    retur_bub = zeros(nvtr,ntaugp,nem);
    
    for t=vtr
        for i=1:ntaugp       
            for k=1:nem
                if ind_ts == 1
                    buy_bub(t,i,j,k) = max((rbubmu_gp(t,i,k)>r2bubmu_gp(t,i,k)),(rsout(t)>r2sout(t)));
                elseif ind_ts == 2                
                    buy_bub(t,i,j,k) = ((rbubmu_gp(t,i,k)>r2bubmu_gp(t,i,k))*(rsout(t)>r2sout(t))); 
                else
                    buy_bub(t,i,j,k) = (rbubmu_gp(t,i,k)>r2bubmu_gp(t,i,k));
                end
                
                profi_bub(t,i,k) = buy_bub(t,i,j,k)*(sout(t+hdwin)-sout(t));
                netprofi_bub(t,i,k) = profi_bub(t,i,k)*(1-tranr);
                retur_bub(t,i,k) = buy_bub(t,i,j,k)*(log(sout(t+hdwin))-log(sout(t)));
            end
        end
    end

    profi_rbub = zeros(nvtr,1);
    netprofi_rbub = zeros(nvtr,1);
    retur_mkt = zeros(nvtr,1);    
    for t=vtr
        buy_rbub(t,j) = (rsout(t)>r2sout(t));
        profi_rbub(t) = buy_rbub(t,j)*(sout(t+hdwin)-sout(t));
        netprofi_rbub(t) = profi_rbub(t)*(1-tranr);
        retur_rbub(t,j) = buy_rbub(t,j)*(log(sout(t+hdwin))-log(sout(t)));
        
        retur_mkt(t) = log(sout(t+hdwin))-log(sout(t));
    end

    netr_bub = retur_bub;
    netr_rbub = retur_rbub;
    
    vtr1 = hdwin+ma2+1:hdwin:nperiod-hdwin;
    nvtr1 = length(vtr1);
    tranc_bub = zeros(nvtr1,ntaugp,nem);
    tranc_rbub = zeros(nvtr1,1);
    for t=vtr1
        for k=1:nem
            tranc_bub(t,:,k) = buy_bub(t,:,j,k)-buy_bub(t-hdwin,:,j,k);                      
            netr_bub(t,:,k) = netr_bub(t,:,k)-abs(tranc_bub(t,:,k))*tranr;        
        end
        tranc_rbub(t) = buy_rbub(t,j)-buy_rbub(t-hdwin,j);
        netr_rbub(t,j) = netr_rbub(t,j)-abs(tranc_rbub(t))*tranr;
    end    
    
    for k=1:nem
        pbuy_bub(:,j,k) = mean(buy_bub(vtr,:,j,k))';
        sumprofi_bub(:,j,k) = sum(profi_bub(vtr,:,k))';
        sumnetprofi_bub(:,j,k) = sum(netprofi_bub(vtr,:,k))';
    end
    pbuy_rbub(j) = mean(buy_rbub(vtr,j));
    sumprofi_rbub(j) = sum(profi_rbub(vtr));
    sumnetprofi_rbub(j) = sum(netprofi_rbub(vtr));
    
    for i=1:ntaugp   
        for k=1:nem
            nbuy_bub(i,j,k) = sum(tranc_bub(vtr,i,k)>0);
            nsell_bub(i,j,k) = sum(tranc_bub(vtr,i,k)<0);
            sumpprofi_bub(i,j,k) = sum(max(profi_bub(vtr,i,k),0));
            sumnprofi_bub(i,j,k) = -sum(min(profi_bub(vtr,i,k),0));
            yrretur_bub(i,j,k) = mean(retur_bub(vtr,i,k))*252/hdwin;
            yrnetr_bub(i,j,k) = mean(netr_bub(vtr,i,k))*252/hdwin;
            yrnetse_bub(i,j,k) = std(netr_bub(vtr,i,k))*sqrt(252/hdwin);            
            yrnetexr_bub(i,j,k) = mean(netr_bub(vtr,i,k)*252/hdwin - trate(vtr));        % mean excess return
        end
    end        
    nbuy_rbub(j) = sum(tranc_rbub(vtr1)>0);
    nsell_rbub(j) = sum(tranc_rbub(vtr1)<0);        
    sumpprofi_rbub(j) = sum(max(profi_rbub(vtr),0));
    sumnprofi_rbub(j) = -sum(min(profi_rbub(vtr),0));         
    yrretur_rbub(j) = mean(retur_rbub(vtr,j))*252/hdwin; 
    yrnetr_rbub(j) = mean(netr_rbub(vtr,j))*252/hdwin;  
    yrnetse_rbub(j) = std(netr_rbub(vtr,j))*sqrt(252/hdwin);
    yrnetexr_rbub(j) = mean(netr_rbub(vtr,j)*252/hdwin - trate(vtr));        % mean excess return
       
    yrreturse_mkt(j) = std(retur_mkt)*252/hdwin;
end
yrretur_mkt = (log(sout(end))-log(sout(ma2+1)))/nyr;

ntranc_bub = nbuy_bub + nsell_bub;
ntranc_rbub = nbuy_rbub + nsell_rbub;

% Sharpe ratio
yrnetsrpraw_bub = yrnetr_bub./yrnetse_bub;
yrnetsrpraw_rbub = yrnetr_rbub./yrnetse_rbub;
yrnetsrp_bub = yrnetexr_bub./yrnetse_bub;
yrnetsrp_rbub = yrnetexr_rbub./yrnetse_rbub;

%% correlation of current bub and future ds

%vlag = dwin:dwin:dwin*5;
nlag = length(vlag);
rho = zeros(nlag,3);
rho_pval = zeros(nlag,3);
rrho = zeros(nlag,3);
rrho_pval = zeros(nlag,3);
rho0 = zeros(nlag,1);
rho0_pval = zeros(nlag,1);
rrho0 = zeros(nlag,1);
rrho0_pval = zeros(nlag,1);

for m=vlag
    for i=1:ntaugp
        [R,P] = corrcoef(bub_gp_dwin(1:end-m,i,3),sout_dwin(m+1:end),'rows','complete');
        rho(m,i) = R(1,2);
        rho_pval(m,i) = P(1,2);
        
        [R,P] = corrcoef(rbubmu_gp_dwin(1:end-m,i,3),sout_ma1(m+1:end),'rows','complete');
        rrho(m,i) = R(1,2);
        rrho_pval(m,i) = P(1,2);

    end
    [R,P] = corrcoef(sout(1:nperiod-m),sout(m+1:nperiod),'rows','complete');
    rho0(m) = R(1,2);
    rho0_pval(m) = P(1,2);  
    
    [R,P] = corrcoef(rsout(ma1:nperiod-m),rsout(ma1+m:nperiod),'rows','complete');
    rrho0(m) = R(1,2);
    rrho0_pval(m) = P(1,2);  
end


disp(' ')


%% report
estmethod = {'scdf(p) bc','scdf(c) bc','scdf(cp) bc'};

disp(['Model  = ', modelname,'    Data = ',filetype,' ',yr{1},' to ',yr{end}])

fprintf('s retur win = ')
fprintf('%d ',vbuywin)
fprintf('\n')
fprintf('bub ma1 win = %d \n',ma1)
fprintf('bub ma2 win = %d \n',ma2)
fprintf('opth        = %d \n',opth)
fprintf('pow         = %d \n',pow)
fprintf('hnumsd      = %d \n',hnumsd)
fprintf('x grid step = %d \n',step)
fprintf('tranr       = %6.3f \n',tranr)
fprintf('taucut:')
fprintf('  %4.2f',taucut)
fprintf('\n')
disp(' ')

fprintf('                               puts              calls\n')
fprintf(' all mass        :')
fprintf('  %6.2f',mean(allmass(:,:,1)))
fprintf('   ')
fprintf('  %6.2f',mean(allmass(:,:,2)))
fprintf('\n')

fprintf('qcdf lb          :')
fprintf('  %6.2f',mean(qcdflb(:,:,1)))
fprintf('   ')
fprintf('  %6.2f',mean(qcdflb(:,:,2)))
fprintf('\n')

fprintf('qcdf ub          :')
fprintf('  %6.2f',mean(qcdfub(:,:,1)))
fprintf('   ')
fprintf('  %6.2f',mean(qcdfub(:,:,2)))
fprintf('\n')

fprintf('k lb             :')
fprintf('  %6.1f',mean(klb(:,:,1)))
fprintf('   ')
fprintf('  %6.1f',mean(klb(:,:,2)))
fprintf('\n')

fprintf('k ub             :')
fprintf('  %6.1f',mean(kub(:,:,1)))
fprintf('   ')
fprintf('  %6.1f',mean(kub(:,:,2)))
fprintf('\n')

fprintf('nk               :')
fprintf('  %6.1f',mean(nk(:,:,1)))
fprintf('   ')
fprintf('  %6.1f',mean(nk(:,:,2)))
fprintf('\n')

fprintf('sum vol in 1,000 :')
fprintf('  %6.1f',mean(sumvol(:,:,1))/1000)
fprintf('   ')
fprintf('  %6.1f',mean(sumvol(:,:,2))/1000)
fprintf('\n')
fprintf('\n')
fprintf('pc pvol > cvol   :')
fprintf('  %6.2f',mean(sumvol(:,:,1)>sumvol(:,:,2)))
fprintf('\n')

fprintf('pc pmass > cmass :')
fprintf('  %6.2f',mean(allmass(:,:,1)>allmass(:,:,2)))
fprintf('\n')

disp(' ')
fprintf('               ')
fprintf('              group 1                 ')
fprintf('              group 2                 ')
fprintf('              group 3                 ')
fprintf('\n')
for k=1:nem
    fprintf('%s:\n',estmethod{k})
    fprintf('trade on bub       ')
    fprintf('sw%3d    ',vbuywin)
    fprintf('  ')
    fprintf('sw%3d    ',vbuywin)
    fprintf('  ')
    fprintf('sw%3d    ',vbuywin)
    fprintf('\n')
    
    fprintf('num transac   :  ')
    for i=1:ntaugp
        fprintf('%7d  ',ntranc_bub(i,:,k))  
        fprintf('  ')
    end
    fprintf('\n')

    fprintf('prop holding  :  ')
    for i=1:ntaugp        
        fprintf('%7.2f  ',pbuy_bub(i,:,k))
        fprintf('  ')
    end
    fprintf('\n')
    %{
    fprintf('sum profit    :  ')
    for i=1:ntaugp     
    fprintf('%7.1f  ',sumprofi_bub(i,:,k))
    fprintf('  ')
    end
    fprintf('\n')
    
    fprintf('net profit    :  ')
    for i=1:ntaugp    
    fprintf('%7.1f  ',sumnetprofi_bub(i,:,k))
    fprintf('  ')
    end
    fprintf('\n')
    %}
    fprintf('yr return     :  ')
    for i=1:ntaugp    
    fprintf('%7.1f  ',yrretur_bub(i,:,k)*100)
    fprintf('  ')
    end
    fprintf('\n')
    
    fprintf('net return    :  ')
    for i=1:ntaugp    
    fprintf('%7.1f  ',yrnetr_bub(i,:,k)*100)
    fprintf('  ')
    end
    fprintf('\n')

    fprintf('Sharpe raw    :  ')  
    for i=1:ntaugp
    fprintf('%7.2f  ',yrnetsrpraw_bub(i,:,k))
    fprintf('  ')    
    end
    fprintf('\n')

    fprintf('Sharpe        :  ')  
    for i=1:ntaugp
    fprintf('%7.2f  ',yrnetsrp_bub(i,:,k))
    fprintf('  ')    
    end
    fprintf('\n')
    
    fprintf('NET return    :  ')
    for i=1:ntaugp    
    fprintf('%7.1f  ',(yrnetr_bub(i,:,k)-yrretur_mkt)*100)
    fprintf('  ')
    end
    fprintf('\n')
    %{
    fprintf('NET return se :  ')
    for i=1:ntaugp    
    fprintf('%7.1f  ',yrnetse_bub(i,:,k)*100)
    fprintf('  ')
    end
    fprintf('\n')
    fprintf('\n')
    %}
    fprintf('\n')
end

fprintf('mom only : \n')
fprintf('trade on bub       ')
fprintf('sw%3d    ',vbuywin)
fprintf('  ')
fprintf('\n')

fprintf('num transac   :  ')
fprintf('%7d  ',ntranc_rbub)  
fprintf('\n')

fprintf('prop holding  :  ')      
fprintf('%7.2f  ',pbuy_rbub)
fprintf('\n')
%{
fprintf('sum profit    :  ')   
fprintf('%7.1f  ',sumprofi_rbub)
fprintf('\n')

fprintf('net profit    :  ')
fprintf('%7.1f  ',sumnetprofi_rbub)
fprintf('\n')
%}
fprintf('yr return     :  ')    
fprintf('%7.1f  ',yrretur_rbub*100)
fprintf('\n')

fprintf('net return    :  ')  
fprintf('%7.1f  ',yrnetr_rbub*100)
fprintf('\n')

fprintf('Sharpe raw    :  ')  
fprintf('%7.2f  ',yrnetsrpraw_rbub)
fprintf('\n')

fprintf('Sharpe        :  ')  
fprintf('%7.2f  ',yrnetsrp_rbub)
fprintf('\n')

fprintf('NET return    :  ')
fprintf('%7.1f  ',(yrnetr_rbub-yrretur_mkt)*100)
fprintf('\n')
%{
fprintf('NET return se :  ')
fprintf('%7.1f  ',yrnetse_rbub(i,:)*100)
fprintf('\n')
fprintf('\n')
%}
fprintf('mkt yr return :  ')
fprintf('%7.1f  ',yrretur_mkt*100)
fprintf('\n')
%{
fprintf('mkt yr ret se :  ')
fprintf('%7.1f  ',yrreturse_mkt*100)
fprintf('\n')

fprintf('\n')
fprintf('Corr(bub(t), s(t+m)): \n')
fprintf('lag    rho  pval      rho  pval      rho  pval \n')
for m=1:nlag
fprintf(' %2d',vlag(m)*dwin)
for i=1:3
fprintf('  %5.2f  %4.2f  ',rho(m,i),rho_pval(m,i))
end
fprintf('\n')
end
fprintf('\n')

fprintf('Corr(MA bub(t), s(t+m)): \n')
fprintf('lag    rho  pval      rho  pval      rho  pval \n')
for m=1:nlag
fprintf(' %2d',vlag(m)*dwin)
for i=1:3
fprintf('  %5.2f  %4.2f  ',rrho(m,i),rrho_pval(m,i))
end
fprintf('\n')
end
fprintf('\n')
%}

% plotting
color{1}=[0.85,0.33,0.10];  %red
color{2}=[0.00,0.45,0.74];  %blue
color{3}=[0.47,0.67,0.19];  %green

esttitle = {    '$$\hat\Pi_{p}(\tau)$$','$$\hat\Pi_{c}(\tau)$$','$$\hat\Pi_{cp}(\tau)$$'};
corrcptitle = {'$$Corr(\hat\Pi_{cp,t}(\tau),S_{t+j}),\tau\in[0,0.25]$$',...
    '$$Corr(\hat\Pi_{cp,t}(\tau),S_{t+j}),\tau\in[0.25,0.5]$$',...
    '$$Corr(\hat\Pi_{cp,t}(\tau),S_{t+j}),\tau\in[0.5,1]$$'};
legname4 = {'$$\hat\Pi(\tau),\tau\in[0,0.25]$$','$$\hat\Pi(\tau),\tau\in[0.25,0.5]$$','$$\hat\Pi(\tau),\tau\in[0.5,1]$$','S\&P 500'};

if plotind==1
    
bubleg = cell(3,1);
for k=1:3
    bubleg{k} = sprintf('bub tau gp %d',k);
end

sraw = sout;
ttr0 = (ma1:nperiod)';
%ttr1 = ttr0-ma1+1;

close all

for j=1:3
rbubplot=[rsbubcdf_bc_mu(ttr0,1,j),rsbubcdf_bc_mu(ttr0,2,j),rsbubcdf_bc_mu(ttr0,3,j)];
rsplot=sout(ttr0)';

figure
hold on
yyaxis left
plot(da(ttr0),rbubplot);
x_ax = da(ttr0);
x_plot = [x_ax; flipud(x_ax)];
y1_plot = [rsbubcdf_bc_lb(ttr0,1,j); flipud(rsbubcdf_bc_ub(ttr0,1,j))];
y2_plot = [rsbubcdf_bc_lb(ttr0,2,j); flipud(rsbubcdf_bc_ub(ttr0,2,j))];
y3_plot = [rsbubcdf_bc_lb(ttr0,3,j); flipud(rsbubcdf_bc_ub(ttr0,3,j))];

b1=fill(x_plot, y1_plot, 'b','edgecolor','none','facealpha',0.5);
b2=fill(x_plot, y2_plot, 'b','edgecolor','none','facealpha',0.3);
b3=fill(x_plot, y3_plot, 'b','edgecolor','none','facealpha',0.1);
set( get( get( b1, 'Annotation'), 'LegendInformation' ), 'IconDisplayStyle', 'off' );
set( get( get( b2, 'Annotation'), 'LegendInformation' ), 'IconDisplayStyle', 'off' );
set( get( get( b3, 'Annotation'), 'LegendInformation' ), 'IconDisplayStyle', 'off' );

xlim([da(1),da(end)])
datetick('x','yy')
ax = gca;
ax.XAxisLocation = 'origin';
yyaxis right
plot(da(ttr0),rsplot);
%yup = ylim;

legend(legname4,'Location', 'NorthWest','Interpreter','latex', 'FontSize',12)
title(sprintf('%s',esttitle{j}), 'Interpreter','latex', 'FontSize',12)
hold off
end

k=3;
figure
for i=1:ntaugp
    subplot(ntaugp,1,i)
    crosscorr(real(bub_gp_dwin(:,i,k)),sout_dwin','NumLags',40)
    title(corrcptitle{i},'Interpreter','latex', 'FontSize',12)
end

k=3;
for j=1:3
rbubplot=[rbubmu_gp(ttr0,j,k),r2bubmu_gp(ttr0,j,k)];
rsplot=[sout(ttr0)',rsout(ttr0)',r2sout(ttr0)'];
nttr0=length(ttr0);

figure
hold on
yyaxis left
plot(da(ttr0),rbubplot);
xlim([da(1),da(end)])
datetick('x','yy')
ax = gca;
ax.XAxisLocation = 'origin';
yyaxis right
plot(da(ttr0),rsplot);
yup = ylim;

x_ax = da(ttr0);
x_plot = [x_ax; flipud(x_ax)];
y1_plot = [zeros(nttr0,1); yup(2)*flipud(buy_bub(ttr0,j,1,k))];
y2_plot = [zeros(nttr0,1); yup(2)*flipud(buy_rbub(ttr0,1))];

fill(x_plot, y1_plot, 'b','edgecolor','none','facealpha',0.3);
fill(x_plot, y2_plot, 'r','edgecolor','none','facealpha',0.3);

legend({sprintf('%s (MA1)',legname4{j}),sprintf('%s (MA2)',legname4{j}),'S\&P 500','S\&P 500 (MA1)','S\&P 500 (MA2)','buy on momentum of $$\hat\Pi$$ or $$S$$','buy on momentum of $$S$$'},'Location', 'NorthWest','Interpreter','latex', 'FontSize',12)
title('trading on momentum of S\&P 500 and $$\hat\Pi_{cp}$$','Interpreter','latex', 'FontSize',12)
hold off
end


figure
hold on
yyaxis left
plot(da(ttr0),rsumvolp(ttr0)/1000,'Color',color{2},'Linestyle','-','Linewidth',1);
plot(da(ttr0),rsumvolc(ttr0)/1000,'Color',color{3},'Linestyle','-','Linewidth',1);
xlim([da(1),da(end)])
datetick('x','yy')
ax = gca;
ax.XAxisLocation = 'origin';
yyaxis right
plot(da(ttr0),sraw(ttr0),'Color',color{1});
legend({'puts','calls','S&P 500'},'Location', 'NorthWest')
title('volume of traded options (in 1,000)')
hold off

figure
hold on
yyaxis left
plot(da(ttr0),rnkp(ttr0),'Color',color{2},'Linestyle','-','Linewidth',1);
plot(da(ttr0),rnkc(ttr0),'Color',color{3},'Linestyle','-','Linewidth',1);
xlim([da(1),da(end)])
datetick('x','yy')
ax = gca;
ax.XAxisLocation = 'origin';
yyaxis right
plot(da(ttr0),sraw(ttr0),'Color',color{1});
legend({'puts','calls','S&P 500'},'Location', 'NorthWest')
title('number of traded options')
hold off


figure
hold on
yyaxis left
plot(da(ttr0),rallmassp(ttr0),'Color',color{2},'Linestyle','-','Linewidth',1);
plot(da(ttr0),rallmassc(ttr0),'Color',color{3},'Linestyle','-','Linewidth',1);

xlim([da(1),da(end)])
datetick('x','yy')
ax = gca;
ax.XAxisLocation = 'origin';
yyaxis right
plot(da(ttr0),sraw(ttr0),'Color',color{1});
legend({'puts','calls','S&P 500'},'Location', 'NorthWest')
title('probability mass')
hold off


rsplot=sout(ttr0);
figure
hold on
yyaxis left

x_ax = da(ttr0);
x_plot = [x_ax; flipud(x_ax)];
y1_plot = [rqcdfp_lb(ttr0); flipud(rqcdfp_ub(ttr0))];
y2_plot = [rqcdfc_lb(ttr0); flipud(rqcdfc_ub(ttr0))];

b1=fill(x_plot, y1_plot, color{2},'edgecolor','none','facealpha',0.5);
b2=fill(x_plot, y2_plot, color{3},'edgecolor','none','facealpha',0.5);
%set( get( get( b1, 'Annotation'), 'LegendInformation' ), 'IconDisplayStyle', 'off' );
%set( get( get( b2, 'Annotation'), 'LegendInformation' ), 'IconDisplayStyle', 'off' );

xlim([da(1),da(end)])
datetick('x','yy')
ax = gca;
ax.XAxisLocation = 'origin';
yyaxis right
plot(da(ttr0),rsplot,'Color',color{1});
%yup = ylim;

legend({'puts','calls','S&P 500'},'Location', 'NorthWest')
title('empirical support of Q*')
hold off

figure
hold on
yyaxis left

x_ax = da(ttr0);
x_plot = [x_ax; flipud(x_ax)];
y1_plot = [rlp(ttr0); flipud(rup(ttr0))];
y2_plot = [rlc(ttr0); flipud(ruc(ttr0))];

b1=fill(x_plot, y1_plot, color{2},'edgecolor','none','facealpha',0.5);
b2=fill(x_plot, y2_plot, color{3},'edgecolor','none','facealpha',0.5);
%set( get( get( b1, 'Annotation'), 'LegendInformation' ), 'IconDisplayStyle', 'off' );
%set( get( get( b2, 'Annotation'), 'LegendInformation' ), 'IconDisplayStyle', 'off' );

xlim([da(1),da(end)])
datetick('x','yy')
ax = gca;
ax.XAxisLocation = 'origin';
yyaxis right
plot(da(ttr0),rsplot,'Color',color{1});
%yup = ylim;

legend({'puts','calls','S&P 500'},'Location', 'NorthWest')
title('strike price range')
hold off

end

end

