% trading strategies on bubbles estimated using simple and naive bounds
% author: Simon Kwok
% date: 20210105

% input variables:
% dataname      name of the input dataset
% ma1           length of MA1 local window (in days)
% ma2           length of MA2 local window (in days)

% input dataset:
% dataname.mat (storing all variables generated from bub12.m)

function bub12_trade(dataname,ma1,ma2)

load(strcat(dataname,'.mat'))

plotind = 0;    % with plots if =1; no plots if =0

% trading parameters
tranr = 0.005;                  % transaction cost
vbuywin = [1,5,10,20];          % rebalance frequency (in days)
taucut = [0,0.25,0.5,1];        % endpoints for tau groups
ind_ts = 1;                     % momentum trading strategy indicator
                                % ind_ts=1 for S or Pi; =2 for S and Pi; =3 for Pi only 
dwin = 10;                      % step for returns in correl analysis

ntaugp = length(taucut)-1;

motmc = bubout.motmc;
sbub_simbb_mu = bubout.sbub_simbb_mu;
sbub_simbb_m = bubout.sbub_simbb_m;
sbub_simbb_lb = bubout.sbub_simbb_lb;
sbub_simbb_ub = bubout.sbub_simbb_ub;
pckcnt = bubout.pckcnt;
pcp_mu = bubout.pcp_mu;
pcp_atm_mu = bubout.pcp_atm_mu;
pcp_otm_mu = bubout.pcp_otm_mu;
pcp_itm_mu = bubout.pcp_itm_mu;

pcp_absmu = bubout.pcp_absmu;
ncpk = bubout.ncpk;
ncpk_atm = bubout.ncpk_atm;
ncpk_otm = ncpk_atm;
ncpk_itm = ncpk_atm;

sumvolp = bubout.sumvolp;
sumvolc = bubout.sumvolc;
sumnump = bubout.sumnump;
sumnumc = bubout.sumnumc;
lp = bubout.lp;
lc = bubout.lc;
up = bubout.up;
uc = bubout.uc;
lpall = bubout.lpall;
lcall = bubout.lcall;
upall = bubout.upall;
ucall = bubout.ucall;

filetype = setout.filetype;
modelname = setout.modelname;
yr = setout.yr;
nperiod = setout.nperiod;
moneycut = setout.moneycut;

sout = dataout.sout;
cp = dataout.cp;
tau = dataout.tau;
tr = dataout.tr;
da = dataout.da;

trate = zeros(nperiod,1);
for t=1:nperiod
    trate(t) = tr{t}(1);
end

% bubble by gp
nem = 9; % number of estimation methods
bub_gp = zeros(nperiod,ntaugp,nem);
sumvol = zeros(nperiod,3,2);    % T x (no. of $ gp) x (option type)
sumnum = zeros(nperiod,3,2);    % T x (no. of $ gp) x (option type)
volp = zeros(nperiod,3,ntaugp);
volc = zeros(nperiod,3,ntaugp);
nump = zeros(nperiod,3,ntaugp);
numc = zeros(nperiod,3,ntaugp);
lk = zeros(nperiod,3,2);
uk = zeros(nperiod,3,2);
lkall = zeros(nperiod,2);
ukall = zeros(nperiod,2);

% bub by tau group
for t=1:nperiod
    taulist = unique(tau{t}(cp{t}));
    for i=1:ntaugp
        gp = find((taulist>taucut(i))&(taulist<=taucut(i+1)));
        ngp = length(gp);
        if ngp>0
            bub_gp(t,i,1) = nanmean(motmc(t,gp));
            bub_gp(t,i,2) = nanmean(sbub_simbb_mu(t,gp));
            bub_gp(t,i,3) = nanmean(sbub_simbb_m(t,gp));
            bub_gp(t,i,4) = nanmean(sbub_simbb_lb(t,gp));
            bub_gp(t,i,5) = nanmean(sbub_simbb_ub(t,gp));
            bub_gp(t,i,6) = nansum(pcp_mu(t,gp))/nansum(ncpk(t,gp));
            bub_gp(t,i,7) = nansum(pcp_otm_mu(t,gp))/nansum(ncpk_otm(t,gp));
            bub_gp(t,i,8) = nansum(pcp_atm_mu(t,gp))/nansum(ncpk_atm(t,gp));
            bub_gp(t,i,9) = nansum(pcp_itm_mu(t,gp))/nansum(ncpk_itm(t,gp));
            for k=1:3
                volp(t,k,i) = nansum(sumvolp(t,k,gp));
                volc(t,k,i) = nansum(sumvolc(t,k,gp));
                nump(t,k,i) = nansum(sumnump(t,k,gp));
                numc(t,k,i) = nansum(sumnumc(t,k,gp));
            end
        end
    end
    
    gp = find((taulist>taucut(1))&(taulist<=taucut(end)));
    lkall(t,1) = nanmean(lpall(t,lpall(t,gp)>0));
    lkall(t,2) = nanmean(lcall(t,lcall(t,gp)>0));
    ukall(t,1) = nanmean(upall(t,upall(t,gp)>0));
    ukall(t,2) = nanmean(ucall(t,ucall(t,gp)>0));
    for k=1:3        
        sumvol(t,k,1) = nansum(sumvolp(t,k,gp));
        sumvol(t,k,2) = nansum(sumvolc(t,k,gp));  
        sumnum(t,k,1) = nansum(sumnump(t,k,gp));
        sumnum(t,k,2) = nansum(sumnumc(t,k,gp));
        lk(t,k,1) = nanmean(lp(t,k,lp(t,k,gp)>0));
        lk(t,k,2) = nanmean(lc(t,k,lc(t,k,gp)>0));
        uk(t,k,1) = nanmean(up(t,k,up(t,k,gp)>0));
        uk(t,k,2) = nanmean(uc(t,k,uc(t,k,gp)>0));
    end
end

% if blank, impute the missing entry with mean over time
for i=1:ntaugp
    for k=1:nem
        bub_gp(isnan(bub_gp(:,i,k)),i,k)=nanmean(bub_gp(:,i,k));
    end
end


%% trading
nyr = length(yr);
nbuywin = length(vbuywin);
buy_bub = zeros(nperiod,ntaugp,nbuywin,nem);
buy_rbub = zeros(nperiod,nbuywin);
retur_rbub = zeros(nperiod,nbuywin);
retur_mkt = zeros(nperiod,nbuywin);

rbubmu_gp = zeros(nperiod,ntaugp,nem);
r2bubmu_gp = zeros(nperiod,ntaugp,nem);
rsout = zeros(1,nperiod);
r2sout = zeros(1,nperiod);
rsumvol = zeros(nperiod,3,2);
rsumnum = zeros(nperiod,3,2);
rlk = zeros(nperiod,3,2);
ruk = zeros(nperiod,3,2);
rlkall = zeros(nperiod,2);
rukall = zeros(nperiod,2);

pbuy_bub = zeros(ntaugp,nbuywin,nem);
sumprofi_bub = zeros(ntaugp,nbuywin,nem);
sumnetprofi_bub = zeros(ntaugp,nbuywin,nem);
pbuy_rbub = zeros(1,nbuywin);
sumprofi_rbub = zeros(1,nbuywin);
sumnetprofi_rbub = zeros(1,nbuywin);

nbuy_bub = zeros(ntaugp,nbuywin,nem);
nsell_bub = zeros(ntaugp,nbuywin,nem);
sumpprofi_bub = zeros(ntaugp,nbuywin,nem);
sumnprofi_bub = zeros(ntaugp,nbuywin,nem);
yrretur_bub = zeros(ntaugp,nbuywin,nem);
yrnetr_bub = zeros(ntaugp,nbuywin,nem);
yrnetse_bub = zeros(ntaugp,nbuywin,nem);
yrnetexr_bub = zeros(ntaugp,nbuywin,nem);

nbuy_rbub = zeros(1,nbuywin);
nsell_rbub = zeros(1,nbuywin);
sumpprofi_rbub = zeros(1,nbuywin);
sumnprofi_rbub = zeros(1,nbuywin);
yrretur_rbub = zeros(1,nbuywin);
yrnetr_rbub = zeros(1,nbuywin);
yrnetse_rbub = zeros(1,nbuywin);
yrnetexr_rbub = zeros(1,nbuywin);
yrreturse_mkt = zeros(1,nbuywin);
yrse_mkt = zeros(1,nbuywin);

for t=ma1:nperiod
    vt = t-ma1+1:t;
    rbubmu_gp(t,:,:) = nanmean(bub_gp(vt,:,:));
    rsout(t) = nanmean(sout(vt));
    rsumvol(t,:,:) = nanmean(sumvol(vt,:,:));
    rsumnum(t,:,:) = nanmean(sumnum(vt,:,:));
    rlk(t,:,:) = nanmean(lk(vt,:,:));
    ruk(t,:,:) = nanmean(uk(vt,:,:));
    rlkall(t,:) = nanmean(lkall(vt,:));
    rukall(t,:) = nanmean(ukall(vt,:));
end

for t=ma2+1:nperiod
    r2bubmu_gp(t,:,:) = nanmean(bub_gp(max(1,t-ma2):t,:,:));  
    r2sout(t) = nanmean(sout(max(1,t-ma2):t));
end

bub_gp_dwin = bub_gp(1:dwin:end-dwin,:,:);
rbubmu_gp_dwin = rbubmu_gp(ma1:dwin:end-dwin,:,:);
sout_dwin = sout(1:dwin:end-dwin);
sout_ma1 = sout(ma1:dwin:end-dwin);

for j=1:nbuywin
    hdwin = vbuywin(j);

    vtr = ma2+1:hdwin:nperiod-hdwin;
    nvtr = length(vtr);
    profi_bub = zeros(nvtr,ntaugp,nem);
    netprofi_bub = zeros(nvtr,ntaugp,nem);
    retur_bub = zeros(nvtr,ntaugp,nem);
    
    for t=vtr
        for i=1:ntaugp       
            for k=1:nem
                if ind_ts == 1
                    buy_bub(t,i,j,k) = max((rbubmu_gp(t,i,k)>r2bubmu_gp(t,i,k)),(rsout(t)>r2sout(t)));
                elseif ind_ts == 2                
                    buy_bub(t,i,j,k) = ((rbubmu_gp(t,i,k)>r2bubmu_gp(t,i,k))*(rsout(t)>r2sout(t))); 
                else
                    buy_bub(t,i,j,k) = (rbubmu_gp(t,i,k)>r2bubmu_gp(t,i,k));
                end
                
                profi_bub(t,i,k) = buy_bub(t,i,j,k)*(sout(t+hdwin)-sout(t));
                netprofi_bub(t,i,k) = profi_bub(t,i,k)*(1-tranr);
                retur_bub(t,i,k) = buy_bub(t,i,j,k)*(log(sout(t+hdwin))-log(sout(t)));
            end
        end
    end

    profi_rbub = zeros(nvtr,1);
    netprofi_rbub = zeros(nvtr,1);
     
    for t=vtr
        buy_rbub(t,j) = (rsout(t)>r2sout(t));
        profi_rbub(t) = buy_rbub(t,j)*(sout(t+hdwin)-sout(t));
        netprofi_rbub(t) = profi_rbub(t)*(1-tranr);
        retur_rbub(t,j) = buy_rbub(t,j)*(log(sout(t+hdwin))-log(sout(t)));

        retur_mkt(t,j) = log(sout(t+hdwin))-log(sout(t));
    end

    netr_bub = retur_bub;
    netr_rbub = retur_rbub;

    vtr1 = hdwin+ma2+1:hdwin:nperiod-hdwin;
    nvtr1 = length(vtr1);
    tranc_bub = zeros(nvtr1,ntaugp,nem);
    tranc_rbub = zeros(nvtr1,1);
    for t=vtr1
        for k=1:nem
            tranc_bub(t,:,k) = buy_bub(t,:,j,k)-buy_bub(t-hdwin,:,j,k);                      
            netr_bub(t,:,k) = netr_bub(t,:,k)-abs(tranc_bub(t,:,k))*tranr;        
        end
        tranc_rbub(t) = buy_rbub(t,j)-buy_rbub(t-hdwin,j);
        netr_rbub(t,j) = netr_rbub(t,j)-abs(tranc_rbub(t))*tranr;
    end    
    
    for k=1:nem
        pbuy_bub(:,j,k) = mean(buy_bub(vtr,:,j,k))';
        sumprofi_bub(:,j,k) = sum(profi_bub(vtr,:,k))';
        sumnetprofi_bub(:,j,k) = sum(netprofi_bub(vtr,:,k))';
    end
    pbuy_rbub(j) = mean(buy_rbub(vtr,j));
    sumprofi_rbub(j) = sum(profi_rbub(vtr));
    sumnetprofi_rbub(j) = sum(netprofi_rbub(vtr));
    
    for i=1:ntaugp   
        for k=1:nem
            nbuy_bub(i,j,k) = sum(tranc_bub(vtr,i,k)>0);
            nsell_bub(i,j,k) = sum(tranc_bub(vtr,i,k)<0);
            sumpprofi_bub(i,j,k) = sum(max(profi_bub(vtr,i,k),0));
            sumnprofi_bub(i,j,k) = -sum(min(profi_bub(vtr,i,k),0));
            yrretur_bub(i,j,k) = mean(retur_bub(vtr,i,k))*252/hdwin;
            yrnetr_bub(i,j,k) = mean(netr_bub(vtr,i,k))*252/hdwin;
            yrnetse_bub(i,j,k) = std(netr_bub(vtr,i,k))*sqrt(252/hdwin);            
            yrnetexr_bub(i,j,k) = mean(netr_bub(vtr,i,k)*252/hdwin - trate(vtr));        % mean excess return
        end
    end        
    nbuy_rbub(j) = sum(tranc_rbub(vtr1)>0);
    nsell_rbub(j) = sum(tranc_rbub(vtr1)<0);        
    sumpprofi_rbub(j) = sum(max(profi_rbub(vtr),0));
    sumnprofi_rbub(j) = -sum(min(profi_rbub(vtr),0));        
    yrretur_rbub(j) = mean(retur_rbub(vtr,j))*252/hdwin; 
    yrnetr_rbub(j) = mean(netr_rbub(vtr,j))*252/hdwin;  
    yrnetse_rbub(j) = std(netr_rbub(vtr,j))*sqrt(252/hdwin);
    yrnetexr_rbub(j) = mean(netr_rbub(vtr,j)*252/hdwin - trate(vtr));        % mean excess return   

    yrreturse_mkt(j) = std(retur_mkt(vtr,j))*252/hdwin;
    yrse_mkt(j) = std(retur_mkt(vtr,j))*sqrt(252/hdwin);
end
yrretur_mkt = (log(sout(end))-log(sout(ma2+1)))/nyr;
yrexr_mkt = yrretur_mkt - mean(trate);

ntranc_bub = nbuy_bub + nsell_bub;
ntranc_rbub = nbuy_rbub + nsell_rbub;

% Sharpe ratio
yrnetsrpraw_bub = yrnetr_bub./yrnetse_bub;
yrnetsrpraw_rbub = yrnetr_rbub./yrnetse_rbub;
yrnetsrp_bub = yrnetexr_bub./yrnetse_bub;
yrnetsrp_rbub = yrnetexr_rbub./yrnetse_rbub;
yrsrpraw_mkt = yrretur_mkt./yrse_mkt;
yrsrp_mkt = yrexr_mkt./yrse_mkt;

%% correlation of current bub and future ds

%vlag = dwin:dwin:dwin*5;
vlag = 1:1:5;
nlag = length(vlag);
rho = zeros(nlag,ntaugp);
rho_pval = zeros(nlag,ntaugp);
rrho = zeros(nlag,ntaugp);
rrho_pval = zeros(nlag,ntaugp);
rho0 = zeros(nlag,1);
rho0_pval = zeros(nlag,1);
rrho0 = zeros(nlag,1);
rrho0_pval = zeros(nlag,1);

for m=vlag
    for i=1:ntaugp
        [R,P] = corrcoef(bub_gp_dwin(1:end-m,i,2),sout_dwin(m+1:end),'rows','complete');        
        rho(m,i) = R(1,2);
        rho_pval(m,i) = P(1,2);
        
        [R,P] = corrcoef(rbubmu_gp_dwin(1:end-m,i,2),sout_ma1(m+1:end),'rows','complete');
        rrho(m,i) = R(1,2);
        rrho_pval(m,i) = P(1,2);

    end
    [R,P] = corrcoef(sout(1:nperiod-m),sout(m+1:nperiod),'rows','complete');
    rho0(m) = R(1,2);
    rho0_pval(m) = P(1,2);  
    
    [R,P] = corrcoef(rsout(ma1:nperiod-m),rsout(ma1+m:nperiod),'rows','complete');
    rrho0(m) = R(1,2);
    rrho0_pval(m) = P(1,2);  
end

for i=1:7
    pck_cn(i) = sum(sum(pckcnt==i));
    pck_pc(i) = sum(sum(pckcnt==i))/sum(sum(pckcnt>0));
end

disp(' ')


%% report
estmethod = {'otmc','simbb mu'};

disp(['Model  = ', modelname,'    Data = ',filetype,' ',yr{1},' to ',yr{end}])

fprintf('s retur win = ')
fprintf('%d ',vbuywin)
fprintf('\n')
%fprintf('s ma2 win   = %d \n',ma1)
fprintf('bub ma1 win = %d \n',ma1)
fprintf('bub ma2 win = %d \n',ma2)
fprintf('taucut:')
fprintf('  %4.2f',taucut)
fprintf('\n')
fprintf('\n')
fprintf('num of opt type   :')
fprintf('moneyness cutoff : %5.2f \n',moneycut)
fprintf('                       k<S     k~S     k>S        k>S     k~S     k<S\n')
fprintf('                      otmp    atmp    itmp       otmc    atmc    itmc\n')
for i=1:ntaugp
fprintf('tau gp %d         :',i)
fprintf('  %6.1f',mean(nump(:,:,i)))
fprintf('   ')
fprintf('  %6.1f',mean(numc(:,:,i)))
fprintf('\n')
end
fprintf('\n')
fprintf('\n')
fprintf('sum vol in 1,000 :')
fprintf('moneyness cutoff : %5.2f \n',moneycut)
fprintf('                       k<S     k~S     k>S        k>S     k~S     k<S\n')
fprintf('                      otmp    atmp    itmp       otmc    atmc    itmc\n')
for i=1:ntaugp
fprintf('tau gp %d         :',i)
fprintf('  %6.1f',mean(volp(:,:,i))/1000)
fprintf('   ')
fprintf('  %6.1f',mean(volc(:,:,i))/1000)
fprintf('\n')
end
fprintf('\n')
fprintf('\n')
fprintf('moneyness cutoff : %5.2f \n',moneycut)
fprintf('                       k<S     k~S     k>S        k>S     k~S     k<S\n')
fprintf('                      otmp    atmp    itmp       otmc    atmc    itmc\n')
fprintf('nk               :')
fprintf('  %6.1f',mean(sumnum(:,:,1)))
fprintf('   ')
fprintf('  %6.1f',mean(sumnum(:,:,2)))
fprintf('\n')

fprintf('sum vol in 1,000 :')
fprintf('  %6.1f',mean(sumvol(:,:,1))/1000)
fprintf('   ')
fprintf('  %6.1f',mean(sumvol(:,:,2))/1000)
fprintf('\n')

fprintf('sum vol pc       :')
fprintf('  %6.1f',mean(sumvol(:,:,1))/sum(sum(mean(sumvol)))*100)
fprintf('   ')
fprintf('  %6.1f',mean(sumvol(:,:,2))/sum(sum(mean(sumvol)))*100)
fprintf('\n')

fprintf('pc pvol > cvol   :')
fprintf('  %6.1f',mean(sumvol(:,:,1)>sumvol(:,:,2))*100)
fprintf('\n')
fprintf('pc pvol > cvol all k  :')
fprintf('  %6.1f',mean(sum(sumvol(:,:,1),2)>sum(sumvol(:,:,2),2))*100)
fprintf('\n')

fprintf('\n')
for i=1:7
    fprintf('pck type %d  :  %6d  %6.4f\n',i,pck_cn(i),pck_pc(i))
end
disp(' ')
fprintf('pc par mu     :  %6.2f\n',mean(mean(pcp_mu(pcp_mu~=0))))
fprintf('pc par amu    :  %6.2f\n',mean(mean(pcp_absmu(pcp_absmu~=0))))
fprintf('pc par mu gp  :')
fprintf('  %6.2f',mean(bub_gp(:,:,5)))
fprintf('\n')
fprintf('pc par amu gp :')
fprintf('  %6.2f',mean(bub_gp(:,:,6)))
fprintf('\n')

fprintf('               ')
fprintf('              group 1                 ')
fprintf('              group 2                 ')
fprintf('              group 3                 ')
fprintf('\n')
for k=1:2
    fprintf('%s:\n',estmethod{k})
    fprintf('trade on bub       ')
    fprintf('sw%3d    ',vbuywin)
    fprintf('  ')
    fprintf('sw%3d    ',vbuywin)
    fprintf('  ')
    fprintf('sw%3d    ',vbuywin)
    fprintf('\n')
    
    fprintf('num transac   :  ')
    for i=1:ntaugp
        fprintf('%7d  ',ntranc_bub(i,:,k))  
        fprintf('  ')
    end
    fprintf('\n')

    fprintf('prop holding  :  ')
    for i=1:ntaugp        
        fprintf('%7.2f  ',pbuy_bub(i,:,k))
        fprintf('  ')
    end
    fprintf('\n')
    %{
    fprintf('sum profit    :  ')
    for i=1:ntaugp     
    fprintf('%7.1f  ',sumprofi_bub(i,:,k))
    fprintf('  ')
    end
    fprintf('\n')
    
    fprintf('net profit    :  ')
    for i=1:ntaugp    
    fprintf('%7.1f  ',sumnetprofi_bub(i,:,k))
    fprintf('  ')
    end
    fprintf('\n')
    %}
    fprintf('yr return     :  ')
    for i=1:ntaugp    
    fprintf('%7.1f  ',yrretur_bub(i,:,k)*100)
    fprintf('  ')
    end
    fprintf('\n')
    
    fprintf('net return    :  ')
    for i=1:ntaugp    
    fprintf('%7.1f  ',yrnetr_bub(i,:,k)*100)
    fprintf('  ')
    end
    fprintf('\n')

    fprintf('Sharpe raw    :  ')  
    for i=1:ntaugp
    fprintf('%7.2f  ',yrnetsrpraw_bub(i,:,k))
    fprintf('  ')    
    end
    fprintf('\n')

    fprintf('Sharpe        :  ')  
    for i=1:ntaugp
    fprintf('%7.2f  ',yrnetsrp_bub(i,:,k))
    fprintf('  ')    
    end
    fprintf('\n')

    fprintf('NET return    :  ')
    for i=1:ntaugp    
    fprintf('%7.1f  ',(yrnetr_bub(i,:,k)-yrretur_mkt)*100)
    fprintf('  ')
    end
    fprintf('\n')
    %{
    fprintf('NET return se :  ')
    for i=1:ntaugp    
    fprintf('%7.1f  ',yrnetse_bub(i,:,k)*100)
    fprintf('  ')
    end
    fprintf('\n')
    fprintf('\n')
    %}
    fprintf('\n')
end

fprintf('mom only : \n')
fprintf('trade on bub       ')
fprintf('sw%3d    ',vbuywin)
fprintf('  ')
fprintf('\n')

fprintf('num transac   :  ')
fprintf('%7d  ',ntranc_rbub)  
fprintf('\n')

fprintf('prop holding  :  ')      
fprintf('%7.2f  ',pbuy_rbub)
fprintf('\n')
%{
fprintf('sum profit    :  ')   
fprintf('%7.1f  ',sumprofi_rbub)
fprintf('\n')

fprintf('net profit    :  ')
fprintf('%7.1f  ',sumnetprofi_rbub)
fprintf('\n')
%}
fprintf('yr return     :  ')    
fprintf('%7.1f  ',yrretur_rbub*100)
fprintf('\n')

fprintf('net return    :  ')  
fprintf('%7.1f  ',yrnetr_rbub*100)
fprintf('\n')

fprintf('Sharpe raw    :  ')  
fprintf('%7.2f  ',yrnetsrpraw_rbub)
fprintf('\n')

fprintf('Sharpe        :  ')  
fprintf('%7.2f  ',yrnetsrp_rbub)
fprintf('\n')

fprintf('NET return    :  ')
fprintf('%7.1f  ',(yrnetr_rbub-yrretur_mkt)*100)
fprintf('\n')
%{
fprintf('NET return se :  ')
fprintf('%7.1f  ',yrnetse_rbub(i,:)*100)
fprintf('\n')
fprintf('\n')
%}
fprintf('mkt : \n')

fprintf('Sharpe raw    :  ')  
fprintf('%7.2f  ',yrsrpraw_mkt)
fprintf('\n')

fprintf('Sharpe        :  ')  
fprintf('%7.2f  ',yrsrp_mkt)
fprintf('\n')

fprintf('mkt yr return :  ')
fprintf('%7.1f  ',yrretur_mkt*100)
fprintf('\n')
%{
fprintf('mkt yr ret se :  ')
fprintf('%7.1f  ',yrreturse_mkt*100)
fprintf('\n')
%}
fprintf('\n')
fprintf('Corr(bub(t), s(t+m)): \n')
fprintf('lag    rho  pval      rho  pval      rho  pval \n')
for m=1:nlag
fprintf(' %2d',vlag(m)*dwin)
for i=1:3
fprintf('  %5.2f  %4.2f  ',rho(m,i),rho_pval(m,i))
end
fprintf('\n')
end
fprintf('\n')

fprintf('Corr(MA bub(t), s(t+m)): \n')
fprintf('lag    rho  pval      rho  pval      rho  pval \n')
for m=1:nlag
fprintf(' %2d',vlag(m)*dwin)
for i=1:3
fprintf('  %5.2f  %4.2f  ',rrho(m,i),rrho_pval(m,i))
end
fprintf('\n')
end
fprintf('\n')


% plotting
color{1}=[0.85,0.33,0.10];  %red
color{2}=[0.00,0.45,0.74];  %blue
color{3}=[0.47,0.67,0.19];  %green

if plotind==1
    
bubleg = cell(3,1);
for k=1:3
    bubleg{k} = sprintf('bub tau gp %d',k);
end

sraw = sout;
ttr0 = (ma1:nperiod)';

close all

esttitle = {'$$\tilde\Pi_1(\tau)$$','$$\tilde\Pi_2(\tau)$$','simbb m','simbb lb','simbb ub','PCP mu','PCP atm mu'};
corrtitle = {'$$Corr(\Pi_t(\tau),S_{t+j}),\tau\in[0,0.25]$$','$$Corr(\Pi_t(\tau),S_{t+j}),\tau\in[0.25,0.5]$$','$$Corr(\Pi_t(\tau),S_{t+j}),\tau\in[0.5,1]$$'};
legname4 = {'$$\hat\Pi(\tau),\tau\in[0,0.25]$$','$$\hat\Pi(\tau),\tau\in[0.25,0.5]$$','$$\hat\Pi(\tau),\tau\in[0.5,1]$$','S\&P 500'};
dtitle = {'$$\bar D_t(\tau),\tau\in[0,0.25]$$','$$\bar D_t(\tau),\tau\in[0.25,0.5]$$','$$\bar D_t(\tau),\tau\in[0.5,1]$$','S\&P 500'};
appenfigtitle = {'put-call disparity, all options','put-call disparity, OTM options','put-call disparity, ATM options','put-call disparity, ITM options'};
set(gca,'DefaultTextFontSize',12)
for k=1:nem-2
figure
hold on
yyaxis left
plot(da(ttr0),rbubmu_gp(ttr0,1:3,k));
xlim([da(1),da(end)])
datetick('x','yy')
ax = gca;
ax.XAxisLocation = 'origin';
yyaxis right
plot(da(ttr0),sraw(ttr0));
legend(legname4,'Location', 'NorthWest', 'Interpreter','latex', 'FontSize',12)
title(esttitle{k}, 'Interpreter','latex', 'FontSize',12)
hold off
end

for k=6:9
figure
hold on
yyaxis left
plot(da(ttr0),rbubmu_gp(ttr0,1:3,k),'Linewidth',1);
ylim([-5,30])
xlim([da(1),da(end)])
datetick('x','yy')
ax = gca;
ax.XAxisLocation = 'origin';
yyaxis right
plot(da(ttr0),sraw(ttr0));
legend(dtitle,'Location', 'NorthWest', 'Interpreter','latex', 'FontSize',12)
title(appenfigtitle{k}, 'FontSize',12)
hold off
end

figure
for i=1:ntaugp
    subplot(ntaugp,1,i)
    crosscorr(real(bub_gp_dwin(:,i,2)),sout_dwin','NumLags',40)
    title(corrtitle{i},'Interpreter','latex', 'FontSize',12)
end

figure
for i=1:ntaugp
    subplot(ntaugp,1,i)
    crosscorr(real(rbubmu_gp_dwin(:,i,2)),sout_ma1','NumLags',40)
    title(['Corr(MA {\Pi_t}, MA {S_{t+m}})',sprintf(', gp %d',i)], 'FontSize',12)
end


for k=2
for j=1:3
rbubplot=[rbubmu_gp(ttr0,j,k),r2bubmu_gp(ttr0,j,k)];
rsplot=[sout(ttr0)',rsout(ttr0)',r2sout(ttr0)'];
nttr0=length(ttr0);

figure
hold on
yyaxis left
plot(da(ttr0),rbubplot);
xlim([da(1),da(end)])
datetick('x','yy')
ax = gca;
ax.XAxisLocation = 'origin';
yyaxis right
plot(da(ttr0),rsplot);
yup = ylim;

x_ax = da(ttr0);
x_plot = [x_ax; flipud(x_ax)];
y1_plot = [zeros(nttr0,1); yup(2)*flipud(buy_bub(ttr0,j,1,k))];
y2_plot = [zeros(nttr0,1); yup(2)*flipud(buy_rbub(ttr0,1))];

fill(x_plot, y1_plot, 'b','edgecolor','none','facealpha',0.3);
fill(x_plot, y2_plot, 'r','edgecolor','none','facealpha',0.3);

legend({sprintf('%s MA1',legname4{j}),sprintf('%s MA2',legname4{j}),'S\&P 500','S\&P 500 MA1','S\&P 500 MA2','buy on momentum of $$\Pi$$ or $$S$$','buy on momentum of $$S$$'},'Location', 'NorthWest', 'Interpreter','latex', 'FontSize',12)
title(sprintf('%s',esttitle{k}),'Interpreter','latex', 'FontSize',12)
hold off
end
end


figure
hold on
yyaxis left
plot(da(ttr0),rsumvol(ttr0,1,1)/1000,'Color',color{2},'Linestyle','--','Linewidth',1);
plot(da(ttr0),rsumvol(ttr0,2,1)/1000,'Color',color{2},'Linestyle','-','Linewidth',1);
plot(da(ttr0),rsumvol(ttr0,3,1)/1000,'Color',color{2},'Linestyle','-.','Linewidth',1);
plot(da(ttr0),rsumvol(ttr0,1,2)/1000,'Color',color{3},'Linestyle','--','Linewidth',1,'Marker','none');
plot(da(ttr0),rsumvol(ttr0,2,2)/1000,'Color',color{3},'Linestyle','-','Linewidth',1,'Marker','none');
plot(da(ttr0),rsumvol(ttr0,3,2)/1000,'Color',color{3},'Linestyle','-.','Linewidth',1,'Marker','none');
xlim([da(1),da(end)])
datetick('x','yy')
ax = gca;
ax.XAxisLocation = 'origin';
yyaxis right
plot(da(ttr0),sraw(ttr0),'Color',color{1});
legend({'OTM puts','ATM puts','ITM puts','OTM calls','ATM calls','ITM calls','S&P 500'},'Location', 'NorthWest', 'FontSize',12)
title('volume of traded options (in 1,000)', 'FontSize',12)
hold off

figure
hold on
yyaxis left
plot(da(ttr0),rsumnum(ttr0,1,1),'Color',color{2},'Linestyle','--','Linewidth',1);
plot(da(ttr0),rsumnum(ttr0,2,1),'Color',color{2},'Linestyle','-','Linewidth',1);
plot(da(ttr0),rsumnum(ttr0,3,1),'Color',color{2},'Linestyle','-.','Linewidth',1);
plot(da(ttr0),rsumnum(ttr0,1,2),'Color',color{3},'Linestyle','--','Linewidth',1,'Marker','none');
plot(da(ttr0),rsumnum(ttr0,2,2),'Color',color{3},'Linestyle','-','Linewidth',1,'Marker','none');
plot(da(ttr0),rsumnum(ttr0,3,2),'Color',color{3},'Linestyle','-.','Linewidth',1,'Marker','none');
xlim([da(1),da(end)])
datetick('x','yy')
ax = gca;
ax.XAxisLocation = 'origin';
yyaxis right
plot(da(ttr0),sraw(ttr0),'Color',color{1});
legend({'OTM puts','ATM puts','ITM puts','OTM calls','ATM calls','ITM calls','S&P 500'},'Location', 'NorthWest', 'FontSize',12)
title('number of traded options', 'FontSize',12)
hold off


rsplot=[sout(ttr0)'];
figure
hold on
yyaxis left

x_ax = da(ttr0);
x_plot = [x_ax; flipud(x_ax)];
y1_plot = [rlkall(ttr0,1); flipud(rukall(ttr0,1))];
y2_plot = [rlkall(ttr0,2); flipud(rukall(ttr0,2))];

b1=fill(x_plot, y1_plot, color{2},'edgecolor','none','facealpha',0.5);
b2=fill(x_plot, y2_plot, color{3},'edgecolor','none','facealpha',0.5);

%set( get( get( b1, 'Annotation'), 'LegendInformation' ), 'IconDisplayStyle', 'off' );
%set( get( get( b2, 'Annotation'), 'LegendInformation' ), 'IconDisplayStyle', 'off' );

plot(da(ttr0),rsplot,'Color',color{1},'Linestyle','-','Linewidth',0.5,'Marker','none');

xlim([da(1),da(end)])
datetick('x','yy')
ax = gca;
ax.XAxisLocation = 'origin';
ax.YAxis(2).Visible = 'off';

legend({'puts','calls','S&P 500'},'Location', 'NorthWest', 'FontSize',12)
title('strike price range', 'FontSize',12)
hold off


figure
hold on
yyaxis left

x_ax = da(ttr0);
x_plot = [x_ax; flipud(x_ax)];
y1_plot = [rlkall(ttr0,1); flipud(rukall(ttr0,1))];
y2_plot = [rlkall(ttr0,2); flipud(rukall(ttr0,2))];

b1=fill(x_plot, y1_plot, color{2},'edgecolor','none','facealpha',0.5);
b2=fill(x_plot, y2_plot, color{3},'edgecolor','none','facealpha',0.5);

%set( get( get( b1, 'Annotation'), 'LegendInformation' ), 'IconDisplayStyle', 'off' );
%set( get( get( b2, 'Annotation'), 'LegendInformation' ), 'IconDisplayStyle', 'off' );

p1=plot(da(ttr0),rlk(ttr0,2,1),'Color',color{2},'Linestyle','--','Linewidth',1,'Marker','none');
p2=plot(da(ttr0),ruk(ttr0,2,1),'Color',color{2},'Linestyle','--','Linewidth',1,'Marker','none');
p3=plot(da(ttr0),rlk(ttr0,2,2),'Color',[0.26,0.38,0.10],'Linestyle','-.','Linewidth',1,'Marker','none');
p4=plot(da(ttr0),ruk(ttr0,2,2),'Color',[0.26,0.38,0.10],'Linestyle','-.','Linewidth',1,'Marker','none');
set( get( get( p1, 'Annotation'), 'LegendInformation' ), 'IconDisplayStyle', 'off' );
%set( get( get( p2, 'Annotation'), 'LegendInformation' ), 'IconDisplayStyle', 'off' );
set( get( get( p3, 'Annotation'), 'LegendInformation' ), 'IconDisplayStyle', 'off' );
%set( get( get( p4, 'Annotation'), 'LegendInformation' ), 'IconDisplayStyle', 'off' );

plot(da(ttr0),rsplot,'Color',color{1},'Linestyle','-','Linewidth',0.5,'Marker','none');

xlim([da(1),da(end)])
datetick('x','yy')
ax = gca;
ax.XAxisLocation = 'origin';
ax.YAxis(2).Visible = 'off';

legend({'puts','calls','ATM puts','ATM calls','S&P 500'},'Location', 'NorthWest', 'FontSize',12)
title('strike price range', 'FontSize',12)
hold off


rbubplot=[rbubmu_gp(ttr0,1,3),rbubmu_gp(ttr0,2,3),rbubmu_gp(ttr0,3,3)];
rsplot=[sout(ttr0)'];

figure
hold on
yyaxis left
plot(da(ttr0),rbubplot);
x_ax = da(ttr0);
x_plot = [x_ax; flipud(x_ax)];
y1_plot = [rbubmu_gp(ttr0,1,4); flipud(rbubmu_gp(ttr0,1,5))];
y2_plot = [rbubmu_gp(ttr0,2,4); flipud(rbubmu_gp(ttr0,2,5))];
y3_plot = [rbubmu_gp(ttr0,3,4); flipud(rbubmu_gp(ttr0,3,5))];

b1 = fill(x_plot, y1_plot, 'b','edgecolor','none','facealpha',0.5);
b2 = fill(x_plot, y2_plot, 'b','edgecolor','none','facealpha',0.3);
b3 = fill(x_plot, y3_plot, 'b','edgecolor','none','facealpha',0.1);
set( get( get( b1, 'Annotation'), 'LegendInformation' ), 'IconDisplayStyle', 'off' );
set( get( get( b2, 'Annotation'), 'LegendInformation' ), 'IconDisplayStyle', 'off' );
set( get( get( b3, 'Annotation'), 'LegendInformation' ), 'IconDisplayStyle', 'off' );

xlim([da(1),da(end)])
datetick('x','yy')
ax = gca;
ax.XAxisLocation = 'origin';
yyaxis right
plot(da(ttr0),rsplot);
%yup = ylim;

legend(legname4,'Location', 'NorthWest','Interpreter','latex', 'FontSize',12)
title('$$\tilde\Pi_2(\tau)$$','Interpreter','latex', 'FontSize',12)
hold off

for i=1:3
    for j=3:5
        for t=1:nperiod
            rbubmu_gp_min0(t,i,j) = max(0,rbubmu_gp(t,i,j));
        end
    end
end

rbubplot=[rbubmu_gp_min0(ttr0,1,3),rbubmu_gp_min0(ttr0,2,3),rbubmu_gp_min0(ttr0,3,3)];
rsplot=[sout(ttr0)'];

figure
hold on
yyaxis left
plot(da(ttr0),rbubplot);
x_ax = da(ttr0);
x_plot = [x_ax; flipud(x_ax)];
y1_plot = [rbubmu_gp_min0(ttr0,1,4); flipud(rbubmu_gp_min0(ttr0,1,5))];
y2_plot = [rbubmu_gp_min0(ttr0,2,4); flipud(rbubmu_gp_min0(ttr0,2,5))];
y3_plot = [rbubmu_gp_min0(ttr0,3,4); flipud(rbubmu_gp_min0(ttr0,3,5))];

b1 = fill(x_plot, y1_plot, 'b','edgecolor','none','facealpha',0.5);
b2 = fill(x_plot, y2_plot, 'b','edgecolor','none','facealpha',0.3);
b3 = fill(x_plot, y3_plot, 'b','edgecolor','none','facealpha',0.1);
set( get( get( b1, 'Annotation'), 'LegendInformation' ), 'IconDisplayStyle', 'off' );
set( get( get( b2, 'Annotation'), 'LegendInformation' ), 'IconDisplayStyle', 'off' );
set( get( get( b3, 'Annotation'), 'LegendInformation' ), 'IconDisplayStyle', 'off' );

xlim([da(1),da(end)])
datetick('x','yy')
ax = gca;
ax.XAxisLocation = 'origin';
yyaxis right
plot(da(ttr0),rsplot);
%yup = ylim;

legend(legname4,'Location', 'NorthWest','Interpreter','latex', 'FontSize',12)
title('$$\tilde\Pi_2(\tau)$$','Interpreter','latex', 'FontSize',12)
hold off
end

end
