######################################################################################################
# This code reproduces Figures 2 and 3 from the paper "The Double-Edged Sword of Global Integration: #
# Robustness, Fragility & Contagion in the International Firm Network" by E. Grant and J. Yung,      #
# Journal of Applied Econometrics, forthcoming.                                                      #
#                                                                                                    #  
# Last checked using R studio Version 1.1.456.                                                       #
#                                                                                                    #
# Calls the function ForceAtlas2.R to create spring plots and reads the files in the "data" folder   #
# Con_Long*dates*_lag1_horizon1.RData and NetwEdges_Long*dates*_lag1_horizon1.csv, which contain     #
# the firm identifier (network nodes) with the weights (network edges) connecting firms during       #
# periods determined by *dates* and NetwNodes_Long.txt which contains node characteristics.          # 
#                                                                                                    #
# All figures are saved as individual .eps files in the "output" folder, use Adobe to open as pdf.   #
######################################################################################################

######################################################################################################
# Prep work                                                                                          #
######################################################################################################
rm(list=ls()) # clear the environment

# install packages if necessary
#install.packages("igraph")
#install.packages("MASS")
library(igraph)
library(MASS)

set.seed(1011) # set seed for random iterations
source("./data/ForceAtlas2.R") # calls on the file for springplot algorithm

# For loading the data
lag                <- 1 # VAR lag
HorizonList        <- 1 # IRF horizon
CurrentNetworkType <- "GIRFAbs" # generalized impulse response function in absolute value
ModelNameList <-c("Long1992_1996","Long1997_2001","Long2002_2006","Long2007_2011","Long2012_2016")

######################################################################################################
# Cycle through different datasets and recreate the springplots of networks                          #
######################################################################################################
for(ModelName in ModelNameList){

# Load edge data 
  # As full weights matrix
    load(file=paste("./data/Con_", ModelName, "_lag", lag, "_horizon", HorizonList,".RData", sep=""))
    WeightMatrixData <- ConnectionTable
    rm(ConnectionTable)
  # As edge data
    EdgeData <- as.matrix(read.csv2(file=paste("./data/NetwEdges_", ModelName, "_lag", lag, "_horizon", HorizonList,".csv", sep=""), dec=".", sep=","))
    EdgeData[,1:2]<-sub("StdRet", "", EdgeData[,1:2])
    
# Get node data
  NodeData <- as.matrix(read.csv2(file=paste("./data/NetwNodes_", substr(ModelName, 1, regexpr('[0-9]',ModelName)-1), ".txt", sep=""), dec=".", sep="\t"))

  # Create a graph from the network to use in plots and analysis.  Full network with both To and From connections, and self loops
    NetworkGraph <- graph_from_data_frame(d=EdgeData[,1:2], directed=TRUE, vertices=NodeData)
    E(NetworkGraph)$weight = as.numeric(EdgeData[,3])
    rm(EdgeData,NodeData)
  # Add metadata without underscores for LaTex tables
    V(NetworkGraph)$FX_IndShort<-sub("_", "-", V(NetworkGraph)$FX_IndShort)
    V(NetworkGraph)$HQCntry<-sub(",", "", V(NetworkGraph)$HQCntry)
    V(NetworkGraph)$Ticker<-sub("_", "-", V(NetworkGraph)$name)
    V(NetworkGraph)$Firm<-sub("_", " ", V(NetworkGraph)$FirmID)
    V(NetworkGraph)$Firm<-sub(",", " ", V(NetworkGraph)$Firm)
    V(NetworkGraph)$Firm<-sub("&", "AND", V(NetworkGraph)$Firm)
    CurrentGraphToPlot <- NetworkGraph
    rm(NetworkGraph)
    
# Create colors for firm categories to match the figures in the paper
  # IndShort
    V(CurrentGraphToPlot)$IndShort.color[V(CurrentGraphToPlot)$IndShort=="Fin"]          <- "black"
    V(CurrentGraphToPlot)$IndShort.color[V(CurrentGraphToPlot)$IndShort=="IndDiv"]       <- "gray70"
    V(CurrentGraphToPlot)$IndShort.color[V(CurrentGraphToPlot)$IndShort=="ConsNonCycl"]  <- "gold"
    V(CurrentGraphToPlot)$IndShort.color[V(CurrentGraphToPlot)$IndShort=="Util"]         <- "blue"
    V(CurrentGraphToPlot)$IndShort.color[V(CurrentGraphToPlot)$IndShort=="BasMater"]     <- "red"
    V(CurrentGraphToPlot)$IndShort.color[V(CurrentGraphToPlot)$IndShort=="ConsCycl"]     <- "forestgreen"
    V(CurrentGraphToPlot)$IndShort.color[V(CurrentGraphToPlot)$IndShort=="ICT"]          <- "aquamarine"
    V(CurrentGraphToPlot)$IndShort.color[V(CurrentGraphToPlot)$IndShort=="Energy"]       <- "darkorange"
  # FXRegion
    V(CurrentGraphToPlot)$FXRegion.color[V(CurrentGraphToPlot)$FXRegion=="US"]          <- "black"
    V(CurrentGraphToPlot)$FXRegion.color[V(CurrentGraphToPlot)$FXRegion=="Africa"]      <- "gray70"
    V(CurrentGraphToPlot)$FXRegion.color[V(CurrentGraphToPlot)$FXRegion=="Europe"]      <- "blue"
    V(CurrentGraphToPlot)$FXRegion.color[V(CurrentGraphToPlot)$FXRegion=="Asia"]        <- "red"
    V(CurrentGraphToPlot)$FXRegion.color[V(CurrentGraphToPlot)$FXRegion=="Canada"]      <- "forestgreen"
    V(CurrentGraphToPlot)$FXRegion.color[V(CurrentGraphToPlot)$FXRegion=="UK"]          <- "aquamarine"
    V(CurrentGraphToPlot)$FXRegion.color[V(CurrentGraphToPlot)$FXRegion=="Australia"]   <- "darkorange"
    
# Calculate ForceAtlas2 positions to be used in firm level plots
  FAPositions_Firm <- layout.forceatlas2(CurrentGraphToPlot, #An igraph network or a data frame of three columns: source, target, and weights
       #directed=TRUE, #Logical. TRUE if the network is directed. Ignored if graph is an igraph object
       dim = 2, #Dimension of the positions
       pos = NULL, #A data frame or matrix (NumberOfNodes x dimension) of the initial locations of points. If NULL the initial positions are random
       center=NULL, #Center of gravity
       nohubs = FALSE, #Logical. If TRUE nodes with high indegree have more central position than nodes with outdegree (for directed graphs)
       linlog = FALSE, #Logical. If TRUE the algorithm uses logarithmic attraction force `F <- log (1+F)`
       iterations =5000, #Number of iterations to be performed    5000
       plotstep   =50001, #Frequency of plotting intermediate iterations
       k = 1.0,         #Repel constant: the greater the constant k the stronger the repulsion force between points
       gravity=1.0,     #Indicates how strongly the nodes should be attracted to the center of gravity
       tolerance = 0.1, #Tolerance to swinging constant
       ks=0.1,         #Speed constant: the greater the value of `ks` the more movement the nodes make under the acting forces
       ksmax=10,        #Limits the speed from above
       delta = 2.0      #Modify attraction force, the weights are raised to the power of delta
       )

# Create Force Atlas plot with various color legends
  PlotTypeList<-c("IndShort","FXRegion")

# Export figures as .eps files (view w/Adobe)   
for(CurrentField in PlotTypeList){
  SaveFAPlot(CurrentGraphToPlot, FAPositions_Firm, "",
    paste("./output/SpringPlots_", ModelName, "_lag", lag, "_horizon", HorizonList,"_",CurrentNetworkType,"_ForceAtlas2", CurrentField, ".eps", sep=""),
    unique(cbind(vertex_attr(CurrentGraphToPlot, CurrentField),vertex_attr(CurrentGraphToPlot, paste(CurrentField, ".color", sep="")))),
    vertex_attr(CurrentGraphToPlot, paste(CurrentField, ".color", sep="")),2, FALSE)
  }
}
print("Done! Your figures were saved in the output folder as .eps files. Use Adobe to view as pdf.")
