
% First Application: The Environmental Kuznets Curve

clear

% Import dataset from Li & Qian & Su (2016)
year      = 1971:2010;
T         = length(year);
co2_num   = csvread('WBDI_CO2pc.csv',0,1);
[N1,T1]   = size(co2_num);
co2       = co2_num(2:N1,T1-T+1:T1)';

gdppc_num = csvread('WBDI_GDPPC.csv',0,1);
gdppc     = gdppc_num(2:N1,T1-T+1:T1)';

energy_num = csvread('WBDI_Epc.csv',0,1);
energy     = energy_num(2:N1,T1-T+1:T1)';

pop_num    = csvread('WBDI_Pop.csv',0,1);
pop        = pop_num(2:N1,T1-T+1:T1)';

pop(isnan(pop))=-99999;


% Exclude countries with population less than 5 million (in 1992: t == 19)
countries_small = [];
for i = 1 : length(co2')
    if (pop(19,i) <= 5000000); countries_small = [countries_small i]; 
    end
end
exclude = countries_small;

index  = 1:N1-1;
co2    = co2(:,setdiff(index,exclude));
gdppc  = gdppc(:,setdiff(index,exclude));
energy = energy(:,setdiff(index,exclude));


start_date = 1992 - 1970; % min 1971
end_date   = 2010 - 1970; % max 2010
year       = start_date:end_date;

% Observations with non-missing entries
c_i = (min(co2(start_date:end_date,:))~=-99999);
g_i = (min(gdppc(start_date:end_date,:))~=-99999);
e_i = (min(energy(start_date:end_date,:))~=-99999);

co2    = co2(start_date:end_date,:);
gdppc  = gdppc(start_date:end_date,:);
energy = energy(start_date:end_date,:);

si = c_i & g_i & e_i;

[T,N] = size(gdppc(:,si));
id = repmat((1:1:N)',T,1);

G  = reshape(log(gdppc(:,si))' ,N*T,1);
G2 = reshape((G.^2)'           ,N*T,1);
y  = reshape(log(co2(:,si))'   ,N*T,1);
E  = reshape(log(energy(:,si))',N*T,1);
xMatrix = [ones(N*T,1) G G2 E];

start_obsperiod = start_date + 1970;
end_obsperiod   = end_date   + 1970;


% Assumptions for Standard Errors
se_type = 0.3; % 0.3 0.2 0.1
% 0.3: Case A (autocorrelation-, conditional heteroskedasticity-robust)
% 0.2: Case B (conditional heteroskedasticity-robust and no autocorrelation)
% 0.1: Case C (conditional homoskedasticity and no autocorrelation)
% 0  :        (no heteroskedasticity and no autocorrleation; 
%              se from standard OLS on demeaned data)

% Regressor names
varNames_timevar =  {'cons','G', 'G2', 'E'};
varNames_timecons = {};
% Variable name list for output table
varNames_all =      {'cons',' ','G',' ','G2',' ','E',' '};
% Set row labels (use empty string for no label):
varNames_outtable = {'Cons',' ','GDP',' ','GDP2',' ','Energy',' '};

% vector of indicators for time-varying regressors
timevarReg_ind = [2:length(varNames_timevar)]';
regressorlist = [varNames_timevar(2:length(varNames_timevar)), varNames_timecons];

[~,p] = size(xMatrix);
xArray = reshape(xMatrix,N,T,p);


%%%%% Estimate Breakpoints %%%%%

% Estimate breakpoint vector with known number of breakpoints
%break_numb = 1; % integer between 1 and T
%breakvecEst = est_multibreak_pls(y,xMatrix,N,T,break_numb)'

% Call estimator est_multibreak_pls
penaltytype = 2; % 1: BIC-type log(NT)/NT; 2: log(log(NT))/NT
breakvecEst = est_numbbreaks_pls(y,xMatrix,N,T,penaltytype)';
disp(breakvecEst + start_date + 1970)

% Estimate model with no break (standard FE) for compariason in output table
breakvecEst_temp = NaN;
[betaSfeEstimator,~,varBetaSfeEstimator,~]...
    = est_beta_panelmultibreak(y,xArray,xMatrix,breakvecEst_temp,timevarReg_ind,se_type);
estoutMatrix_standardFE = [betaSfeEstimator,sqrt(varBetaSfeEstimator)];


%%%%% Estimate Beta coefficients %%%%%

if isnan(breakvecEst) % Special case of no breaks
    break_numb = 0;
    [betaSfeEstimator,betaIvEstimator,betaFfeEstimator,varBetaSfeEstimator,...
     varBetaIvEstimator,varBetaFfeEstimator]...
        = est_beta_panelmultibreak(y,xArray,xMatrix,breakvecEst,timevarReg_ind,se_type);
    disp('----- Fixed Effects Estimator over full sample -----');
    disp(' ');
    segments = {'coeff','sd'};
    estoutMatrix_standardFE = [SFEbetaEst_Matrix(:,j),SEbetaSfeEstimator(:,j)];
    varNames = varNames_timevar(1+1:length(varNames_timevar));
    r=[{'---'},segments;varNames.',num2cell([betaSfeEstimator,sqrt(varBetaSfeEstimator)])];
    disp(r);
else 
    % Estimate coefficients with SFE (within-regime demeaning) and FFE (full-sample demeaning)
    [betaSfeEstimator,betaFfeEstimator,varBetaSfeEstimator,...
     CovBetaSfeEstimator,varBetaFfeEstimator,CovBetaFfeEstimator]...
        = est_beta_panelmultibreak(y,xArray,xMatrix,breakvecEst,timevarReg_ind,se_type);
    [pTimevar,~] = size(timevarReg_ind);
    [break_numb,~] = size(breakvecEst);
    
    disp('----- FE Estimator by Break Segments -----')
    disp(' ')
    SFEbetaEst_Matrix  = reshape(betaSfeEstimator,[pTimevar,break_numb+1]);
    SEbetaSfeEstimator = reshape(sqrt(varBetaSfeEstimator),[pTimevar,break_numb+1]);
    segments = {'coeff','sd'};  
    estoutMatrix_FE = [SFEbetaEst_Matrix(:,1),SEbetaSfeEstimator(:,1)];
    for j = 2 : break_numb + 1
        if j < break_numb + 2 
            segments     = [ segments, {'coeff' ,'sd'} ];  
            estoutMatrix_FE = [ estoutMatrix_FE, [SFEbetaEst_Matrix(:,j),SEbetaSfeEstimator(:,j)] ];
        end
    end
    varNames = varNames_timevar(2:length(varNames_timevar));
    r=[{'---'},segments;varNames.',num2cell(estoutMatrix_FE)];
    disp(r);
        
    fprintf('----- FFE Estimator by Break Segments (first row: intercepts) -----')
    disp(' ')
    %disp(' ')
    FFEbetaEst_Matrix  = reshape(betaFfeEstimator(1:(break_numb+1)*p),[p,break_numb+1]);
    SEbetaFfeEstimator = reshape(sqrt(varBetaFfeEstimator(1:(break_numb+1)*p)),[p,break_numb+1]);
    segments = {'coeff','sd'};  
    varNames = [varNames_timevar(1:length(varNames_timevar)), varNames_timecons]; 
    estoutMatrix_FFE = [FFEbetaEst_Matrix(:,1),SEbetaFfeEstimator(:,1)];
    for j = 2 : break_numb + 1
        segments     = [ segments, {'coeff','sd'} ];
        estoutMatrix_FFE = [ estoutMatrix_FFE, [FFEbetaEst_Matrix(:,j),SEbetaFfeEstimator(:,j)] ];
    end
    out = [{'---'},segments;varNames.',num2cell(estoutMatrix_FFE)];
    disp(out);

end




%%%%% Wald Test; restriction vector: beta_q+1 = beta_q (joint test) %%%%%%%%%
Wald_test_allseg_FFE = [];
Wald_test_allseg_FE = [];
for FFE_ind = 0 : 1 % 1: FFE estimates; 0: FE estimates
for q = 1 : length(breakvecEst) % loop over all pairs beta_q+1 - beta_q
    l = q+1;
    if     FFE_ind == 1 
        beta_hat       = FFEbetaEst_Matrix;
        beta_hat_covar = CovBetaFfeEstimator;
    elseif FFE_ind == 0 
        beta_hat       = SFEbetaEst_Matrix;
        beta_hat_covar = CovBetaSfeEstimator;
    end
    k    = min(sum(~isnan(beta_hat(:,q))), sum(~isnan(beta_hat(:,l))));
    kmax = max(sum(~isnan(beta_hat(:,q))), sum(~isnan(beta_hat(:,l))));
    r_beta = beta_hat(1+kmax-k:end,l) - beta_hat(1+kmax-k:end,q);
    R_beta = [ diag(ones(k,1)), (-1)*diag(ones(k,1)) ];
    if     FFE_ind == 1
        if     q == 1; Var_beta = [beta_hat_covar(1:k,        1:k), beta_hat_covar(1:k,        k+2:k+2+k-1);
                                  beta_hat_covar(k+2:k+2+k-1,1:k), beta_hat_covar(k+2:k+2+k-1,k+2:k+2+k-1)  ];
        elseif q > 1;  Var_beta = beta_hat_covar(kmax-1+(q-2)*kmax+1:kmax-1+(q-2)*kmax+2*kmax,kmax-1+(q-2)*kmax+1:kmax-1+(q-2)*kmax+2*kmax);
        end
    elseif FFE_ind == 0; Var_beta = beta_hat_covar(1+(q-1)*k:(q-1)*k+2*k,1+(q-1)*k:(q-1)*k+2*k);
    end  
    W = r_beta'*((R_beta*Var_beta*R_beta')^(-1))*r_beta;
    if     W > chi2inv(0.99,length(r_beta)); W = {[num2str(W,'%.3f') '***']};
    elseif W > chi2inv(0.95,length(r_beta)); W = {[num2str(W,'%.3f') '**']};
    elseif W > chi2inv(0.90,length(r_beta)); W = {[num2str(W,'%.3f') '*']};
    else;                                    W = {[num2str(W,'%.3f') '']};
    end
    if     FFE_ind == 1; Wald_test_allseg_FFE = [Wald_test_allseg_FFE ; W];
    elseif FFE_ind == 0; Wald_test_allseg_FE  = [Wald_test_allseg_FE ; W];
    end
end
end
if     length(breakvecEst) == 1; rowNames = {'beta_2 - beta_1'}';
elseif length(breakvecEst) == 2; rowNames = {'beta_2 - beta_1','beta_3 - beta_2'}';
elseif length(breakvecEst) == 3; rowNames = {'beta_2 - beta_1','beta_3 - beta_2','beta_4 - beta_3'}';
elseif length(breakvecEst) == 4; rowNames = {'beta_2 - beta_1','beta_3 - beta_2','beta_4 - beta_3','beta_4 - beta_3'}';
end

disp('Wald tests for H0: beta_q+1 = beta_q (for FE):')
[rowNames,Wald_test_allseg_FE]
%disp('Chi^2 Statistics for FE:')
%[ 0.90, chi2inv(0.90,3); 0.95, chi2inv(0.95,3); 0.99, chi2inv(0.99,3) ]

disp('Wald tests for H0: beta_q+1 = beta_q (for FFE):')
[rowNames,Wald_test_allseg_FFE]
%disp('Chi^2 Statistics for FFE:')
%[ 0.90, chi2inv(0.90,4); 0.95, chi2inv(0.95,4); 0.99, chi2inv(0.99,4) ]




%%%%% Generate output tables for latex %%%%%

% Add significance stars to columns with beta estimates
[~,estout_colnumb] = size(estoutMatrix_FE);
estoutSum = [];
for m = 0 : 2
if     m == 0 
    estoutMatrix_temp = [NaN(1,2);estoutMatrix_standardFE];
    [estout_rownumb,estout_colnumb] = size(estoutMatrix_temp);
elseif m == 1 
    [estout_rownumb,estout_colnumb] = size(estoutMatrix_FFE);
    estoutMatrix_temp = estoutMatrix_FFE;
elseif m == 2 
    estoutMatrix_temp = [estoutMatrix_FE;NaN(length(varNames_timecons),estout_colnumb)];
    estoutMatrix_temp = [NaN(1,estout_colnumb);estoutMatrix_temp];
    [estout_rownumb,estout_colnumb] = size(estoutMatrix_temp);
end
estoutMatrix = cell(2*estout_rownumb,estout_colnumb/2);
q = 0;
for i = 1 : 2 : estout_colnumb 
for j = 1 : 1 : estout_rownumb
    if abs(estoutMatrix_temp(j,i)/estoutMatrix_temp(j,i+1)) > norminv(0.99,0,1) 
        estoutMatrix(2*j-1,i-q) = {[num2str(estoutMatrix_temp(j,i),'%.3f') '***']};
        estoutMatrix(2*j  ,i-q) = {['(' num2str(estoutMatrix_temp(j,i+1),'%.3f') ')']};
    elseif abs(estoutMatrix_temp(j,i)/estoutMatrix_temp(j,i+1)) > norminv(0.95,0,1)
        estoutMatrix(2*j-1,i-q) = {[num2str(estoutMatrix_temp(j,i),'%.3f') '**']};
        estoutMatrix(2*j  ,i-q) = {['(' num2str(estoutMatrix_temp(j,i+1),'%.3f') ')']};
    elseif abs(estoutMatrix_temp(j,i)/estoutMatrix_temp(j,i+1)) > norminv(0.90,0,1)
        estoutMatrix(2*j-1,i-q) = {[num2str(estoutMatrix_temp(j,i),'%.3f') '*']};
        estoutMatrix(2*j  ,i-q) = {['(' num2str(estoutMatrix_temp(j,i+1),'%.3f') ')']};
    else 
        estoutMatrix(2*j-1,i-q) = {num2str(estoutMatrix_temp(j,i),'%.3f')};
        estoutMatrix(2*j  ,i-q) = {['(' num2str(estoutMatrix_temp(j,i+1),'%.3f') ')']};
    end
end
q = q + 1;
end
estoutSum = [estoutSum, estoutMatrix];
end


ColLabels2 = {};
for j = 1:1: estout_colnumb/2 
    ColLabels2 = [ColLabels2, {['Beta' num2str(j)]}];   
end
ColLabels2 = [ColLabels2,ColLabels2];
if     estout_colnumb/2 == 6 ColLabels1 = {'+----','----+','----+','FFE','----+','----+','+----','----+','FE','----+','----+','----+'};  
elseif estout_colnumb/2 == 5 ColLabels1 = {'+----','----+','FFE','----+','----+','+----','----+','FE','----+','----+'};  
elseif estout_colnumb/2 == 4 ColLabels1 = {'+----','FFE','----+','----+','+----','FE','----+','----+'};  
elseif estout_colnumb/2 == 3 ColLabels1 = {'+----','FFE','----+','+----','FE','----+'};  
elseif estout_colnumb/2 == 2 ColLabels1 = {'+--FFE','--+','+--FE','--+'}; 
elseif estout_colnumb/2 == 1 ColLabels1 = {'FFE','FE'}; 
end
ColLabels2 = [['Beta'], ColLabels2];
ColLabels1 = [['Standard FE'], ColLabels1];
%disp('----- FFE and FE Estimator (first row: intercepts) -----');
%disp(' ')
%r=[{'---'},ColLabels2; varNames_all.',estoutSum];
%disp(r);


% Test if estimated slope coefficients are significantly different
% across segments (Assume: 1 or more breaks)
t_test_out = varNames_timevar(1:length(varNames_timevar))';
table_out_betadiff  = varNames_outtable(1:length(varNames_timevar)*2)';
input.tableRowLabels = table_out_betadiff;
for EstType = 1:2 % 1: FFE; 2: FE
    ptvar = length(timevarReg_ind);
    if EstType == 1 
        betaEstimator    = betaFfeEstimator;
        varBetaEstimator = varBetaFfeEstimator;
        CovBetaEstimator = CovBetaFfeEstimator;
        pfull = p;
        betadiff    = betaEstimator(pfull+1:pfull+ptvar+1,1) - [zeros(1,1);betaEstimator(2:ptvar+1,1)];
        betadiff_se = sqrt( varBetaEstimator(pfull+1:pfull+1+ptvar,1) + [zeros(1,1);varBetaEstimator(1+1:1+ptvar,1)] ...
                            - [zeros(1,1);2*diag(CovBetaEstimator(ptvar+1+1:ptvar+1+ptvar,1:ptvar))]                    );
        for b = 2:break_numb
            betadiff    = [betadiff,    betaEstimator(b*pfull+1:(b+1)*pfull) - betaEstimator((b-1)*pfull+1:b*pfull,1)];
            betadiff_se = [betadiff_se, sqrt( varBetaEstimator(b*pfull+1:(b+1)*pfull,1) + varBetaEstimator((b-1)*pfull+1:b*pfull,1) ...
                                 - 2*diag(CovBetaEstimator(ptvar+(b-1)*pfull+1:ptvar+b*pfull,ptvar+(b-2)*pfull+1:ptvar+(b-1)*pfull+1)) )];
        end
        t_test = betadiff ./ betadiff_se;
    elseif EstType == 2 
        betaEstimator    = betaSfeEstimator;
        varBetaEstimator = varBetaSfeEstimator;
        CovBetaEstimator = CovBetaSfeEstimator;
        pfull = ptvar;
        betadiff    = betaEstimator(pfull+1:pfull+ptvar,1) - betaEstimator(1:ptvar,1);
        betadiff_se = sqrt(varBetaEstimator(pfull+1:pfull+ptvar,1) + varBetaEstimator(1:ptvar,1) ...
                              - 2*diag(CovBetaEstimator(pfull+1:pfull+ptvar,1:ptvar))                 );                   
        for b = 2:break_numb
            betadiff    = [betadiff,    betaEstimator(b*pfull+1:(b+1)*pfull,1) - betaEstimator((b-1)*pfull+1:b*pfull,1)];                       
            betadiff_se = [betadiff_se, sqrt(varBetaEstimator(b*pfull+1:(b+1)*pfull,1) + varBetaEstimator((b-1)*pfull+1:b*pfull,1) ...
                                               - 2*diag(CovBetaEstimator(b*pfull+1:(b+1)*pfull,(b-1)*pfull+1:b*pfull))                )];
        end
        betadiff = [nan(1,break_numb);betadiff];
        betadiff_se = [nan(1,break_numb);betadiff_se];
        t_test = betadiff ./ betadiff_se;
    end
    signif = {};
    diff   = {};
    for j = 1 : pfull + EstType - 1
        signif_temp = {};
        diff_temp = {};
        for b = 1:break_numb
            if     abs(t_test(j,b)) > norminv(0.99,0,1); signif_temp = [signif_temp, {[num2str(t_test(j,b),'%.3f') '***']}];
            elseif abs(t_test(j,b)) > norminv(0.95,0,1); signif_temp = [signif_temp, {[num2str(t_test(j,b),'%.3f') '**']} ];
            elseif abs(t_test(j,b)) > norminv(0.9 ,0,1); signif_temp = [signif_temp, {[num2str(t_test(j,b),'%.3f') '*']}  ];
            else;                                        signif_temp = [signif_temp, {[num2str(t_test(j,b),'%.3f') ' ']}  ];
            end
            if     abs(t_test(j,b)) > norminv(0.99,0,1); diff_temp   = [diff_temp, [{[num2str(betadiff(j,b),'%.3f') '***']}; {['(' num2str(betadiff_se(j,b),'%.3f') ')']}] ];
            elseif abs(t_test(j,b)) > norminv(0.95,0,1); diff_temp   = [diff_temp, [{[num2str(betadiff(j,b),'%.3f') '**' ]}; {['(' num2str(betadiff_se(j,b),'%.3f') ')']}] ];
            elseif abs(t_test(j,b)) > norminv(0.9 ,0,1); diff_temp   = [diff_temp, [{[num2str(betadiff(j,b),'%.3f') '*'  ]}; {['(' num2str(betadiff_se(j,b),'%.3f') ')']}] ];
            else;                                        diff_temp   = [diff_temp, [{[num2str(betadiff(j,b),'%.3f') ' '  ]}; {['(' num2str(betadiff_se(j,b),'%.3f') ')']}] ];
            end
        end
        signif = [ signif; signif_temp ];
        diff = [ diff; diff_temp ];
    end
    t_test_out = [t_test_out, signif];
    table_out_betadiff  = [table_out_betadiff, diff];
end
disp([{' ','FE','FE','FE', 'FFE','FFE','FFE' };
      {' ','beta2-beta1','beta3-beta2','beta4-beta3', 'beta2-beta1','beta3-beta2','beta4-beta3'}; 
      table_out_betadiff])


% Combine Results before generating Latex table
% table_out_betadif: estimated beta_j - beta_j-1 with standard errors of FFE and FE
% estoutSum:         estimated beta_j and standard errors of FFE and FE
if     break_numb == 2
    table_combresults = [ [ [ [ {'-'},{'-'},{'-'},{'-'};{' '},{' '},{' '},{' '}]; estoutSum(3:end,1:1+break_numb+1)] ,...
                               table_out_betadiff(:,2:break_numb+1), estoutSum(:,break_numb+3:2*break_numb+3), table_out_betadiff(:,break_numb+2:2*break_numb+1) ]; ...
                               ['NaN','NaN','NaN','NaN', Wald_test_allseg_FFE', 'NaN','NaN','NaN', Wald_test_allseg_FE' ] ];
    ColLabels2_combresults = [ ColLabels2(1:break_numb+1+1),        {'Beta2-Beta1'}, {'Beta3-Beta2'}, ...
                               ColLabels2(break_numb+3:2*break_numb+3),{'Beta2-Beta1'}, {'Beta3-Beta2'}     ];
    ColLabels1_combresults = [ ColLabels1(1:break_numb+1+1),        {'---+'}, {'---+'}, ...
                               ColLabels1(break_numb+3:2*break_numb+3),{'---+'}, {'---+'}     ];
elseif break_numb == 3
    table_combresults = [ [ [ [ {'-'},{'-'},{'-'},{'-'},{'-'};{' '},{' '},{' '},{' '},{' '}]; estoutSum(3:end,1:1+break_numb+1)] ,...
                               table_out_betadiff(:,2:break_numb+1), estoutSum(:,break_numb+3:2*break_numb+3), table_out_betadiff(:,break_numb+2:2*break_numb+1) ]; ...
                               ['NaN','NaN','NaN','NaN', 'NaN', Wald_test_allseg_FFE', 'NaN','NaN','NaN','NaN', Wald_test_allseg_FE' ] ];
    ColLabels2_combresults = [ ColLabels2(1:break_numb+1+1),            {'Beta2-Beta1'}, {'Beta3-Beta2'}, {'Beta4-Beta3'}, ...
                               ColLabels2(break_numb+3:2*break_numb+3), {'Beta2-Beta1'}, {'Beta3-Beta2'}, {'Beta4-Beta3'}  ];
    ColLabels1_combresults = [ ColLabels1(1:break_numb+1+1),           {'---+'}, {'---+'}, {'---+'}, ...
                               ColLabels1(break_numb+3:2*break_numb+3),{'---+'}, {'---+'}, {'---+'}     ];
end                           
table_combresults = strrep(table_combresults,'(NaN)',' ');
table_combresults = strrep(table_combresults,'NaN','-');                      
                       
input.data = table_combresults;
outtable = [ ColLabels2_combresults; num2cell(input.data)];
RowLabels = ['-',varNames_all,'Wald Test'];
varNames_outtable = [varNames_outtable,'Wald Test'];
out = [{'---'},ColLabels1_combresults;RowLabels.',outtable];
%disp(out);

%%%%% Export matlab matrix input.data to Latex table %%%%%
% Set column labels (use empty string for no label):
input.tableColLabels = ColLabels2_combresults;
% Set row labels (use empty string for no label):
input.tableRowLabels = varNames_outtable;
% Switch transposing/pivoting your table:
input.transposeTable = 0;
% Determine whether input.dataFormat is applied column or row based:
input.dataFormatMode = 'column'; % use 'column' or 'row'
input.dataFormat = {'%.3f'};
input.dataNanString = '-';
% Column alignment in Latex table ('l'=left-justified,'c'=centered,'r'=right-justified):  
input.tableColumnAlignment = 'c';
% Switch table borders on/off:
input.tableBorders = 0;
% LaTex table caption: 
start = start_date + 1970;
if     estout_colnumb/2 == 6; tableCap = ['FFE and FE estimates of slope coefficients; Breaks were estimated at ' num2str(breakvecEst(1,1)+start) ' and ' num2str(breakvecEst(2,1)+start) ' and ' num2str(breakvecEst(3,1)+start) ' and ' num2str(breakvecEst(4,1)+start) ' and ' num2str(breakvecEst(5,1)+start)];  
elseif estout_colnumb/2 == 5; tableCap = ['FFE and FE estimates of slope coefficients; Breaks were estimated at ' num2str(breakvecEst(1,1)+start) ' and ' num2str(breakvecEst(2,1)+start) ' and ' num2str(breakvecEst(3,1)+start) ' and ' num2str(breakvecEst(4,1)+start)];  
elseif estout_colnumb/2 == 4; tableCap = ['FFE and FE estimates of slope coefficients; Breaks were estimated at ' num2str(breakvecEst(1,1)+start) ' and ' num2str(breakvecEst(2,1)+start) ' and ' num2str(breakvecEst(3,1)+start)];  
elseif estout_colnumb/2 == 3; tableCap = ['FFE and FE estimates of slope coefficients; Breaks were estimated at ' num2str(breakvecEst(1,1)+start) ' and ' num2str(breakvecEst(2,1)+start)];  
elseif estout_colnumb/2 == 2; tableCap = ['FFE and FE estimates of slope coefficients; Break was estimated at '   num2str(breakvecEst(1,1)+start)];
elseif estout_colnumb/2 == 1; tableCap = 'FFE and FE estimates of slope coefficients; No break found';
end
input.tableCaption = tableCap;
% LaTex table label:
input.tableLabel = 'MyTableLabel';
% Switch to generate a complete LaTex document or just a table:
input.makeCompleteLatexDocument = 0;
% call latexTable:
latex = latexTable(input);



