% ML_SmokingGun_efactorsML -  Maximum Likelihood Estimation of Dynamic Factor Model (Special Version)
% 
%  The Model:  
%   x_t = LF_t + xsi
%   F_t = A(L)F_{t-1}+ G u_t
% 
% [F, L, A, e, u, G, LL] = ML_SmokingGun_efactorsML(X,q,r,p,max_iter,method)
%
% Inputs
%   X - matrix of observable variables
%   r - number of static factors
%   q - number of dynamic factors
%   p - number of lags in the VAR for the static factors     
%   max_iter - maximum number of iterations in estimation
%
% Outputs
%   F - Static Factors
%   L - Factor Loadings
%   A - VAR Coefficients
%   u - Common Shocks
%   G - rxq matrix
%  LL - Log Likelihood
% 

% This procedure was downloaded from Domenico Giannone's web page as part
% of the replication files of the paper Doz, C., Giannone, D., and Reichlin, L.
% ?A Quasi Maximum Likelihood Approach for Large Approximate Dynamic Factor Models?,
% Review of Economics and Statistics, forthcoming
% This codes were successively customized and commented 
% by Matteo Luciani (matteo.luciani@ulb.ac.be)

function [F, L, A, u, G, LL, SF] = ML_SmokingGun_efactorsML(X,q,r,p,max_iter,method)

thresh = 1e-4;                                                              % Treshold for the convergence of the EM algorithm
OPTS.disp = 0;                                                              % sets off dysplay option for eigs
[T,N] = size(X);                                                            % size of the panel
x = ML_Standardize(X);                                                      % Standardize variables
Q = zeros(p*r,p*r);
Q(1:r,1:r) = eye(r);

%%% ___________________________________________________ %%%
%%%  %%%
%%% Estimate the Factor model with Principal Components %%%
%%% ___________________________________________________ %%%
%%%  %%%
[F,L, ~, xsi]=ML_efactors2(x,r,2); F=F*sqrt(N);L=L/sqrt(N);                 % Estimate Static Factors, Factor Loadings and the idiosyncratic component
[A e]=ML_VAR(F,p,0);                                                        % VAR on the Static Factors
if r > q; [u, G]=ML_edynfactors2(e,q); else u=e; G=eye(r); end;             % Common Shocks
Q(1:r,1:r)=G*G';                                                            % Variance of the VAR residuals
C = [L zeros(N,r*(p-1))];                                                   % This modification is needed in case p>2
R = diag(diag(cov(xsi))); R(68,68)=1e-6;                                    % R is diagonal and theFFR has no idio
initx = ML_lag(F(1:p,:),p-1,0)';                                            % Initial condition for the factor
I = eye(r*p,r*p); A =[A'; I(1:end-r,1:end)];                                % Rewrite the model to get a VAR(1)
initV = reshape(inv(kron(A,A))*Q(:),r*p,r*p);                               % initial state covariance ==> vec(V) = (I- A \kron A) vec(Q)
if method == 1;
    A=A(1:r,1:r*p)';
    F=F/sqrt(N);L=C(:,1:r)*sqrt(N); 
    e=(F(p+1:end,:)-ML_lag(F,p)*A);                                         % Factors VAR Residuals
    if r > q; [u, G]=ML_edynfactors2(e,q); else u=e; G=eye(r); end;         % Common Shocks
    LL=NaN; SF=NaN;
elseif method == 2;     
    %%% ________________________________________________ %%%
    %%%  %%%
    %%% Estimate the Factor model with the Kalman Filter %%%
    %%% ________________________________________________ %%%
    %%%  %%%
    [xitt,xittm,Ptt,Pttm,loglik]=SmokingFilter(initx,initV,x,A,C,R,Q);      % Kalman Filter Estimates of the Factors
    [xsmooth]=SmokingSmoother(A,xitt,xittm,Ptt,Pttm,C,R);                   % Kalman Smoother Estimates of the Factors
    F =  xsmooth(1:r,:)';                                                   % Static factors
    A=A(1:r,1:r*p)'; F=F/sqrt(N);L=C(:,1:r)*sqrt(N);
    e=(F(p+1:end,:)-ML_lag(F,p)*A);                                         % Factors VAR Residuals
    if r > q; [u, G]=ML_edynfactors2(e,q); else u=e; G=eye(r); end;         % Common Shocks
    LL=loglik;
    SF.initx=initx; SF.initV=initV; SF.A=A; SF.C=C; SF.R=R; SF.Q=Q;         % Store the starting values of the last Filter
elseif method == 3; 
    %%% _______________________________________________ %%%
    %%%  %%%
    %%% Estimate the Factor model with the EM Algorithm %%%
    %%% _______________________________________________ %%%
    %%%  %%%
    previous_loglik = -inf;                                                     % Initialize Log Likelihood
    num_iter = 0;                                                               % Number of iterations
    converged = 0;                                                              % Convergence has not reached yet
    LL=zeros(max_iter,1);
    y = x';
    disp('Starting the EM Algorithm');
    while (num_iter < max_iter) && ~converged
        % __________________________________________ %
        % E step : compute the sufficient statistics %
        %  %
        delta = zeros(N,r*p); beta=zeros(r*p,r*p); gamma=zeros(r*p,r*p);                % Initialize the sufficient statistics
        [xitt,xittm,Ptt,Pttm,loglik]=SmokingFilter(initx,initV,y',A,C,R,Q);             % Kalman Filter estimation of the factors
        [xsmooth, Vsmooth, VVsmooth]=SmokingSmoother(A,xitt,xittm,Ptt,Pttm,C,R);        % Kalman Smoother estimation of the factors
        x1 = xsmooth(:,1);                                                              % Initial Condition
        V1 = Vsmooth(:,:,1);
        for t=1:T
            delta = delta + y(:,t)*xsmooth(:,t)';                                       % delta = sum_t=1^T (x_t * f'_t)
            gamma = gamma + xsmooth(:,t)*xsmooth(:,t)' + Vsmooth(:,:,t);                % gamma = sum_t=1^T (f_t * f'_t)
            if t>1; beta = beta + xsmooth(:,t)*xsmooth(:,t-1)' + VVsmooth(:,:,t); end   % beta = sum_t=1^T (f_t * f'_t-1)
        end
        gamma1 = gamma - xsmooth(:,T)*xsmooth(:,T)' - Vsmooth(:,:,T);                   % gamma1 = sum_t=2^T (f_t-1 * f'_t-1)
        gamma2 = gamma - xsmooth(:,1)*xsmooth(:,1)' - Vsmooth(:,:,1);                   % gamma2 = sum_t=2^T (f_t * f'_t)
        % ____________________________________________ %
        % M step : compute the parameters of the model %
        %  %
        C(:,1:r) = delta(:,1:r) / (gamma(1:r,1:r));                                     % C = (sum_t=1^T x_t*f'_t)* (sum_t=1^T f_t*f'_t)^-1
        if p > 0
            Atemp = beta(1:r,1:r*p)/(gamma1(1:r*p,1:r*p));  A(1:r,1:r*p) = Atemp;   	% VAR Parameter
            H = (gamma2(1:r,1:r) - Atemp*beta(1:r,1:r*p)') / (T-1);                     % Variance of VAR Residuals
            if r > q,
                [ P , M ] = eigs(H,q,'lm',OPTS); Q(1:r,1:r) = P*M*P';                   % Q = ( (sum_t=2^T f_t*f'_t) - A * (sum_2=1^T f_t-1*f'_t) )/(T-1)
            else
                Q(1:r,1:r) = H;                                                         % Q = ( (sum_t=2^T f_t*f'_t) - A * (sum_2=1^T f_t-1*f'_t) )/(T-1)
            end;
        end;
        R = (x'*x - C*delta')/T;                                                        % R = ( sum_t=1^T (x_t*x'_t) - C * f_t*x'_t) )/T
        RR = diag(R); RR(RR<1e-7) = 1e-7; R = diag(RR); R(68,68)=1e-6;                     % R is diagonal and theFFR has no idio
        initx = x1;                                                                     % Update the Initial Condition for the factor
        initV = V1;                                                                     % Update the Initial Condition for the MSE of the factor
        converged = em_converged(loglik, previous_loglik, thresh,0);                    % Check Convergence
        previous_loglik = loglik;                                                       % update the loglikelihood
        num_iter =  num_iter + 1;  disp(num_iter)                                       % Counter
        LL(num_iter)=loglik;                                                            % Store the Likelihood
    end
    LL(num_iter+1:end)=[];
    
    SF.initx=initx; SF.initV=initV; SF.A=A; SF.C=C; SF.R=R; SF.Q=Q;         % Store the starting values of the last Filter
    [xitt,xittm,Ptt,Pttm]=SmokingFilter(initx,initV,x,A,C,R,Q);             % Kalman Filter estimation of the factors
    [xsmooth]=SmokingSmoother(A,xitt,xittm,Ptt,Pttm,C,R);                   % Kalman Smoother estimation of the factors
    F =  xsmooth(1:r,:)';                                                   % Static factors
    A=A(1:r,1:r*p)'; F=F/sqrt(N);L=C(:,1:r)*sqrt(N);
    e=(F(p+1:end,:)-ML_lag(F,p)*A);                                         % Factors VAR Residuals
    % u=e*P*(M^-.5);   G=(P(:,1:q))*(M^.5);                                 % Common Shocks
    if r > q; [u, G]=ML_edynfactors2(e,q); else u=e; G=eye(r); end;         % Common Shocks
    if converged == 1; disp('Convergence Reached');
    else disp('Max Number of Iteration Reached');
    end
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% SmokingFilter - Kalman Filter Estimation of a Factor Model
%  _________
%  THE MODEL
%  
%   X_t = C S_t + e_t, e_t ~ N(0,R), R is diagonal
%   S_t = A S_{t-1} + u_t, u_t ~ N(0,Q)
% 
% S_t|X^{t-1} ~ N( S_{t|t-1} , P_{t|t-1} )
% X_t|X^{t-1} ~ N( X_{t|t-1} , H_{t|t-1} )
% 
%  _____________
%  THE PROCEDURE
%  
% [xitt,xittm,Ptt,Pttm,loglik]=SmokingFilter(initx,initV,x,A,C,R,Q)
% 
% INPUTS:
%   x - the data
%   C - the observation matrix 
%   A - the system matrix
%   Q - the system covariance 
%   R - the observation covariance
%   initx - the initial state (column) vector 
%   initV - the initial state covariance 
% OUTPUTS:
%   xittm = S_{t|t-1}
%    Pttm = P_{t|t-1}
%    xitt = S_{t|t} 
%     Ptt = P_{t|t}
%  loglik = value of the loglikelihood
% 

% This procedure was downloaded from Domenico Giannone's web page as part
% of the replication files of the paper Doz, C., Giannone, D., and Reichlin, L.
% ?A Quasi Maximum Likelihood Approach for Large Approximate Dynamic Factor Models?,
% Review of Economics and Statistics, forthcoming
% This codes were successively customized and commented 
% by Matteo Luciani (matteo.luciani@ulb.ac.be)

function [xitt,xittm,Ptt,Pttm,loglik]=SmokingFilter(initx,initV,x,A,C,R,Q)

T=size(x,1);                                                                % Number of Observations
r=size(A,1);                                                                % Number of states
xittm=[initx zeros(r,T)];                                                   % S_{t|t-1}
xitt=zeros(r,T);                                                            % S_{t|t} 
Pttm=zeros(r,r,T); Pttm(:,:,1)=initV;                                       % P_{t|t-1}
Ptt=zeros(r,r,T);                                                           % P_{t|t}
loglik=0;                                                                   % Initialize the log-likelihood
y=x';                                                                       % transpose for convenience

for j=1:T;
    
    %%% _____________ %%%
    %%% Updating Step %%%
    %%%  %%%   
    X=C*xittm(:,j);                                                         % X_{t|t-1} - Observables    
    H=C*Pttm(:,:,j)*C'+R;                                                   % H_{t|t-1} - Conditional Variance of the Observable   
    H1 = inv(H);     
    e = y(:,j) - X;                                                         % error (innovation)
    xitt(:,j)=xittm(:,j)+Pttm(:,:,j)*C'*H1*e;                               % S_{t|t}
    Ptt(:,:,j)=Pttm(:,:,j)-Pttm(:,:,j)*C'*H1*C*Pttm(:,:,j);                 % P_{t|t}   

    %%% _______________ %%%
    %%% Prediction Step %%%
    %%%  %%%
    xittm(:,j+1)=A*xitt(:,j);                                               % S_{t|t-1} - States
    Pttm(:,:,j+1)=A*Ptt(:,:,j)*A'+Q;                                        % P_{t|t-1} - Conditional Variance of the State 
    loglik = loglik + 0.5*(log(det(H))  - e'*H1*e);                         % Log Likelihood   
    
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% SmokingSmoother - Kalman Smoother for a Factor Model
% 
%  THE MODEL
%  
%   X_t = C S_t + e_t, e_t ~ N(0,R), R is diagonal
%   S_t = A S_{t-1} + u_t, u_t ~ N(0,Q)
% 
% S_t|X^{t-1} ~ N( S_{t|t-1} , P_{t|t-1} )
% X_t|X^{t-1} ~ N( X_{t|t-1} , H_{t|t-1} )
% 
%  _____________
%  THE PROCEDURE
%  
% [xitT,PtT,PtTm]=SmokingSmoother(A,xitt,xittm,Ptt,Pttm,C,R)
% 
% INPUTS:
%       A - the system matrix
%    xitt - S_{t|t}
%   xittm - S_{t|t-1}
%     Ptt - P_{t|t}
%    Pttm - P_{t|t-1}
%       C - the observation matrix 
%       R - the observation covariance
% OUTPUTS:
%    xitT = S_{t|T}
%     PtT = P_{t|T} 
%    PtTm = P_{t+1|T} 
%     

% This procedure was downloaded from Domenico Giannone's web page as part
% of the replication files of the paper Doz, C., Giannone, D., and Reichlin, L.
% ?A Quasi Maximum Likelihood Approach for Large Approximate Dynamic Factor Models?,
% Review of Economics and Statistics, forthcoming
% This codes were successively customized and commented 
% by Matteo Luciani (matteo.luciani@ulb.ac.be)

function [xitT,PtT,PtTm]=SmokingSmoother(A,xitt,xittm,Ptt,Pttm,C,R)

[T]=size(xitt,2);                                                           % Number of Observations
[N r]=size(C);                                                              % Number of Variables and Number of States
Pttm=Pttm(:,:,1:end-1);                                                     % P_{t|t-1}, remove the last observation because it has dimension T+1
xittm=xittm(:,1:end-1);                                                     % S_{t|t-1}, remove the last observation because it has dimension T+1
J=zeros(r,r,T);
xitT=[zeros(r,T-1)  xitt(:,T)];                                             % S_{t|T} 
PtT=cat(3,zeros(r,r,T-1),Ptt(:,:,T));                                       % P_{t|T} 
PtTm=zeros(r,r,T);                                                          % P_{t+1|T} 

for ii=1:T-1; J(:,:,ii)=Ptt(:,:,ii)*A'*inv(Pttm(:,:,ii+1)); end;

for jj = 1:T-1;    
    xitT(:,T-jj)= xitt(:,T-jj)+J(:,:,T-jj)*(xitT(:,T+1-jj)-xittm(:,T+1-jj));                    % S_{t|T} 
    PtT(:,:,T-jj)=Ptt(:,:,T-jj)+J(:,:,T-jj)*(PtT(:,:,T+1-jj)-Pttm(:,:,T+1-jj))*J(:,:,T-jj)';    % P_{t|T}         
end

L=zeros(N,N,T);
K=zeros(r,N,T);

for i=1:T
    L(:,:,i)=inv(C*Pttm(:,:,i)*C'+R);
    K(:,:,i)=Pttm(:,:,i)*C'*L(:,:,i);
end

PtTm(:,:,T)=(eye(r)-K(:,:,T)*C)*A*Ptt(:,:,T-1);                             % P_{t+1|T} 
for j = 1:T-2
    PtTm(:,:,T-j)=Ptt(:,:,T-j)*J(:,:,T-j-1)'+J(:,:,T-j)*(PtTm(:,:,T-j+1)-A*Ptt(:,:,T-j))*J(:,:,T-j-1)';
end


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [converged, decrease] = em_converged(loglik, previous_loglik, threshold, check_increased)
% EM_CONVERGED Has EM converged?
% [converged, decrease] = em_converged(loglik, previous_loglik, threshold)
%
% We have converged if the slope of the log-likelihood function falls below 'threshold', 
% i.e., |f(t) - f(t-1)| / avg < threshold,
% where avg = (|f(t)| + |f(t-1)|)/2 and f(t) is log lik at iteration t.
% 'threshold' defaults to 1e-4.
%
% This stopping criterion is from Numerical Recipes in C p423
%
% If we are doing MAP estimation (using priors), the likelihood can decrase,
% even though the mode of the posterior is increasing.

% This procedure was downloaded from Domenico Giannone's web page as part
% of the replication files of the paper Doz, C., Giannone, D., and Reichlin, L.
% ?A Quasi Maximum Likelihood Approach for Large Approximate Dynamic Factor Models?,
% Review of Economics and Statistics, forthcoming
% This codes were successively customized and commented 
% by Matteo Luciani (matteo.luciani@ulb.ac.be)

if nargin < 3, threshold = 1e-4; end
if nargin < 4, check_increased = 1; end

converged = 0;
decrease = 0;

if check_increased
    if loglik - previous_loglik < -1e-3 % allow for a little imprecision
             fprintf(1, '******likelihood decreased from %6.4f to %6.4f!\n', previous_loglik, loglik);
        decrease = 1;
    end
end

delta_loglik = abs(loglik - previous_loglik);
avg_loglik = (abs(loglik) + abs(previous_loglik) + eps)/2;
if (delta_loglik / avg_loglik) < threshold, converged = 1; end



