% ML_SmokingGun_SignBoot - Bootstrap Intervals with Sign Restricitions (Special Version)
% [CC,Y,u] = ML_SignBoot2(X,cd,q,r,p,h,L,nrest,rest,ndraws,horiz,maxrot)
% 
% Inputs:
%     CC - Impulse responses
%      Y - Percentiles of the distribution of the IRF 
%      u - number of boostrap which finds at least one rotation
% 
% Outputs:
%      X - Data                                                             (1)
%     cd - code for data transformation                                     (2)
%      q - n of dynamic factors                                            (3)
%      r - n of stati factors                                              (4)
%      p - n of lags in the VAR                                            (5)
%      h - n of lags in the MA representation                              (6)
% 
%  nrest - normalization restrictions                                       (8)
%   rest - restrictions                                                     (9)
% ndraws - n of draws for the bootstrap algorithm                          (10)
%  horiz - n of horizon at which the restriction is imposed                (11)
% nrepli - n of replications                                               (12)
% maxrot - Maximum Number of Rotation                                       (13) optional
%

% Written by Mario Forni
% Modified and commented by Matteo Luciani (matteo.luciani@ulb.ac.be)

function [CC] = ML_SmokingGun_SignBoot(x,cd,q,r,p,h,nrest,rest,ndraws,horiz,nboot,maxrot,FP)
if nargin<13; FP=1; end
inizio=clock;                                                               % when the code starts
N = size(x,2);                                                              % Number of Variables
WW = diag(std(x)); WWW=repmat(diag(WW),1,q);                                % Standard deviation
if  isempty(maxrot); maxrot = 200; end                                      % Maximum Number of Rotation
det=0;                                                                      % no constant is included
[fhat, lambda, A, eta, G] = ML_SmokingGun_efactorsML(x,q,r,p,100,3);        % ML Estimation of the DFM
u=eta*G';                                                                   % Residual from the VAR
A2=ML_VARbias(fhat,u,A,p,det,nboot,eta,G');                                 % Computes the bias induced by VAR Estimation - Kilian (1998)
n=size(u,1);
CLstar=ones(N,q,h);                                                         % preallocates variables

tic
for bb=1:nboot;        
    bootsam=ceil(n*rand(n,1));                                              % reshuffling residuals    
    fstar=ML_decomposition(fhat,A2,eta(bootsam,:),det,G');                  % Generates Static Factors
    [Astar ustar]=ML_VAR(fstar,p,det);                                      % Estimating the Law of Motion fo the Static Factors   
    [~, Gstar]=ML_edynfactors2(ustar,q);                                    % Estimating Dynamic Factors Innovation
    Bstar = ML_MA(h,Astar,det);                                             % MA Representation of the static factors
    for ii=1:h; CLstar(:,:,ii)=(lambda*Bstar(:,:,ii)*Gstar).*WWW; end;      % MA Representation of common components
    CLstar = CumImp(CLstar, cd);                                            % Comulates IRF if necessary   
    GR = ML_GoodPolar(CLstar,nrest,rest,ndraws,horiz,maxrot);               % Good Rotations        
    C = ML_ComputeIrf(CLstar,GR);                                           % Compute IRF   
    clear fstar Astar ustar Gstar Bstar                                     % Clear Objects to free space in the memory
    ngr=size(GR,3);   
    if ngr ~= 0; 
        if (ngr > 1) && (FP==1);                    
            BR=ML_SignRestrictionFryPagan(C,rest);                          % Identify the best rotation in the sense of Fry & Pagan                                           
            C=C(:,:,:,BR);                                                  % Keep Only the FP rotation     
        end
    end;                 
    CC{bb} = C;                                                             % Store IRF        
    disp([nboot-bb ngr toc])                                                % Counter    
    tic
end


fine=clock;                                                                 % when the code ends
format bank;                                                                % I change the format of data just for displaying purposes
disp(''); 
fprintf('Procedure startet at:'); disp(inizio(4:end));
fprintf('Procedure ended at:  '); disp(fine(4:end));
disp(''); 
format                                                                      % Back to normal format
